-- SinglesZonne Dating Platform - Complete Database Schema
-- Created for comprehensive international dating website with 45+ countries support

-- ============================================================================
-- CORE USER MANAGEMENT TABLES
-- ============================================================================

-- Users table with comprehensive profile information
CREATE TABLE users (
    id BIGINT PRIMARY KEY AUTO_INCREMENT,
    email VARCHAR(255) UNIQUE NOT NULL,
    username VARCHAR(50) UNIQUE NOT NULL,
    password_hash VARCHAR(255) NOT NULL,
    salt VARCHAR(255) NOT NULL,
    
    -- Basic Profile Information
    first_name VARCHAR(100) NOT NULL,
    last_name VARCHAR(100) NOT NULL,
    display_name VARCHAR(100) NOT NULL,
    date_of_birth DATE NOT NULL,
    gender ENUM('male', 'female', 'non-binary', 'other') NOT NULL,
    
    -- Location Information
    city VARCHAR(100) NOT NULL,
    state_province VARCHAR(100),
    country VARCHAR(100) NOT NULL,
    country_code VARCHAR(3) NOT NULL,
    latitude DECIMAL(10, 8),
    longitude DECIMAL(11, 8),
    timezone VARCHAR(50),
    
    -- Profile Details
    bio TEXT,
    occupation VARCHAR(200),
    education VARCHAR(200),
    height_cm INT,
    relationship_status ENUM('single', 'divorced', 'widowed', 'separated') DEFAULT 'single',
    
    -- Account Status
    email_verified BOOLEAN DEFAULT FALSE,
    phone_verified BOOLEAN DEFAULT FALSE,
    id_verified BOOLEAN DEFAULT FALSE,
    is_premium BOOLEAN DEFAULT FALSE,
    is_online BOOLEAN DEFAULT FALSE,
    last_seen TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    
    -- Account Settings
    profile_visibility ENUM('public', 'members_only', 'hidden') DEFAULT 'public',
    show_distance BOOLEAN DEFAULT TRUE,
    show_age BOOLEAN DEFAULT TRUE,
    show_last_seen BOOLEAN DEFAULT TRUE,
    
    -- Account Management
    status ENUM('active', 'suspended', 'banned', 'deleted') DEFAULT 'active',
    suspension_reason TEXT,
    suspension_until TIMESTAMP NULL,
    
    -- Timestamps
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    deleted_at TIMESTAMP NULL,
    
    INDEX idx_location (country_code, city),
    INDEX idx_status (status, is_online),
    INDEX idx_premium (is_premium),
    INDEX idx_created (created_at),
    INDEX idx_coordinates (latitude, longitude)
);

-- User roles and permissions
CREATE TABLE user_roles (
    id BIGINT PRIMARY KEY AUTO_INCREMENT,
    user_id BIGINT NOT NULL,
    role ENUM('user', 'premium', 'moderator', 'admin', 'super_admin') NOT NULL,
    granted_by BIGINT,
    granted_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    expires_at TIMESTAMP NULL,
    is_active BOOLEAN DEFAULT TRUE,
    
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (granted_by) REFERENCES users(id) ON DELETE SET NULL,
    INDEX idx_user_role (user_id, role, is_active)
);

-- Email verification tokens
CREATE TABLE email_verifications (
    id BIGINT PRIMARY KEY AUTO_INCREMENT,
    user_id BIGINT NOT NULL,
    token VARCHAR(255) UNIQUE NOT NULL,
    expires_at TIMESTAMP NOT NULL,
    used_at TIMESTAMP NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    INDEX idx_token (token),
    INDEX idx_expires (expires_at)
);

-- Password reset tokens
CREATE TABLE password_resets (
    id BIGINT PRIMARY KEY AUTO_INCREMENT,
    user_id BIGINT NOT NULL,
    token VARCHAR(255) UNIQUE NOT NULL,
    expires_at TIMESTAMP NOT NULL,
    used_at TIMESTAMP NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    INDEX idx_token (token),
    INDEX idx_expires (expires_at)
);

-- ============================================================================
-- USER PROFILE AND MEDIA TABLES
-- ============================================================================

-- User photos with enhanced metadata
CREATE TABLE user_photos (
    id BIGINT PRIMARY KEY AUTO_INCREMENT,
    user_id BIGINT NOT NULL,
    file_path VARCHAR(500) NOT NULL,
    file_name VARCHAR(255) NOT NULL,
    file_size INT NOT NULL,
    mime_type VARCHAR(100) NOT NULL,
    width INT,
    height INT,
    
    -- Photo metadata
    is_primary BOOLEAN DEFAULT FALSE,
    position_order INT DEFAULT 1,
    caption TEXT,
    
    -- Moderation
    moderation_status ENUM('pending', 'approved', 'rejected') DEFAULT 'pending',
    moderated_by BIGINT,
    moderated_at TIMESTAMP NULL,
    rejection_reason TEXT,
    
    -- Analytics
    view_count INT DEFAULT 0,
    like_count INT DEFAULT 0,
    
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (moderated_by) REFERENCES users(id) ON DELETE SET NULL,
    INDEX idx_user_photos (user_id, position_order),
    INDEX idx_moderation (moderation_status),
    INDEX idx_primary (user_id, is_primary)
);

-- User interests and hobbies
CREATE TABLE interests (
    id INT PRIMARY KEY AUTO_INCREMENT,
    name VARCHAR(100) UNIQUE NOT NULL,
    category VARCHAR(50),
    is_active BOOLEAN DEFAULT TRUE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

CREATE TABLE user_interests (
    id BIGINT PRIMARY KEY AUTO_INCREMENT,
    user_id BIGINT NOT NULL,
    interest_id INT NOT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (interest_id) REFERENCES interests(id) ON DELETE CASCADE,
    UNIQUE KEY unique_user_interest (user_id, interest_id)
);

-- User preferences for matching
CREATE TABLE user_preferences (
    id BIGINT PRIMARY KEY AUTO_INCREMENT,
    user_id BIGINT NOT NULL,
    
    -- Relationship preferences
    relationship_type ENUM('casual', 'serious', 'friendship', 'activity_partner', 'anything') DEFAULT 'serious',
    looking_for_text TEXT,
    
    -- Age preferences
    min_age INT DEFAULT 18,
    max_age INT DEFAULT 65,
    
    -- Distance preferences
    max_distance INT DEFAULT 50, -- in kilometers
    
    -- Location preferences
    countries_preference ENUM('all', 'specific', 'multiple') DEFAULT 'all',
    preferred_countries JSON, -- Array of country codes
    
    -- Physical preferences
    height_preference_min INT, -- in centimeters
    height_preference_max INT,
    
    -- Lifestyle preferences
    deal_breakers JSON, -- Array of deal breaker strings
    important_qualities JSON, -- Array of important quality strings
    
    -- Communication preferences
    receive_messages_from ENUM('everyone', 'premium_only', 'verified_only') DEFAULT 'everyone',
    
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    UNIQUE KEY unique_user_preferences (user_id)
);

-- User media (videos, audio)
CREATE TABLE user_media (
    id BIGINT PRIMARY KEY AUTO_INCREMENT,
    user_id BIGINT NOT NULL,
    media_type ENUM('intro_video', 'profile_music', 'custom_background') NOT NULL,
    file_path VARCHAR(500) NOT NULL,
    file_name VARCHAR(255) NOT NULL,
    file_size INT NOT NULL,
    mime_type VARCHAR(100) NOT NULL,
    duration INT, -- in seconds for video/audio
    
    -- Media metadata
    title VARCHAR(200),
    artist VARCHAR(200), -- for music
    description TEXT,
    
    -- Settings
    is_active BOOLEAN DEFAULT TRUE,
    auto_play BOOLEAN DEFAULT FALSE,
    volume_level DECIMAL(3,2) DEFAULT 0.30, -- 0.00 to 1.00
    
    -- Moderation
    moderation_status ENUM('pending', 'approved', 'rejected') DEFAULT 'pending',
    moderated_by BIGINT,
    moderated_at TIMESTAMP NULL,
    
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (moderated_by) REFERENCES users(id) ON DELETE SET NULL,
    INDEX idx_user_media (user_id, media_type, is_active),
    INDEX idx_moderation (moderation_status)
);

-- ============================================================================
-- MATCHING AND INTERACTION TABLES
-- ============================================================================

-- User likes/passes for Hot or Not and general matching
CREATE TABLE user_interactions (
    id BIGINT PRIMARY KEY AUTO_INCREMENT,
    user_id BIGINT NOT NULL, -- User who performed the action
    target_user_id BIGINT NOT NULL, -- User who received the action
    interaction_type ENUM('like', 'pass', 'super_like', 'block', 'report') NOT NULL,
    context ENUM('search', 'hot_or_not', 'discover', 'profile_view') NOT NULL,
    
    -- Additional data
    message TEXT, -- Optional message with like
    report_reason VARCHAR(200), -- For reports
    
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (target_user_id) REFERENCES users(id) ON DELETE CASCADE,
    INDEX idx_user_interactions (user_id, interaction_type),
    INDEX idx_target_interactions (target_user_id, interaction_type),
    INDEX idx_context (context, created_at),
    UNIQUE KEY unique_user_target_context (user_id, target_user_id, context)
);

-- Matches when two users like each other
CREATE TABLE matches (
    id BIGINT PRIMARY KEY AUTO_INCREMENT,
    user1_id BIGINT NOT NULL,
    user2_id BIGINT NOT NULL,
    matched_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    
    -- Match status
    status ENUM('active', 'unmatched_user1', 'unmatched_user2', 'mutual_unmatch') DEFAULT 'active',
    unmatched_at TIMESTAMP NULL,
    unmatched_by BIGINT,
    
    -- Match scoring
    compatibility_score DECIMAL(5,2), -- AI-generated compatibility score
    
    FOREIGN KEY (user1_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (user2_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (unmatched_by) REFERENCES users(id) ON DELETE SET NULL,
    INDEX idx_user_matches (user1_id, status),
    INDEX idx_user2_matches (user2_id, status),
    INDEX idx_match_date (matched_at),
    UNIQUE KEY unique_match (LEAST(user1_id, user2_id), GREATEST(user1_id, user2_id))
);

-- Photo likes (individual photo interactions)
CREATE TABLE photo_likes (
    id BIGINT PRIMARY KEY AUTO_INCREMENT,
    user_id BIGINT NOT NULL, -- User who liked the photo
    photo_id BIGINT NOT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (photo_id) REFERENCES user_photos(id) ON DELETE CASCADE,
    UNIQUE KEY unique_photo_like (user_id, photo_id)
);

-- Profile views tracking
CREATE TABLE profile_views (
    id BIGINT PRIMARY KEY AUTO_INCREMENT,
    viewer_id BIGINT NOT NULL,
    viewed_user_id BIGINT NOT NULL,
    view_duration INT, -- in seconds
    viewed_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    
    FOREIGN KEY (viewer_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (viewed_user_id) REFERENCES users(id) ON DELETE CASCADE,
    INDEX idx_viewer (viewer_id, viewed_at),
    INDEX idx_viewed (viewed_user_id, viewed_at)
);

-- ============================================================================
-- MESSAGING SYSTEM TABLES
-- ============================================================================

-- Conversations between users
CREATE TABLE conversations (
    id BIGINT PRIMARY KEY AUTO_INCREMENT,
    user1_id BIGINT NOT NULL,
    user2_id BIGINT NOT NULL,
    match_id BIGINT,
    
    -- Conversation settings
    is_active BOOLEAN DEFAULT TRUE,
    archived_by_user1 BOOLEAN DEFAULT FALSE,
    archived_by_user2 BOOLEAN DEFAULT FALSE,
    
    -- Last message info
    last_message_id BIGINT,
    last_message_at TIMESTAMP,
    
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    
    FOREIGN KEY (user1_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (user2_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (match_id) REFERENCES matches(id) ON DELETE SET NULL,
    INDEX idx_user_conversations (user1_id, is_active),
    INDEX idx_user2_conversations (user2_id, is_active),
    INDEX idx_last_message (last_message_at DESC),
    UNIQUE KEY unique_conversation (LEAST(user1_id, user2_id), GREATEST(user1_id, user2_id))
);

-- Messages within conversations
CREATE TABLE messages (
    id BIGINT PRIMARY KEY AUTO_INCREMENT,
    conversation_id BIGINT NOT NULL,
    sender_id BIGINT NOT NULL,
    
    -- Message content
    message_type ENUM('text', 'image', 'gif', 'emoji', 'system') DEFAULT 'text',
    content TEXT NOT NULL,
    image_url VARCHAR(500),
    
    -- Message status
    is_read BOOLEAN DEFAULT FALSE,
    read_at TIMESTAMP NULL,
    is_deleted BOOLEAN DEFAULT FALSE,
    deleted_at TIMESTAMP NULL,
    
    -- Moderation
    is_flagged BOOLEAN DEFAULT FALSE,
    flagged_reason VARCHAR(200),
    moderated_by BIGINT,
    
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    
    FOREIGN KEY (conversation_id) REFERENCES conversations(id) ON DELETE CASCADE,
    FOREIGN KEY (sender_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (moderated_by) REFERENCES users(id) ON DELETE SET NULL,
    INDEX idx_conversation_messages (conversation_id, created_at),
    INDEX idx_sender_messages (sender_id, created_at),
    INDEX idx_unread (conversation_id, is_read, created_at)
);

-- Update conversations with last message reference
ALTER TABLE conversations 
ADD FOREIGN KEY (last_message_id) REFERENCES messages(id) ON DELETE SET NULL;

-- ============================================================================
-- GROUPS AND EVENTS SYSTEM
-- ============================================================================

-- Groups created by admins
CREATE TABLE groups (
    id BIGINT PRIMARY KEY AUTO_INCREMENT,
    name VARCHAR(200) NOT NULL,
    description TEXT,
    group_type ENUM('general', 'age_group', 'location', 'interest', 'event') NOT NULL,
    
    -- Group settings
    is_public BOOLEAN DEFAULT TRUE,
    requires_approval BOOLEAN DEFAULT FALSE,
    max_members INT DEFAULT 1000,
    
    -- Location-based groups
    country VARCHAR(100),
    city VARCHAR(100),
    
    -- Age-based groups
    min_age INT,
    max_age INT,
    
    -- Group image
    cover_image_url VARCHAR(500),
    
    -- Management
    created_by BIGINT NOT NULL,
    status ENUM('active', 'suspended', 'archived') DEFAULT 'active',
    
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    
    FOREIGN KEY (created_by) REFERENCES users(id) ON DELETE CASCADE,
    INDEX idx_group_type (group_type, status),
    INDEX idx_location (country, city),
    INDEX idx_age_range (min_age, max_age)
);

-- Group memberships
CREATE TABLE group_members (
    id BIGINT PRIMARY KEY AUTO_INCREMENT,
    group_id BIGINT NOT NULL,
    user_id BIGINT NOT NULL,
    role ENUM('member', 'moderator', 'admin') DEFAULT 'member',
    
    -- Membership status
    status ENUM('active', 'pending', 'banned') DEFAULT 'active',
    joined_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    approved_by BIGINT,
    approved_at TIMESTAMP,
    
    FOREIGN KEY (group_id) REFERENCES groups(id) ON DELETE CASCADE,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (approved_by) REFERENCES users(id) ON DELETE SET NULL,
    INDEX idx_group_members (group_id, status),
    INDEX idx_user_groups (user_id, status),
    UNIQUE KEY unique_group_member (group_id, user_id)
);

-- Events (can be linked to groups or standalone)
CREATE TABLE events (
    id BIGINT PRIMARY KEY AUTO_INCREMENT,
    title VARCHAR(200) NOT NULL,
    description TEXT,
    
    -- Event details
    event_type ENUM('virtual', 'in_person', 'hybrid') NOT NULL,
    start_date TIMESTAMP NOT NULL,
    end_date TIMESTAMP,
    timezone VARCHAR(50),
    
    -- Location (for in-person events)
    venue_name VARCHAR(200),
    address TEXT,
    city VARCHAR(100),
    country VARCHAR(100),
    latitude DECIMAL(10, 8),
    longitude DECIMAL(11, 8),
    
    -- Virtual event details
    meeting_url VARCHAR(500),
    meeting_password VARCHAR(100),
    
    -- Event settings
    max_attendees INT,
    requires_approval BOOLEAN DEFAULT FALSE,
    is_premium_only BOOLEAN DEFAULT FALSE,
    cost DECIMAL(10, 2) DEFAULT 0.00,
    currency VARCHAR(3) DEFAULT 'USD',
    
    -- Associated group
    group_id BIGINT,
    
    -- Management
    created_by BIGINT NOT NULL,
    status ENUM('draft', 'published', 'cancelled', 'completed') DEFAULT 'draft',
    
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    
    FOREIGN KEY (created_by) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (group_id) REFERENCES groups(id) ON DELETE SET NULL,
    INDEX idx_event_date (start_date, status),
    INDEX idx_event_location (country, city),
    INDEX idx_event_group (group_id)
);

-- Event attendees
CREATE TABLE event_attendees (
    id BIGINT PRIMARY KEY AUTO_INCREMENT,
    event_id BIGINT NOT NULL,
    user_id BIGINT NOT NULL,
    
    -- Attendance status
    status ENUM('interested', 'going', 'maybe', 'not_going', 'pending_approval') DEFAULT 'interested',
    rsvp_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    approved_by BIGINT,
    approved_at TIMESTAMP,
    
    -- Check-in for tracking
    checked_in BOOLEAN DEFAULT FALSE,
    checked_in_at TIMESTAMP,
    
    FOREIGN KEY (event_id) REFERENCES events(id) ON DELETE CASCADE,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (approved_by) REFERENCES users(id) ON DELETE SET NULL,
    INDEX idx_event_attendees (event_id, status),
    INDEX idx_user_events (user_id, status),
    UNIQUE KEY unique_event_attendee (event_id, user_id)
);

-- ============================================================================
-- ADS AND POINTS SYSTEM
-- ============================================================================

-- User points tracking
CREATE TABLE user_points (
    id BIGINT PRIMARY KEY AUTO_INCREMENT,
    user_id BIGINT NOT NULL,
    current_balance INT DEFAULT 0,
    total_earned INT DEFAULT 0,
    total_spent INT DEFAULT 0,
    
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    UNIQUE KEY unique_user_points (user_id)
);

-- Points transactions
CREATE TABLE points_transactions (
    id BIGINT PRIMARY KEY AUTO_INCREMENT,
    user_id BIGINT NOT NULL,
    transaction_type ENUM('earned', 'spent', 'bonus', 'penalty', 'refund') NOT NULL,
    points_amount INT NOT NULL,
    source ENUM('ad_view', 'daily_bonus', 'referral', 'purchase', 'admin_grant', 'super_like', 'boost') NOT NULL,
    
    -- Transaction details
    description VARCHAR(500),
    reference_id BIGINT, -- Reference to ad, purchase, etc.
    reference_type VARCHAR(50), -- Type of reference (ad_id, subscription_id, etc.)
    
    -- Balance tracking
    balance_before INT NOT NULL,
    balance_after INT NOT NULL,
    
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    INDEX idx_user_transactions (user_id, created_at),
    INDEX idx_transaction_type (transaction_type, source),
    INDEX idx_reference (reference_type, reference_id)
);

-- Advertisement campaigns
CREATE TABLE ad_campaigns (
    id BIGINT PRIMARY KEY AUTO_INCREMENT,
    
    -- Campaign details
    title VARCHAR(200) NOT NULL,
    description TEXT,
    category ENUM('dating', 'lifestyle', 'fashion', 'travel', 'technology', 'health', 'other') NOT NULL,
    
    -- Ad content
    image_url VARCHAR(500),
    video_url VARCHAR(500),
    click_url VARCHAR(500) NOT NULL,
    call_to_action VARCHAR(100),
    
    -- Targeting
    target_countries JSON, -- Array of country codes
    target_age_min INT DEFAULT 18,
    target_age_max INT DEFAULT 65,
    target_gender ENUM('male', 'female', 'all') DEFAULT 'all',
    
    -- Budget and pricing
    total_budget DECIMAL(10, 2),
    cost_per_view DECIMAL(5, 4) DEFAULT 0.0010, -- Cost in USD
    points_per_view INT DEFAULT 1,
    
    -- Campaign schedule
    start_date TIMESTAMP NOT NULL,
    end_date TIMESTAMP,
    
    -- Performance tracking
    total_views INT DEFAULT 0,
    total_clicks INT DEFAULT 0,
    total_spent DECIMAL(10, 2) DEFAULT 0.00,
    
    -- Management
    created_by BIGINT NOT NULL,
    status ENUM('draft', 'pending_approval', 'active', 'paused', 'completed', 'rejected') DEFAULT 'draft',
    approved_by BIGINT,
    approved_at TIMESTAMP,
    
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    
    FOREIGN KEY (created_by) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (approved_by) REFERENCES users(id) ON DELETE SET NULL,
    INDEX idx_campaign_status (status, start_date),
    INDEX idx_campaign_targeting (target_age_min, target_age_max, target_gender)
);

-- Ad views and interactions
CREATE TABLE ad_interactions (
    id BIGINT PRIMARY KEY AUTO_INCREMENT,
    campaign_id BIGINT NOT NULL,
    user_id BIGINT NOT NULL,
    
    -- Interaction details
    interaction_type ENUM('view', 'click', 'dismiss') NOT NULL,
    view_duration INT, -- in seconds
    points_awarded INT DEFAULT 0,
    
    -- Context
    page_context VARCHAR(100), -- Which page the ad was shown on
    device_type ENUM('desktop', 'mobile', 'tablet') DEFAULT 'desktop',
    
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    
    FOREIGN KEY (campaign_id) REFERENCES ad_campaigns(id) ON DELETE CASCADE,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    INDEX idx_campaign_interactions (campaign_id, interaction_type),
    INDEX idx_user_interactions (user_id, created_at),
    INDEX idx_points_awarded (points_awarded, created_at)
);

-- User-created ads (member ads)
CREATE TABLE member_ads (
    id BIGINT PRIMARY KEY AUTO_INCREMENT,
    user_id BIGINT NOT NULL,
    
    -- Ad content
    title VARCHAR(200) NOT NULL,
    description TEXT NOT NULL,
    category ENUM('dating', 'friendship', 'activity_partner', 'networking', 'other') NOT NULL,
    
    -- Images
    image_urls JSON, -- Array of image URLs
    
    -- Location and preferences
    location_type ENUM('local', 'national', 'international') DEFAULT 'local',
    target_age_min INT,
    target_age_max INT,
    target_gender ENUM('male', 'female', 'all') DEFAULT 'all',
    
    -- Pricing and promotion
    is_promoted BOOLEAN DEFAULT FALSE,
    promotion_points_spent INT DEFAULT 0,
    promotion_expires_at TIMESTAMP,
    
    -- Performance
    view_count INT DEFAULT 0,
    contact_count INT DEFAULT 0,
    
    -- Management
    status ENUM('draft', 'published', 'paused', 'expired', 'removed') DEFAULT 'draft',
    expires_at TIMESTAMP,
    
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    INDEX idx_member_ads (status, location_type),
    INDEX idx_user_ads (user_id, status),
    INDEX idx_targeting (target_age_min, target_age_max, target_gender),
    INDEX idx_expiry (expires_at, status)
);

-- ============================================================================
-- SUBSCRIPTION AND PAYMENT SYSTEM
-- ============================================================================

-- Subscription plans
CREATE TABLE subscription_plans (
    id INT PRIMARY KEY AUTO_INCREMENT,
    name VARCHAR(100) NOT NULL,
    description TEXT,
    tier ENUM('basic', 'premium', 'vip') NOT NULL,
    
    -- Pricing
    price_monthly DECIMAL(8, 2) NOT NULL,
    price_quarterly DECIMAL(8, 2),
    price_yearly DECIMAL(8, 2),
    currency VARCHAR(3) DEFAULT 'USD',
    
    -- Features (JSON array of feature keys)
    features JSON NOT NULL,
    
    -- Limits
    daily_likes_limit INT DEFAULT 10,
    super_likes_per_day INT DEFAULT 1,
    photo_upload_limit INT DEFAULT 9,
    
    -- Status
    is_active BOOLEAN DEFAULT TRUE,
    sort_order INT DEFAULT 0,
    
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    
    INDEX idx_active_plans (is_active, sort_order)
);

-- User subscriptions
CREATE TABLE user_subscriptions (
    id BIGINT PRIMARY KEY AUTO_INCREMENT,
    user_id BIGINT NOT NULL,
    plan_id INT NOT NULL,
    
    -- Subscription details
    billing_cycle ENUM('monthly', 'quarterly', 'yearly') NOT NULL,
    amount_paid DECIMAL(8, 2) NOT NULL,
    currency VARCHAR(3) NOT NULL,
    
    -- Subscription period
    starts_at TIMESTAMP NOT NULL,
    expires_at TIMESTAMP NOT NULL,
    
    -- Status
    status ENUM('active', 'cancelled', 'expired', 'suspended') DEFAULT 'active',
    cancelled_at TIMESTAMP NULL,
    cancellation_reason TEXT,
    
    -- Auto-renewal
    auto_renew BOOLEAN DEFAULT TRUE,
    next_billing_date TIMESTAMP,
    
    -- Payment reference
    payment_provider VARCHAR(50), -- stripe, paypal, etc.
    payment_subscription_id VARCHAR(255),
    
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (plan_id) REFERENCES subscription_plans(id),
    INDEX idx_user_subscriptions (user_id, status),
    INDEX idx_expiry_check (expires_at, status),
    INDEX idx_billing_dates (next_billing_date, auto_renew)
);

-- Payment transactions
CREATE TABLE payments (
    id BIGINT PRIMARY KEY AUTO_INCREMENT,
    user_id BIGINT NOT NULL,
    subscription_id BIGINT,
    
    -- Payment details
    amount DECIMAL(8, 2) NOT NULL,
    currency VARCHAR(3) NOT NULL,
    payment_method ENUM('credit_card', 'paypal', 'apple_pay', 'google_pay', 'bank_transfer') NOT NULL,
    
    -- Provider details
    payment_provider VARCHAR(50) NOT NULL,
    provider_transaction_id VARCHAR(255),
    provider_payment_id VARCHAR(255),
    
    -- Status
    status ENUM('pending', 'completed', 'failed', 'refunded', 'disputed') NOT NULL,
    failure_reason TEXT,
    
    -- Metadata
    description VARCHAR(500),
    metadata JSON,
    
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (subscription_id) REFERENCES user_subscriptions(id) ON DELETE SET NULL,
    INDEX idx_user_payments (user_id, status),
    INDEX idx_provider_transaction (payment_provider, provider_transaction_id),
    INDEX idx_payment_status (status, created_at)
);

-- ============================================================================
-- CONTENT MODERATION AND REPORTING
-- ============================================================================

-- Content reports
CREATE TABLE content_reports (
    id BIGINT PRIMARY KEY AUTO_INCREMENT,
    reporter_id BIGINT NOT NULL,
    
    -- Reported content
    content_type ENUM('user_profile', 'photo', 'message', 'group', 'event', 'member_ad') NOT NULL,
    content_id BIGINT NOT NULL,
    reported_user_id BIGINT, -- User being reported
    
    -- Report details
    report_category ENUM('inappropriate_content', 'harassment', 'spam', 'fake_profile', 'underage', 'violence', 'other') NOT NULL,
    description TEXT,
    
    -- Moderation
    status ENUM('pending', 'under_review', 'resolved', 'dismissed') DEFAULT 'pending',
    assigned_to BIGINT, -- Moderator assigned
    moderator_notes TEXT,
    action_taken ENUM('none', 'content_removed', 'user_warned', 'user_suspended', 'user_banned') DEFAULT 'none',
    
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    resolved_at TIMESTAMP,
    
    FOREIGN KEY (reporter_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (reported_user_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (assigned_to) REFERENCES users(id) ON DELETE SET NULL,
    INDEX idx_content_reports (content_type, content_id),
    INDEX idx_status (status, created_at),
    INDEX idx_assigned (assigned_to, status)
);

-- Automated content moderation logs
CREATE TABLE content_moderation_logs (
    id BIGINT PRIMARY KEY AUTO_INCREMENT,
    
    -- Content details
    content_type ENUM('user_profile', 'photo', 'message', 'bio', 'group_post') NOT NULL,
    content_id BIGINT NOT NULL,
    user_id BIGINT NOT NULL,
    
    -- AI/Automated moderation results
    ai_confidence_score DECIMAL(5, 4), -- 0.0000 to 1.0000
    detected_issues JSON, -- Array of detected issues
    risk_level ENUM('low', 'medium', 'high', 'critical') NOT NULL,
    
    -- Actions taken
    auto_action ENUM('approved', 'flagged', 'hidden', 'rejected') NOT NULL,
    requires_human_review BOOLEAN DEFAULT FALSE,
    
    -- Human review
    human_reviewed BOOLEAN DEFAULT FALSE,
    reviewed_by BIGINT,
    human_decision ENUM('approve', 'reject', 'needs_edit'),
    review_notes TEXT,
    
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    reviewed_at TIMESTAMP,
    
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (reviewed_by) REFERENCES users(id) ON DELETE SET NULL,
    INDEX idx_content_moderation (content_type, content_id),
    INDEX idx_review_queue (requires_human_review, human_reviewed),
    INDEX idx_risk_level (risk_level, created_at)
);

-- ============================================================================
-- SUPPORT AND HELP SYSTEM
-- ============================================================================

-- Support tickets
CREATE TABLE support_tickets (
    id BIGINT PRIMARY KEY AUTO_INCREMENT,
    user_id BIGINT NOT NULL,
    
    -- Ticket details
    subject VARCHAR(200) NOT NULL,
    description TEXT NOT NULL,
    category ENUM('account', 'billing', 'technical', 'safety', 'general', 'bug_report', 'feature_request') NOT NULL,
    priority ENUM('low', 'medium', 'high', 'urgent') DEFAULT 'medium',
    
    -- Status tracking
    status ENUM('open', 'in_progress', 'waiting_response', 'resolved', 'closed') DEFAULT 'open',
    assigned_to BIGINT, -- Support agent
    
    -- Resolution
    resolution TEXT,
    satisfaction_rating INT, -- 1-5 scale
    satisfaction_feedback TEXT,
    
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    resolved_at TIMESTAMP,
    closed_at TIMESTAMP,
    
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (assigned_to) REFERENCES users(id) ON DELETE SET NULL,
    INDEX idx_support_tickets (status, priority, created_at),
    INDEX idx_user_tickets (user_id, status),
    INDEX idx_assigned_tickets (assigned_to, status)
);

-- Support ticket messages
CREATE TABLE support_messages (
    id BIGINT PRIMARY KEY AUTO_INCREMENT,
    ticket_id BIGINT NOT NULL,
    sender_id BIGINT NOT NULL,
    
    -- Message content
    message TEXT NOT NULL,
    attachments JSON, -- Array of attachment URLs
    
    -- Message type
    message_type ENUM('user_message', 'agent_message', 'system_note') DEFAULT 'user_message',
    is_internal BOOLEAN DEFAULT FALSE, -- Internal notes not visible to user
    
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    
    FOREIGN KEY (ticket_id) REFERENCES support_tickets(id) ON DELETE CASCADE,
    FOREIGN KEY (sender_id) REFERENCES users(id) ON DELETE CASCADE,
    INDEX idx_ticket_messages (ticket_id, created_at),
    INDEX idx_message_type (message_type, is_internal)
);

-- FAQ and knowledge base
CREATE TABLE knowledge_base_articles (
    id INT PRIMARY KEY AUTO_INCREMENT,
    
    -- Article content
    title VARCHAR(300) NOT NULL,
    content TEXT NOT NULL,
    excerpt TEXT,
    
    -- Organization
    category VARCHAR(100) NOT NULL,
    tags JSON, -- Array of tag strings
    
    -- SEO and metadata
    slug VARCHAR(200) UNIQUE NOT NULL,
    meta_description VARCHAR(500),
    
    -- Publishing
    status ENUM('draft', 'published', 'archived') DEFAULT 'draft',
    featured BOOLEAN DEFAULT FALSE,
    view_count INT DEFAULT 0,
    helpful_votes INT DEFAULT 0,
    unhelpful_votes INT DEFAULT 0,
    
    -- Management
    author_id BIGINT NOT NULL,
    
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    published_at TIMESTAMP,
    
    FOREIGN KEY (author_id) REFERENCES users(id) ON DELETE CASCADE,
    INDEX idx_kb_articles (status, category),
    INDEX idx_featured (featured, status),
    INDEX idx_slug (slug)
);

-- ============================================================================
-- USERNAME CHANGE SYSTEM
-- ============================================================================

-- Username change requests
CREATE TABLE username_change_requests (
    id BIGINT PRIMARY KEY AUTO_INCREMENT,
    user_id BIGINT NOT NULL,
    
    -- Request details
    current_username VARCHAR(50) NOT NULL,
    requested_username VARCHAR(50) NOT NULL,
    reason TEXT,
    
    -- AI suggestions
    ai_suggestions JSON, -- Array of suggested usernames
    
    -- Status
    status ENUM('pending', 'approved', 'rejected', 'completed') DEFAULT 'pending',
    reviewed_by BIGINT,
    review_notes TEXT,
    rejection_reason TEXT,
    
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    reviewed_at TIMESTAMP,
    completed_at TIMESTAMP,
    
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (reviewed_by) REFERENCES users(id) ON DELETE SET NULL,
    INDEX idx_username_requests (status, created_at),
    INDEX idx_user_requests (user_id, status),
    INDEX idx_requested_username (requested_username)
);

-- ============================================================================
-- ANALYTICS AND TRACKING TABLES
-- ============================================================================

-- User activity tracking
CREATE TABLE user_activity_logs (
    id BIGINT PRIMARY KEY AUTO_INCREMENT,
    user_id BIGINT,
    
    -- Activity details
    activity_type VARCHAR(100) NOT NULL,
    activity_data JSON,
    
    -- Context
    ip_address VARCHAR(45),
    user_agent TEXT,
    page_url VARCHAR(500),
    referrer_url VARCHAR(500),
    
    -- Location
    country VARCHAR(100),
    city VARCHAR(100),
    
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE SET NULL,
    INDEX idx_user_activity (user_id, activity_type, created_at),
    INDEX idx_activity_type (activity_type, created_at),
    INDEX idx_location (country, city)
);

-- System metrics and KPIs
CREATE TABLE system_metrics (
    id BIGINT PRIMARY KEY AUTO_INCREMENT,
    
    -- Metric details
    metric_name VARCHAR(100) NOT NULL,
    metric_value DECIMAL(15, 4) NOT NULL,
    metric_type ENUM('counter', 'gauge', 'histogram') DEFAULT 'gauge',
    
    -- Context
    tags JSON, -- Key-value pairs for filtering/grouping
    
    recorded_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    
    INDEX idx_metrics (metric_name, recorded_at),
    INDEX idx_metric_type (metric_type, recorded_at)
);

-- ============================================================================
-- CONFIGURATION AND SETTINGS TABLES
-- ============================================================================

-- System-wide settings
CREATE TABLE system_settings (
    id INT PRIMARY KEY AUTO_INCREMENT,
    setting_key VARCHAR(100) UNIQUE NOT NULL,
    setting_value TEXT NOT NULL,
    setting_type ENUM('string', 'integer', 'boolean', 'json') DEFAULT 'string',
    description TEXT,
    is_public BOOLEAN DEFAULT FALSE, -- Can be accessed by frontend
    
    updated_by BIGINT,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    
    FOREIGN KEY (updated_by) REFERENCES users(id) ON DELETE SET NULL,
    INDEX idx_setting_key (setting_key),
    INDEX idx_public_settings (is_public)
);

-- Country and location data
CREATE TABLE countries (
    id INT PRIMARY KEY AUTO_INCREMENT,
    country_code VARCHAR(3) UNIQUE NOT NULL,
    country_name VARCHAR(100) NOT NULL,
    continent VARCHAR(50),
    currency_code VARCHAR(3),
    phone_code VARCHAR(10),
    flag_emoji VARCHAR(10),
    is_supported BOOLEAN DEFAULT TRUE,
    
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    
    INDEX idx_country_code (country_code),
    INDEX idx_supported (is_supported)
);

-- Cities data
CREATE TABLE cities (
    id INT PRIMARY KEY AUTO_INCREMENT,
    country_id INT NOT NULL,
    city_name VARCHAR(100) NOT NULL,
    state_province VARCHAR(100),
    latitude DECIMAL(10, 8),
    longitude DECIMAL(11, 8),
    population INT,
    is_major_city BOOLEAN DEFAULT FALSE,
    
    FOREIGN KEY (country_id) REFERENCES countries(id) ON DELETE CASCADE,
    INDEX idx_country_cities (country_id, is_major_city),
    INDEX idx_coordinates (latitude, longitude),
    INDEX idx_city_name (city_name)
);

-- ============================================================================
-- TRIGGERS AND PROCEDURES
-- ============================================================================

-- Trigger to update user points balance
DELIMITER //
CREATE TRIGGER update_user_points_balance 
AFTER INSERT ON points_transactions
FOR EACH ROW
BEGIN
    IF NEW.transaction_type IN ('earned', 'bonus') THEN
        UPDATE user_points 
        SET current_balance = current_balance + NEW.points_amount,
            total_earned = total_earned + NEW.points_amount
        WHERE user_id = NEW.user_id;
    ELSEIF NEW.transaction_type IN ('spent', 'penalty') THEN
        UPDATE user_points 
        SET current_balance = current_balance - NEW.points_amount,
            total_spent = total_spent + NEW.points_amount
        WHERE user_id = NEW.user_id;
    ELSEIF NEW.transaction_type = 'refund' THEN
        UPDATE user_points 
        SET current_balance = current_balance + NEW.points_amount
        WHERE user_id = NEW.user_id;
    END IF;
END//
DELIMITER ;

-- Trigger to update conversation last message
DELIMITER //
CREATE TRIGGER update_conversation_last_message 
AFTER INSERT ON messages
FOR EACH ROW
BEGIN
    UPDATE conversations 
    SET last_message_id = NEW.id,
        last_message_at = NEW.created_at,
        updated_at = NEW.created_at
    WHERE id = NEW.conversation_id;
END//
DELIMITER ;

-- Trigger to update photo likes count
DELIMITER //
CREATE TRIGGER update_photo_likes_count 
AFTER INSERT ON photo_likes
FOR EACH ROW
BEGIN
    UPDATE user_photos 
    SET like_count = like_count + 1
    WHERE id = NEW.photo_id;
END//
DELIMITER ;

-- ============================================================================
-- INITIAL DATA SETUP
-- ============================================================================

-- Insert default subscription plans
INSERT INTO subscription_plans (name, description, tier, price_monthly, price_quarterly, price_yearly, features, daily_likes_limit, super_likes_per_day, photo_upload_limit) VALUES
('Basic', 'Essential features for finding connections', 'basic', 0.00, 0.00, 0.00, '["basic_search", "limited_likes", "basic_messaging"]', 10, 0, 3),
('Premium', 'Enhanced features for serious daters', 'premium', 19.99, 49.99, 149.99, '["unlimited_likes", "advanced_search", "see_who_liked_you", "message_read_receipts", "profile_boost"]', 999, 5, 9),
('VIP', 'Ultimate dating experience with all features', 'vip', 39.99, 99.99, 299.99, '["all_premium_features", "priority_support", "exclusive_events", "ai_matchmaking", "incognito_mode"]', 999, 10, 15);

-- Insert common interests
INSERT INTO interests (name, category) VALUES
('Travel', 'lifestyle'),
('Fitness', 'health'),
('Cooking', 'lifestyle'),
('Music', 'entertainment'),
('Movies', 'entertainment'),
('Reading', 'intellectual'),
('Art', 'creative'),
('Photography', 'creative'),
('Dancing', 'activity'),
('Hiking', 'outdoor'),
('Swimming', 'activity'),
('Yoga', 'health'),
('Technology', 'intellectual'),
('Business', 'professional'),
('Wine Tasting', 'lifestyle'),
('Coffee', 'lifestyle'),
('Gaming', 'entertainment'),
('Sports', 'activity'),
('Fashion', 'lifestyle'),
('Meditation', 'wellness');

-- Insert system settings
INSERT INTO system_settings (setting_key, setting_value, setting_type, description, is_public) VALUES
('site_name', 'SinglesZonne', 'string', 'Website name', true),
('registration_enabled', 'true', 'boolean', 'Allow new user registrations', false),
('maintenance_mode', 'false', 'boolean', 'Site maintenance mode', true),
('max_photos_per_user', '9', 'integer', 'Maximum photos per user profile', false),
('points_per_ad_view', '1', 'integer', 'Points earned per advertisement view', false),
('daily_login_bonus', '10', 'integer', 'Daily login bonus points', false),
('email_verification_required', 'true', 'boolean', 'Require email verification for new accounts', false),
('manual_photo_approval', 'true', 'boolean', 'Require manual approval for user photos', false);

-- Insert major countries
INSERT INTO countries (country_code, country_name, continent, currency_code, phone_code, flag_emoji, is_supported) VALUES
('US', 'United States', 'North America', 'USD', '+1', '🇺🇸', true),
('CA', 'Canada', 'North America', 'CAD', '+1', '🇨🇦', true),
('UK', 'United Kingdom', 'Europe', 'GBP', '+44', '🇬🇧', true),
('DE', 'Germany', 'Europe', 'EUR', '+49', '🇩🇪', true),
('FR', 'France', 'Europe', 'EUR', '+33', '🇫🇷', true),
('IT', 'Italy', 'Europe', 'EUR', '+39', '🇮🇹', true),
('ES', 'Spain', 'Europe', 'EUR', '+34', '🇪🇸', true),
('NL', 'Netherlands', 'Europe', 'EUR', '+31', '🇳🇱', true),
('AU', 'Australia', 'Oceania', 'AUD', '+61', '🇦🇺', true),
('JP', 'Japan', 'Asia', 'JPY', '+81', '🇯🇵', true),
('KR', 'South Korea', 'Asia', 'KRW', '+82', '🇰🇷', true),
('CN', 'China', 'Asia', 'CNY', '+86', '🇨🇳', true),
('IN', 'India', 'Asia', 'INR', '+91', '🇮🇳', true),
('BR', 'Brazil', 'South America', 'BRL', '+55', '🇧🇷', true),
('MX', 'Mexico', 'North America', 'MXN', '+52', '🇲🇽', true),
('AR', 'Argentina', 'South America', 'ARS', '+54', '🇦🇷', true),
('RU', 'Russia', 'Europe/Asia', 'RUB', '+7', '🇷🇺', true),
('ZA', 'South Africa', 'Africa', 'ZAR', '+27', '🇿🇦', true),
('EG', 'Egypt', 'Africa', 'EGP', '+20', '🇪🇬', true),
('TR', 'Turkey', 'Europe/Asia', 'TRY', '+90', '🇹🇷', true);

-- ============================================================================
-- INDEXES FOR PERFORMANCE
-- ============================================================================

-- Additional performance indexes
CREATE INDEX idx_users_location_age ON users(country_code, city, date_of_birth, status);
CREATE INDEX idx_users_premium_online ON users(is_premium, is_online, last_seen);
CREATE INDEX idx_matches_compatibility ON matches(compatibility_score DESC, matched_at DESC);
CREATE INDEX idx_messages_conversation_unread ON messages(conversation_id, is_read, created_at);
CREATE INDEX idx_user_interactions_recommendations ON user_interactions(target_user_id, interaction_type, created_at);
CREATE INDEX idx_profile_views_analytics ON profile_views(viewed_user_id, viewed_at, view_duration);
CREATE INDEX idx_ad_campaigns_active ON ad_campaigns(status, start_date, end_date);
CREATE INDEX idx_points_transactions_recent ON points_transactions(user_id, created_at DESC);
CREATE INDEX idx_support_tickets_queue ON support_tickets(status, priority, assigned_to, created_at);

-- Full-text search indexes
ALTER TABLE users ADD FULLTEXT(first_name, last_name, display_name);
ALTER TABLE users ADD FULLTEXT(bio);
ALTER TABLE groups ADD FULLTEXT(name, description);
ALTER TABLE events ADD FULLTEXT(title, description);
ALTER TABLE member_ads ADD FULLTEXT(title, description);
ALTER TABLE knowledge_base_articles ADD FULLTEXT(title, content);

-- ============================================================================
-- VIEWS FOR COMMON QUERIES
-- ============================================================================

-- Active users with basic profile info
CREATE VIEW active_users_view AS
SELECT 
    u.id,
    u.username,
    u.display_name,
    u.date_of_birth,
    YEAR(CURDATE()) - YEAR(u.date_of_birth) - (DATE_FORMAT(CURDATE(), '%m%d') < DATE_FORMAT(u.date_of_birth, '%m%d')) AS age,
    u.gender,
    u.city,
    u.country,
    u.is_premium,
    u.is_online,
    u.last_seen,
    (SELECT file_path FROM user_photos WHERE user_id = u.id AND is_primary = true AND moderation_status = 'approved' LIMIT 1) AS primary_photo
FROM users u
WHERE u.status = 'active' AND u.email_verified = true;

-- User match compatibility view
CREATE VIEW user_match_compatibility AS
SELECT 
    u1.id AS user1_id,
    u2.id AS user2_id,
    CASE 
        WHEN p1.min_age <= (YEAR(CURDATE()) - YEAR(u2.date_of_birth)) 
         AND p1.max_age >= (YEAR(CURDATE()) - YEAR(u2.date_of_birth))
         AND p2.min_age <= (YEAR(CURDATE()) - YEAR(u1.date_of_birth))
         AND p2.max_age >= (YEAR(CURDATE()) - YEAR(u1.date_of_birth))
        THEN 1 ELSE 0
    END AS age_compatible,
    CASE 
        WHEN ST_Distance_Sphere(
            POINT(u1.longitude, u1.latitude),
            POINT(u2.longitude, u2.latitude)
        ) / 1000 <= p1.max_distance
        THEN 1 ELSE 0
    END AS distance_compatible
FROM users u1
JOIN user_preferences p1 ON u1.id = p1.user_id
JOIN users u2 ON u1.id != u2.id
JOIN user_preferences p2 ON u2.id = p2.user_id
WHERE u1.status = 'active' AND u2.status = 'active';

-- ============================================================================
-- SAMPLE DATA FOR TESTING
-- ============================================================================

-- Create admin user (password should be hashed in real implementation)
INSERT INTO users (email, username, password_hash, salt, first_name, last_name, display_name, date_of_birth, gender, city, country, country_code, bio, occupation, education, is_premium, email_verified, status) VALUES
('admin@singleszone.com', 'administrator', SHA2('admin123salt', 256), 'salt', 'Site', 'Administrator', 'Administrator', '1990-01-01', 'other', 'New York', 'United States', 'US', 'Platform administrator and founder', 'CEO & Founder', 'MBA in Business Administration', true, true, 'active');

-- Assign super admin role
INSERT INTO user_roles (user_id, role, granted_by) VALUES
(1, 'super_admin', 1);

-- Initialize admin user points
INSERT INTO user_points (user_id, current_balance, total_earned) VALUES
(1, 1000, 1000);

-- Create user preferences for admin
INSERT INTO user_preferences (user_id, relationship_type, looking_for_text, min_age, max_age, max_distance, countries_preference, preferred_countries) VALUES
(1, 'serious', 'Looking for meaningful connections and helping build the best dating platform experience.', 25, 45, 50, 'multiple', '["US", "CA", "UK"]');

COMMIT;

-- ============================================================================
-- END OF SCHEMA
-- ============================================================================