import React, { useState, useRef, useEffect } from 'react';
import { ArrowLeft, Heart, X, Star, MapPin, Calendar, Briefcase, GraduationCap, Music, MoreHorizontal, Flag, Shield, Video, Play } from 'lucide-react';
import { Button } from '../ui/button';
import { Card } from '../ui/card';
import { Avatar, AvatarImage, AvatarFallback } from '../ui/avatar';
import { Badge } from '../ui/badge';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '../ui/tabs';
import { Separator } from '../ui/separator';
import { Sheet, SheetContent, SheetTrigger } from '../ui/sheet';
import { Dialog, DialogContent } from '../ui/dialog';
import { PhotoCollage } from '../PhotoCollage';
import { useTranslation } from '../TranslationProvider';
import { useUser } from '../UserProvider';
import { useMedia, useVideoManager, useMusicManager } from '../MediaManager';
import { motion, AnimatePresence } from 'motion/react';

interface MobileUserProfilePageProps {
  onNavigate: (view: string) => void;
  onOpenCookieConsent: () => void;
}

export function MobileUserProfilePage({ onNavigate, onOpenCookieConsent }: MobileUserProfilePageProps) {
  const { t } = useTranslation();
  const { user } = useUser();
  const [currentPhotoIndex, setCurrentPhotoIndex] = useState(0);
  const [showMoreOptions, setShowMoreOptions] = useState(false);
  const [showVideoModal, setShowVideoModal] = useState(false);

  const {
    introVideo,
    profileMusic,
    isVideoPlaying,
    isMusicPlaying,
    setIntroVideo,
    setProfileMusic,
    playMusic,
  } = useMedia();

  const videoRef = useRef<HTMLVideoElement>(null);
  const audioRef = useRef<HTMLAudioElement>(null);
  const videoCloseTimeoutRef = useRef<NodeJS.Timeout | null>(null);

  useVideoManager(videoRef);
  useMusicManager(audioRef);

  // Mock profile data - in real app this would come from props or route params
  const profile = {
    id: 1,
    name: 'Emma',
    age: 28,
    location: 'New York, NY',
    distance: '2 km away',
    photos: [
      'https://images.unsplash.com/photo-1494790108755-2616b612b786?w=600',
      'https://images.unsplash.com/photo-1517841905240-472988babdf9?w=600',
      'https://images.unsplash.com/photo-1496745160297-e1b9bedc9c39?w=600',
      'https://images.unsplash.com/photo-1544005313-94ddf0286df2?w=600'
    ],
    isOnline: true,
    isPremium: true,
    isVerified: true,
    lastSeen: 'Active now',
    compatibility: 89,
    bio: "Love traveling and good coffee ☕ Looking for someone to explore the world with! Adventure seeker, photography enthusiast, and always up for trying new restaurants. Let's create beautiful memories together! ✨",
    details: {
      profession: 'Travel Photographer',
      company: 'Wanderlust Media',
      education: 'Columbia University',
      height: '5\'6" (168 cm)',
      exercise: 'Regularly',
      smoking: 'Never',
      drinking: 'Socially',
      children: 'Want someday',
      religion: 'Christian',
      politics: 'Liberal'
    },
    interests: [
      'Photography', 'Travel', 'Coffee', 'Hiking', 'Art', 'Music', 
      'Cooking', 'Yoga', 'Reading', 'Dancing', 'Wine Tasting', 'Beaches'
    ],
    lookingFor: {
      ageRange: '25-35',
      location: 'Within 50 km',
      relationshipType: 'Long-term relationship',
      description: 'Looking for someone genuine who loves adventure and isn\'t afraid to be spontaneous. Must love travel and good conversations over coffee!'
    },
    music: {
      title: 'Favorite Song',
      artist: 'Ed Sheeran - Perfect',
      isPlaying: false
    },
    introVideo: 'https://commondatastorage.googleapis.com/gtv-videos-bucket/sample/BigBuckBunny.mp4',
    profileMusic: 'https://www.soundjay.com/misc/sounds/magic-chime-02.wav'
  };

  // Set intro video and profile music when component mounts
  useEffect(() => {
    if (profile.introVideo) {
      setIntroVideo(profile.introVideo);
    }
    if (profile.profileMusic) {
      setProfileMusic(profile.profileMusic);
    }
  }, [profile.introVideo, profile.profileMusic, setIntroVideo, setProfileMusic]);

  // Auto-play background music
  useEffect(() => {
    if (profile.profileMusic && audioRef.current && !isVideoPlaying) {
      audioRef.current.muted = false;
      audioRef.current.volume = 0.3;
      
      const tryAutoPlay = () => {
        if (audioRef.current && profile.profileMusic) {
          audioRef.current.play().then(() => {
            console.log('Profile music auto-playing');
            playMusic();
          }).catch((error) => {
            console.log('Auto-play prevented:', error.message);
          });
        }
      };

      tryAutoPlay();
      setTimeout(tryAutoPlay, 100);
      setTimeout(tryAutoPlay, 500);
    }
  }, [profile.profileMusic, isVideoPlaying, playMusic]);

  // Handle video ended event - auto-close 1 second after video finishes
  useEffect(() => {
    const video = videoRef.current;
    
    const handleVideoEnded = () => {
      if (videoCloseTimeoutRef.current) {
        clearTimeout(videoCloseTimeoutRef.current);
      }
      
      videoCloseTimeoutRef.current = setTimeout(() => {
        setShowVideoModal(false);
      }, 1000);
    };
    
    if (video) {
      video.addEventListener('ended', handleVideoEnded);
    }
    
    return () => {
      if (video) {
        video.removeEventListener('ended', handleVideoEnded);
      }
      if (videoCloseTimeoutRef.current) {
        clearTimeout(videoCloseTimeoutRef.current);
      }
    };
  }, [videoRef]);

  // Clean up timeout when modal closes manually
  useEffect(() => {
    if (!showVideoModal && videoCloseTimeoutRef.current) {
      clearTimeout(videoCloseTimeoutRef.current);
      videoCloseTimeoutRef.current = null;
    }
  }, [showVideoModal]);

  const handleLike = () => {
    // Handle like action
  };

  const handlePass = () => {
    // Handle pass action
    onNavigate('search');
  };

  const handleSuperLike = () => {
    // Handle super like action
  };

  const handleMessage = () => {
    onNavigate('messages');
  };

  return (
    <div className="min-h-screen bg-background">
      {/* Background music auto-plays */}
      {profile.profileMusic && (
        <audio
          ref={audioRef}
          src={profile.profileMusic}
          loop
          preload="auto"
          playsInline
          autoPlay
          muted={false}
        />
      )}

      {/* Video Modal */}
      <Dialog open={showVideoModal} onOpenChange={setShowVideoModal}>
        <DialogContent className="p-0 max-w-full h-full bg-black">
          {profile.introVideo && (
            <div className="relative w-full h-full flex items-center justify-center">
              <video
                ref={videoRef}
                src={profile.introVideo}
                className="w-full h-full object-contain"
                controls
                autoPlay
                muted={false}
                playsInline
                onPlay={() => {
                  // Video audio takes priority - ensure it's not muted
                  if (videoRef.current) {
                    videoRef.current.muted = false;
                  }
                }}
              />
              <div className="absolute top-4 right-4">
                <Button
                  size="sm"
                  variant="secondary"
                  onClick={() => setShowVideoModal(false)}
                  className="bg-white/20 text-white hover:bg-white/30 backdrop-blur-sm"
                >
                  Close
                </Button>
              </div>
            </div>
          )}
        </DialogContent>
      </Dialog>

      {/* Header */}
      <div className="sticky top-0 z-10 bg-background/95 backdrop-blur-sm border-b border-border">
        <div className="flex items-center justify-between p-4">
          <Button 
            variant="ghost" 
            size="sm" 
            className="p-2"
            onClick={() => onNavigate('search')}
          >
            <ArrowLeft className="h-5 w-5" />
          </Button>
          
          <div className="flex items-center gap-2">
            <div className="flex items-center gap-1 text-sm text-muted-foreground">
              <div className="w-2 h-2 bg-green-500 rounded-full" />
              {profile.lastSeen}
            </div>
          </div>
          
          <Sheet open={showMoreOptions} onOpenChange={setShowMoreOptions}>
            <SheetTrigger asChild>
              <Button variant="ghost" size="sm" className="p-2">
                <MoreHorizontal className="h-5 w-5" />
              </Button>
            </SheetTrigger>
            <SheetContent side="bottom" className="h-[40vh]">
              <div className="p-4 space-y-4">
                <h3 className="font-semibold">Profile Options</h3>
                <div className="space-y-2">
                  <Button variant="outline" className="w-full justify-start gap-3">
                    <Flag className="h-4 w-4" />
                    Report Profile
                  </Button>
                  <Button variant="outline" className="w-full justify-start gap-3">
                    <Shield className="h-4 w-4" />
                    Block User
                  </Button>
                </div>
              </div>
            </SheetContent>
          </Sheet>
        </div>
      </div>

      {/* Profile Content */}
      <div className="pb-24">
        {/* Photos Section */}
        <div className="relative">
          <div className="aspect-[3/4] relative">
            <img 
              src={profile.photos[currentPhotoIndex]} 
              alt={`${profile.name} photo ${currentPhotoIndex + 1}`}
              className="w-full h-full object-cover"
            />
            
            {/* Photo Navigation Dots */}
            <div className="absolute bottom-4 left-1/2 transform -translate-x-1/2 flex gap-2">
              {profile.photos.map((_, index) => (
                <button
                  key={index}
                  onClick={() => setCurrentPhotoIndex(index)}
                  className={`w-2 h-2 rounded-full transition-all ${
                    index === currentPhotoIndex ? 'bg-white' : 'bg-white/50'
                  }`}
                />
              ))}
            </div>

            {/* Swipe Navigation */}
            <div className="absolute inset-0 flex">
              <button 
                className="flex-1"
                onClick={() => setCurrentPhotoIndex(Math.max(0, currentPhotoIndex - 1))}
              />
              <button 
                className="flex-1"
                onClick={() => setCurrentPhotoIndex(Math.min(profile.photos.length - 1, currentPhotoIndex + 1))}
              />
            </div>

            {/* Intro Video Button */}
            {profile.introVideo && (
              <div className="absolute top-1/2 left-1/2 transform -translate-x-1/2 -translate-y-1/2">
                <Button
                  size="lg"
                  className="rounded-full w-16 h-16 bg-primary/80 hover:bg-primary backdrop-blur-sm"
                  onClick={() => setShowVideoModal(true)}
                >
                  <Play className="h-8 w-8" />
                </Button>
              </div>
            )}

            {/* Profile Badges */}
            <div className="absolute top-4 left-4 flex flex-col gap-2">
              {profile.isVerified && (
                <Badge className="bg-blue-500 text-white">
                  Verified
                </Badge>
              )}
              {profile.isPremium && (
                <Badge className="bg-yellow-500 text-black">
                  Premium
                </Badge>
              )}
            </div>

            {/* Compatibility Score */}
            <div className="absolute top-4 right-4">
              <div className="bg-primary text-primary-foreground px-3 py-1 rounded-full text-sm font-medium">
                {profile.compatibility}% Match
              </div>
            </div>
          </div>
        </div>

        {/* Profile Info */}
        <div className="p-4 space-y-6">
          {/* Basic Info */}
          <div>
            <div className="flex items-center justify-between mb-2">
              <h1 className="text-2xl font-bold">
                {profile.name}, {profile.age}
              </h1>
              {profile.music.title && (
                <Button variant="outline" size="sm" className="gap-2">
                  <Music className="h-4 w-4" />
                  <span className="text-xs">{profile.music.artist}</span>
                </Button>
              )}
            </div>
            
            <div className="flex items-center gap-2 text-muted-foreground mb-3">
              <MapPin className="h-4 w-4" />
              <span>{profile.distance}</span>
            </div>

            <div className="flex items-center gap-2">
              <Briefcase className="h-4 w-4 text-muted-foreground" />
              <span className="text-sm">{profile.details.profession}</span>
              {profile.details.company && (
                <>
                  <span className="text-muted-foreground">at</span>
                  <span className="text-sm">{profile.details.company}</span>
                </>
              )}
            </div>

            {profile.details.education && (
              <div className="flex items-center gap-2 mt-1">
                <GraduationCap className="h-4 w-4 text-muted-foreground" />
                <span className="text-sm">{profile.details.education}</span>
              </div>
            )}
          </div>

          {/* About Me */}
          <Card className="p-4">
            <h3 className="font-semibold mb-3">About Me</h3>
            <p className="text-sm leading-relaxed">{profile.bio}</p>
          </Card>

          {/* Details */}
          <Card className="p-4">
            <h3 className="font-semibold mb-3">Details</h3>
            <div className="grid grid-cols-2 gap-3 text-sm">
              <div>
                <span className="text-muted-foreground">Height</span>
                <p className="font-medium">{profile.details.height}</p>
              </div>
              <div>
                <span className="text-muted-foreground">Exercise</span>
                <p className="font-medium">{profile.details.exercise}</p>
              </div>
              <div>
                <span className="text-muted-foreground">Smoking</span>
                <p className="font-medium">{profile.details.smoking}</p>
              </div>
              <div>
                <span className="text-muted-foreground">Drinking</span>
                <p className="font-medium">{profile.details.drinking}</p>
              </div>
              <div>
                <span className="text-muted-foreground">Children</span>
                <p className="font-medium">{profile.details.children}</p>
              </div>
              <div>
                <span className="text-muted-foreground">Religion</span>
                <p className="font-medium">{profile.details.religion}</p>
              </div>
            </div>
          </Card>

          {/* Interests */}
          <Card className="p-4">
            <h3 className="font-semibold mb-3">Interests</h3>
            <div className="flex flex-wrap gap-2">
              {profile.interests.map((interest) => (
                <Badge key={interest} variant="secondary">
                  {interest}
                </Badge>
              ))}
            </div>
          </Card>

          {/* What I'm Looking For */}
          <Card className="p-4">
            <h3 className="font-semibold mb-3">What I'm Looking For</h3>
            
            <div className="space-y-3 mb-4 text-sm">
              <div className="flex justify-between">
                <span className="text-muted-foreground">Age range</span>
                <span className="font-medium">{profile.lookingFor.ageRange}</span>
              </div>
              <div className="flex justify-between">
                <span className="text-muted-foreground">Location</span>
                <span className="font-medium">{profile.lookingFor.location}</span>
              </div>
              <div className="flex justify-between">
                <span className="text-muted-foreground">Looking for</span>
                <span className="font-medium">{profile.lookingFor.relationshipType}</span>
              </div>
            </div>
            
            <Separator className="my-3" />
            
            <p className="text-sm leading-relaxed">{profile.lookingFor.description}</p>
          </Card>

          {/* Photo Grid */}
          <Card className="p-4">
            <h3 className="font-semibold mb-3">Photos ({profile.photos.length})</h3>
            <div className="grid grid-cols-3 gap-2">
              {profile.photos.map((photo, index) => (
                <button
                  key={index}
                  onClick={() => setCurrentPhotoIndex(index)}
                  className="aspect-square rounded-lg overflow-hidden"
                >
                  <img 
                    src={photo} 
                    alt={`${profile.name} photo ${index + 1}`}
                    className="w-full h-full object-cover"
                  />
                </button>
              ))}
            </div>
          </Card>
        </div>
      </div>

      {/* Action Buttons */}
      <div className="fixed bottom-0 left-0 right-0 bg-background/95 backdrop-blur-sm border-t border-border p-4">
        <div className="flex items-center justify-center gap-4">
          <motion.div whileTap={{ scale: 0.9 }}>
            <Button
              variant="outline"
              size="lg"
              className="w-14 h-14 rounded-full border-2 border-red-500 text-red-500 hover:bg-red-50"
              onClick={handlePass}
            >
              <X className="h-6 w-6" />
            </Button>
          </motion.div>

          <motion.div whileTap={{ scale: 0.9 }}>
            <Button
              variant="outline"
              size="lg"
              className="w-16 h-16 rounded-full border-2 border-blue-500 text-blue-500 hover:bg-blue-50"
              onClick={handleSuperLike}
            >
              <Star className="h-7 w-7" />
            </Button>
          </motion.div>

          <motion.div whileTap={{ scale: 0.9 }}>
            <Button
              variant="outline"
              size="lg"
              className="w-14 h-14 rounded-full border-2 border-green-500 text-green-500 hover:bg-green-50"
              onClick={handleLike}
            >
              <Heart className="h-6 w-6" />
            </Button>
          </motion.div>
        </div>

        <Button 
          className="w-full mt-4" 
          size="lg"
          onClick={handleMessage}
        >
          Send Message
        </Button>
      </div>
    </div>
  );
}