import React, { useState } from 'react';
import { ArrowLeft, Crown, Check, Star, Zap, Heart, Eye, MessageCircle, Shield } from 'lucide-react';
import { Button } from '../ui/button';
import { Card } from '../ui/card';
import { Badge } from '../ui/badge';
import { Separator } from '../ui/separator';
import { useTranslation } from '../TranslationProvider';
import { useUser } from '../UserProvider';
import { motion } from 'motion/react';

interface MobileSubscriptionPageProps {
  onNavigate: (view: string) => void;
  onOpenCookieConsent: () => void;
}

export function MobileSubscriptionPage({ onNavigate, onOpenCookieConsent }: MobileSubscriptionPageProps) {
  const { t } = useTranslation();
  const { user } = useUser();
  const [selectedPlan, setSelectedPlan] = useState<'premium' | 'gold' | 'platinum'>('premium');

  const plans = [
    {
      id: 'premium',
      name: 'Premium',
      price: '$19.99',
      period: '/month',
      color: 'from-blue-500 to-purple-600',
      icon: Crown,
      popular: false,
      features: [
        'Unlimited likes',
        'See who liked you',
        '5 Super Likes per day',
        'Boost your profile',
        'Advanced filters',
        'Read receipts'
      ]
    },
    {
      id: 'gold',
      name: 'Gold',
      price: '$29.99',
      period: '/month',
      color: 'from-yellow-500 to-orange-600',
      icon: Star,
      popular: true,
      features: [
        'All Premium features',
        'Unlimited Super Likes',
        'Monthly profile boost',
        'See who viewed your profile',
        'Priority customer support',
        'Advanced privacy controls',
        'Video chat features'
      ]
    },
    {
      id: 'platinum',
      name: 'Platinum',
      price: '$39.99',
      period: '/month',
      color: 'from-purple-600 to-pink-600',
      icon: Zap,
      popular: false,
      features: [
        'All Gold features',
        'Premium badge',
        'Weekly profile boost',
        'AI-powered matching',
        'Exclusive events access',
        'Personal dating coach',
        'Priority in search results',
        'Unlimited rewinds'
      ]
    }
  ];

  const currentPlan = plans.find(plan => plan.id === selectedPlan);

  const benefits = [
    {
      icon: Heart,
      title: 'More Matches',
      description: 'Get 3x more matches with premium features'
    },
    {
      icon: Eye,
      title: 'Better Visibility',
      description: 'Your profile gets seen by more people'
    },
    {
      icon: MessageCircle,
      title: 'Priority Support',
      description: '24/7 customer support available'
    },
    {
      icon: Shield,
      title: 'Enhanced Privacy',
      description: 'Advanced privacy and security controls'
    }
  ];

  return (
    <div className="min-h-screen bg-background">
      {/* Header */}
      <div className="sticky top-0 z-10 bg-background/95 backdrop-blur-sm border-b border-border">
        <div className="flex items-center justify-between p-4">
          <Button 
            variant="ghost" 
            size="sm" 
            className="p-2"
            onClick={() => onNavigate('search')}
          >
            <ArrowLeft className="h-5 w-5" />
          </Button>
          
          <h1 className="text-lg font-semibold">Subscription Plans</h1>
          
          <div className="w-9" /> {/* Spacer */}
        </div>
      </div>

      <div className="p-4 pb-8 space-y-6">
        {/* Current Status */}
        {user?.subscriptionTier && user.subscriptionTier !== 'free' && (
          <Card className="p-4 bg-gradient-to-r from-green-50 to-blue-50 dark:from-green-950 dark:to-blue-950 border-green-200 dark:border-green-800">
            <div className="flex items-center gap-3">
              <div className="p-2 bg-green-500 rounded-full text-white">
                <Crown className="h-4 w-4" />
              </div>
              <div>
                <h3 className="font-semibold text-green-800 dark:text-green-200">
                  You're on {user.subscriptionTier} Plan
                </h3>
                <p className="text-sm text-green-600 dark:text-green-300">
                  Enjoying premium features
                </p>
              </div>
            </div>
          </Card>
        )}

        {/* Benefits */}
        <div>
          <h2 className="text-xl font-bold mb-4">Why Upgrade?</h2>
          <div className="grid grid-cols-2 gap-4">
            {benefits.map((benefit) => (
              <Card key={benefit.title} className="p-4 text-center">
                <benefit.icon className="h-8 w-8 mx-auto mb-2 text-primary" />
                <h3 className="font-semibold text-sm mb-1">{benefit.title}</h3>
                <p className="text-xs text-muted-foreground">{benefit.description}</p>
              </Card>
            ))}
          </div>
        </div>

        {/* Plan Selection */}
        <div>
          <h2 className="text-xl font-bold mb-4">Choose Your Plan</h2>
          <div className="space-y-4">
            {plans.map((plan) => (
              <motion.div
                key={plan.id}
                whileTap={{ scale: 0.98 }}
                onClick={() => setSelectedPlan(plan.id as any)}
              >
                <Card 
                  className={`relative overflow-hidden cursor-pointer transition-all ${
                    selectedPlan === plan.id 
                      ? 'ring-2 ring-primary border-primary' 
                      : 'hover:border-primary/50'
                  }`}
                >
                  {plan.popular && (
                    <div className="absolute top-0 right-0 bg-primary text-primary-foreground px-3 py-1 text-xs font-medium rounded-bl-lg">
                      Most Popular
                    </div>
                  )}

                  <div className={`h-2 bg-gradient-to-r ${plan.color}`} />
                  
                  <div className="p-6">
                    <div className="flex items-center justify-between mb-4">
                      <div className="flex items-center gap-3">
                        <div className={`p-2 rounded-full bg-gradient-to-r ${plan.color} text-white`}>
                          <plan.icon className="h-5 w-5" />
                        </div>
                        <div>
                          <h3 className="text-lg font-bold">{plan.name}</h3>
                          <div className="flex items-baseline gap-1">
                            <span className="text-2xl font-bold">{plan.price}</span>
                            <span className="text-sm text-muted-foreground">{plan.period}</span>
                          </div>
                        </div>
                      </div>
                      
                      <div className={`w-5 h-5 rounded-full border-2 ${
                        selectedPlan === plan.id 
                          ? 'border-primary bg-primary' 
                          : 'border-muted-foreground'
                      } flex items-center justify-center`}>
                        {selectedPlan === plan.id && (
                          <Check className="h-3 w-3 text-primary-foreground" />
                        )}
                      </div>
                    </div>

                    <div className="space-y-2">
                      {plan.features.map((feature, index) => (
                        <div key={index} className="flex items-center gap-2">
                          <Check className="h-4 w-4 text-green-500 flex-shrink-0" />
                          <span className="text-sm">{feature}</span>
                        </div>
                      ))}
                    </div>
                  </div>
                </Card>
              </motion.div>
            ))}
          </div>
        </div>

        {/* Payment Summary */}
        <Card className="p-6">
          <h3 className="font-semibold mb-4">Payment Summary</h3>
          
          <div className="space-y-3 mb-4">
            <div className="flex justify-between">
              <span>{currentPlan?.name} Plan</span>
              <span>{currentPlan?.price}{currentPlan?.period}</span>
            </div>
            <div className="flex justify-between text-sm text-muted-foreground">
              <span>Taxes</span>
              <span>Calculated at checkout</span>
            </div>
            <Separator />
            <div className="flex justify-between font-semibold">
              <span>Total</span>
              <span>{currentPlan?.price}{currentPlan?.period}</span>
            </div>
          </div>

          <Button 
            className={`w-full bg-gradient-to-r ${currentPlan?.color} hover:opacity-90 text-white`}
            size="lg"
          >
            <Crown className="h-4 w-4 mr-2" />
            Upgrade to {currentPlan?.name}
          </Button>
          
          <p className="text-xs text-center text-muted-foreground mt-3">
            Cancel anytime. No hidden fees.
          </p>
        </Card>

        {/* Features Comparison */}
        <Card className="p-6">
          <h3 className="font-semibold mb-4">What's Included</h3>
          
          <div className="space-y-4">
            <div className="grid grid-cols-4 gap-2 text-xs font-medium">
              <div></div>
              <div className="text-center">Premium</div>
              <div className="text-center">Gold</div>
              <div className="text-center">Platinum</div>
            </div>
            
            <Separator />
            
            <div className="space-y-3 text-sm">
              {[
                { feature: 'Unlimited likes', premium: true, gold: true, platinum: true },
                { feature: 'See who liked you', premium: true, gold: true, platinum: true },
                { feature: 'Super Likes per day', premium: '5', gold: 'Unlimited', platinum: 'Unlimited' },
                { feature: 'Profile boosts', premium: '1/month', gold: '1/month', platinum: '1/week' },
                { feature: 'Advanced filters', premium: true, gold: true, platinum: true },
                { feature: 'Premium badge', premium: false, gold: false, platinum: true },
                { feature: 'AI matching', premium: false, gold: false, platinum: true }
              ].map((item, index) => (
                <div key={index} className="grid grid-cols-4 gap-2 items-center">
                  <div className="text-muted-foreground">{item.feature}</div>
                  <div className="text-center">
                    {typeof item.premium === 'boolean' ? (
                      item.premium ? <Check className="h-4 w-4 text-green-500 mx-auto" /> : '—'
                    ) : item.premium}
                  </div>
                  <div className="text-center">
                    {typeof item.gold === 'boolean' ? (
                      item.gold ? <Check className="h-4 w-4 text-green-500 mx-auto" /> : '—'
                    ) : item.gold}
                  </div>
                  <div className="text-center">
                    {typeof item.platinum === 'boolean' ? (
                      item.platinum ? <Check className="h-4 w-4 text-green-500 mx-auto" /> : '—'
                    ) : item.platinum}
                  </div>
                </div>
              ))}
            </div>
          </div>
        </Card>

        {/* FAQs */}
        <Card className="p-6">
          <h3 className="font-semibold mb-4">Frequently Asked Questions</h3>
          
          <div className="space-y-4 text-sm">
            <div>
              <h4 className="font-medium mb-1">Can I cancel anytime?</h4>
              <p className="text-muted-foreground">
                Yes, you can cancel your subscription at any time from your profile settings.
              </p>
            </div>
            
            <div>
              <h4 className="font-medium mb-1">Do you offer refunds?</h4>
              <p className="text-muted-foreground">
                We offer a 7-day money-back guarantee for first-time subscribers.
              </p>
            </div>
            
            <div>
              <h4 className="font-medium mb-1">What payment methods do you accept?</h4>
              <p className="text-muted-foreground">
                We accept all major credit cards, PayPal, and Apple Pay.
              </p>
            </div>
          </div>
        </Card>
      </div>
    </div>
  );
}