import React, { useState, useRef, useEffect } from 'react';
import { Search, Filter, MapPin, Heart, X, ChevronDown, Menu, Bell, User, Settings } from 'lucide-react';
import { Button } from '../ui/button';
import { Input } from '../ui/input';
import { Card } from '../ui/card';
import { Avatar, AvatarImage, AvatarFallback } from '../ui/avatar';
import { Badge } from '../ui/badge';
import { Sheet, SheetContent, SheetTrigger } from '../ui/sheet';
import { SearchFilters } from '../SearchFilters';
import { PhotoCollage } from '../PhotoCollage';
import { useTranslation } from '../TranslationProvider';
import { useUser } from '../UserProvider';
import { Header } from '../Header';

interface MobileSearchPageProps {
  onNavigate: (view: string) => void;
  onOpenCookieConsent: () => void;
}

export function MobileSearchPage({ onNavigate, onOpenCookieConsent }: MobileSearchPageProps) {
  const { t } = useTranslation();
  const { user } = useUser();
  const [searchQuery, setSearchQuery] = useState('');
  const [showFilters, setShowFilters] = useState(false);
  const [showMobileMenu, setShowMobileMenu] = useState(false);

  // Mock data for mobile display
  const profiles = [
    {
      id: 1,
      name: 'Emma',
      age: 28,
      location: 'New York',
      distance: '2 km',
      photos: [
        'https://images.unsplash.com/photo-1494790108755-2616b612b786?w=400',
        'https://images.unsplash.com/photo-1517841905240-472988babdf9?w=400'
      ],
      isOnline: true,
      isPremium: true,
      bio: 'Love traveling and good coffee ☕'
    },
    {
      id: 2,
      name: 'Sofia',
      age: 25,
      location: 'Los Angeles',
      distance: '5 km',
      photos: [
        'https://images.unsplash.com/photo-1534528741775-53994a69daeb?w=400',
        'https://images.unsplash.com/photo-1506863530036-1efeddceb993?w=400'
      ],
      isOnline: false,
      isPremium: false,
      bio: 'Fitness enthusiast and dog lover 🐕'
    },
    {
      id: 3,
      name: 'Isabella',
      age: 30,
      location: 'Miami',
      distance: '8 km',
      photos: [
        'https://images.unsplash.com/photo-1524504388940-b1c1722653e1?w=400',
        'https://images.unsplash.com/photo-1539571696357-5a69c17a67c6?w=400'
      ],
      isOnline: true,
      isPremium: true,
      bio: 'Artist and music lover 🎨🎵'
    }
  ];

  return (
    <div className="min-h-screen bg-background">
      {/* Mobile Header */}
      <div className="sticky top-0 z-40 bg-background/95 backdrop-blur-sm border-b border-border">
        <div className="flex items-center justify-between p-4">
          <Sheet open={showMobileMenu} onOpenChange={setShowMobileMenu}>
            <SheetTrigger asChild>
              <Button variant="ghost" size="sm" className="p-2">
                <Menu className="h-5 w-5" />
              </Button>
            </SheetTrigger>
            <SheetContent side="left" className="w-80">
              <div className="flex flex-col h-full">
                <div className="flex items-center gap-3 p-4 border-b">
                  <Avatar className="h-12 w-12">
                    <AvatarImage src={user?.profilePicture} />
                    <AvatarFallback>{user?.firstName?.[0]}</AvatarFallback>
                  </Avatar>
                  <div>
                    <p className="font-medium">{user?.firstName} {user?.lastName}</p>
                    <p className="text-sm text-muted-foreground">@{user?.username}</p>
                  </div>
                </div>
                
                <nav className="flex-1 py-4">
                  <div className="space-y-2">
                    <Button 
                      variant="ghost" 
                      className="w-full justify-start gap-3"
                      onClick={() => {
                        onNavigate('search');
                        setShowMobileMenu(false);
                      }}
                    >
                      <Search className="h-4 w-4" />
                      {t('search')}
                    </Button>
                    <Button 
                      variant="ghost" 
                      className="w-full justify-start gap-3"
                      onClick={() => {
                        onNavigate('discover');
                        setShowMobileMenu(false);
                      }}
                    >
                      <Heart className="h-4 w-4" />
                      Discover & Learn
                    </Button>
                    <Button 
                      variant="ghost" 
                      className="w-full justify-start gap-3"
                      onClick={() => {
                        onNavigate('matches');
                        setShowMobileMenu(false);
                      }}
                    >
                      <Heart className="h-4 w-4" />
                      {t('matches')}
                    </Button>
                    <Button 
                      variant="ghost" 
                      className="w-full justify-start gap-3"
                      onClick={() => {
                        onNavigate('messages');
                        setShowMobileMenu(false);
                      }}
                    >
                      <Bell className="h-4 w-4" />
                      {t('messages')}
                    </Button>
                    <Button 
                      variant="ghost" 
                      className="w-full justify-start gap-3"
                      onClick={() => {
                        onNavigate('profile-settings');
                        setShowMobileMenu(false);
                      }}
                    >
                      <Settings className="h-4 w-4" />
                      {t('settings')}
                    </Button>
                  </div>
                </nav>
              </div>
            </SheetContent>
          </Sheet>

          <h1 className="text-lg font-semibold">SinglesZone</h1>
          
          <Button 
            variant="ghost" 
            size="sm" 
            className="p-2"
            onClick={() => onNavigate('profile-settings')}
          >
            <Avatar className="h-6 w-6">
              <AvatarImage src={user?.profilePicture} />
              <AvatarFallback>{user?.firstName?.[0]}</AvatarFallback>
            </Avatar>
          </Button>
        </div>

        {/* Mobile Search Bar */}
        <div className="px-4 pb-4">
          <div className="relative">
            <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 h-4 w-4 text-muted-foreground" />
            <Input
              placeholder={t('searchPlaceholder')}
              value={searchQuery}
              onChange={(e) => setSearchQuery(e.target.value)}
              className="pl-10 pr-12"
            />
            <Button
              variant="ghost"
              size="sm"
              className="absolute right-1 top-1/2 transform -translate-y-1/2 p-2"
              onClick={() => setShowFilters(true)}
            >
              <Filter className="h-4 w-4" />
            </Button>
          </div>
        </div>
      </div>

      {/* Mobile Profile Cards */}
      <div className="p-4 pb-20">
        <div className="space-y-4">
          {profiles.map((profile) => (
            <Card key={profile.id} className="overflow-hidden">
              <div className="relative">
                <PhotoCollage
                  photos={profile.photos}
                  className="aspect-[4/3] w-full"
                  priority={false}
                />
                
                {/* Online Status */}
                {profile.isOnline && (
                  <div className="absolute top-3 left-3">
                    <div className="flex items-center gap-1 bg-green-500 text-white px-2 py-1 rounded-full text-xs">
                      <div className="w-2 h-2 bg-white rounded-full" />
                      Online
                    </div>
                  </div>
                )}

                {/* Premium Badge */}
                {profile.isPremium && (
                  <div className="absolute top-3 right-3">
                    <Badge variant="secondary" className="bg-yellow-500 text-black">
                      Premium
                    </Badge>
                  </div>
                )}

                {/* Quick Actions */}
                <div className="absolute bottom-3 right-3 flex gap-2">
                  <Button size="sm" variant="secondary" className="rounded-full p-2">
                    <Heart className="h-4 w-4" />
                  </Button>
                  <Button size="sm" variant="secondary" className="rounded-full p-2">
                    <X className="h-4 w-4" />
                  </Button>
                </div>
              </div>

              <div className="p-4">
                <div className="flex items-center justify-between mb-2">
                  <h3 className="font-semibold text-lg">{profile.name}, {profile.age}</h3>
                  <div className="flex items-center gap-1 text-muted-foreground text-sm">
                    <MapPin className="h-3 w-3" />
                    {profile.distance}
                  </div>
                </div>
                
                <p className="text-muted-foreground text-sm mb-2">{profile.location}</p>
                <p className="text-sm">{profile.bio}</p>
              </div>
            </Card>
          ))}
        </div>
      </div>

      {/* Mobile Bottom Navigation */}
      <div className="fixed bottom-0 left-0 right-0 bg-background/95 backdrop-blur-sm border-t border-border">
        <div className="flex items-center justify-around py-2">
          <Button 
            variant="ghost" 
            size="sm" 
            className="flex flex-col gap-1 p-2"
            onClick={() => onNavigate('search')}
          >
            <Search className="h-4 w-4" />
            <span className="text-xs">{t('search')}</span>
          </Button>
          <Button 
            variant="ghost" 
            size="sm" 
            className="flex flex-col gap-1 p-2"
            onClick={() => onNavigate('discover')}
          >
            <Heart className="h-4 w-4" />
            <span className="text-xs">Discover</span>
          </Button>
          <Button 
            variant="ghost" 
            size="sm" 
            className="flex flex-col gap-1 p-2"
            onClick={() => onNavigate('matches')}
          >
            <Heart className="h-4 w-4" />
            <span className="text-xs">{t('matches')}</span>
          </Button>
          <Button 
            variant="ghost" 
            size="sm" 
            className="flex flex-col gap-1 p-2"
            onClick={() => onNavigate('messages')}
          >
            <Bell className="h-4 w-4" />
            <span className="text-xs">{t('messages')}</span>
          </Button>
          <Button 
            variant="ghost" 
            size="sm" 
            className="flex flex-col gap-1 p-2"
            onClick={() => onNavigate('profile-settings')}
          >
            <User className="h-4 w-4" />
            <span className="text-xs">{t('profile')}</span>
          </Button>
        </div>
      </div>

      {/* Mobile Filters Sheet */}
      <Sheet open={showFilters} onOpenChange={setShowFilters}>
        <SheetContent side="bottom" className="h-[90vh]">
          <div className="p-4">
            <div className="flex items-center justify-between mb-6">
              <h2 className="text-lg font-semibold">{t('filters')}</h2>
              <Button 
                variant="ghost" 
                size="sm" 
                onClick={() => setShowFilters(false)}
              >
                <X className="h-4 w-4" />
              </Button>
            </div>
            <SearchFilters onApplyFilters={() => setShowFilters(false)} />
          </div>
        </SheetContent>
      </Sheet>
    </div>
  );
}