import React, { useState } from 'react';
import { Heart, MessageCircle, X, MapPin, Clock, Star, Filter, Search } from 'lucide-react';
import { Button } from '../ui/button';
import { Card } from '../ui/card';
import { Avatar, AvatarImage, AvatarFallback } from '../ui/avatar';
import { Badge } from '../ui/badge';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '../ui/tabs';
import { Input } from '../ui/input';
import { Sheet, SheetContent, SheetTrigger } from '../ui/sheet';
import { useTranslation } from '../TranslationProvider';
import { useUser } from '../UserProvider';

interface MobileMatchesPageProps {
  onNavigate: (view: string) => void;
  onOpenCookieConsent: () => void;
}

export function MobileMatchesPage({ onNavigate, onOpenCookieConsent }: MobileMatchesPageProps) {
  const { t } = useTranslation();
  const { user } = useUser();
  const [searchQuery, setSearchQuery] = useState('');
  const [selectedTab, setSelectedTab] = useState('matches');

  // Mock data
  const matches = [
    {
      id: 1,
      name: 'Emma',
      age: 28,
      photo: 'https://images.unsplash.com/photo-1494790108755-2616b612b786?w=400',
      location: 'New York',
      distance: '2 km',
      compatibility: 89,
      matchDate: '2 hours ago',
      isOnline: true,
      lastMessage: "Hey! Thanks for the like 😊",
      unread: true
    },
    {
      id: 2,
      name: 'Sofia',
      age: 25,
      photo: 'https://images.unsplash.com/photo-1534528741775-53994a69daeb?w=400',
      location: 'Los Angeles',
      distance: '5 km',
      compatibility: 76,
      matchDate: '1 day ago',
      isOnline: false,
      lastMessage: "Would love to chat more!",
      unread: false
    },
    {
      id: 3,
      name: 'Isabella',
      age: 30,
      photo: 'https://images.unsplash.com/photo-1524504388940-b1c1722653e1?w=400',
      location: 'Miami',
      distance: '8 km',
      compatibility: 92,
      matchDate: '3 days ago',
      isOnline: true,
      lastMessage: "Coffee sounds great!",
      unread: true
    }
  ];

  const likes = [
    {
      id: 4,
      name: 'Aria',
      age: 26,
      photo: 'https://images.unsplash.com/photo-1517841905240-472988babdf9?w=400',
      location: 'Chicago',
      compatibility: 84,
      isOnline: false
    },
    {
      id: 5,
      name: 'Luna',
      age: 29,
      photo: 'https://images.unsplash.com/photo-1506863530036-1efeddceb993?w=400',
      location: 'Seattle',
      compatibility: 78,
      isOnline: true
    }
  ];

  const superLikes = [
    {
      id: 6,
      name: 'Zara',
      age: 27,
      photo: 'https://images.unsplash.com/photo-1539571696357-5a69c17a67c6?w=400',
      location: 'Boston',
      compatibility: 95,
      isOnline: true
    }
  ];

  return (
    <div className="min-h-screen bg-background">
      {/* Mobile Header */}
      <div className="sticky top-0 z-10 bg-background/95 backdrop-blur-sm border-b border-border">
        <div className="p-4">
          <div className="flex items-center justify-between mb-4">
            <h1 className="text-xl font-bold">{t('matches')}</h1>
            <div className="flex gap-2">
              <Button variant="ghost" size="sm" className="p-2">
                <Filter className="h-4 w-4" />
              </Button>
              <Button variant="ghost" size="sm" className="p-2">
                <Search className="h-4 w-4" />
              </Button>
            </div>
          </div>

          {/* Search Bar */}
          <Input
            placeholder="Search matches..."
            value={searchQuery}
            onChange={(e) => setSearchQuery(e.target.value)}
            className="mb-4"
          />

          {/* Tabs */}
          <Tabs value={selectedTab} onValueChange={setSelectedTab} className="w-full">
            <TabsList className="grid w-full grid-cols-3">
              <TabsTrigger value="matches" className="relative">
                Matches
                {matches.some(m => m.unread) && (
                  <div className="absolute -top-1 -right-1 w-2 h-2 bg-red-500 rounded-full" />
                )}
              </TabsTrigger>
              <TabsTrigger value="likes">
                Likes ({likes.length})
              </TabsTrigger>
              <TabsTrigger value="super-likes">
                Super ({superLikes.length})
              </TabsTrigger>
            </TabsList>
          </Tabs>
        </div>
      </div>

      {/* Content */}
      <div className="p-4 pb-20">
        <Tabs value={selectedTab} className="w-full">
          {/* Matches Tab */}
          <TabsContent value="matches" className="mt-0">
            {matches.length === 0 ? (
              <div className="text-center py-12">
                <Heart className="h-12 w-12 text-muted-foreground mx-auto mb-4" />
                <h3 className="text-lg font-semibold mb-2">No matches yet</h3>
                <p className="text-muted-foreground mb-6">
                  Start swiping to find your perfect match!
                </p>
                <Button onClick={() => onNavigate('discover')}>
                  Start Discovering
                </Button>
              </div>
            ) : (
              <div className="space-y-4">
                {matches.map((match) => (
                  <Card key={match.id} className="overflow-hidden">
                    <div className="p-4">
                      <div className="flex items-center gap-4">
                        <div className="relative">
                          <Avatar className="h-16 w-16">
                            <AvatarImage src={match.photo} />
                            <AvatarFallback>{match.name[0]}</AvatarFallback>
                          </Avatar>
                          {match.isOnline && (
                            <div className="absolute -bottom-1 -right-1 w-5 h-5 bg-green-500 border-2 border-white rounded-full" />
                          )}
                        </div>

                        <div className="flex-1 min-w-0">
                          <div className="flex items-center gap-2 mb-1">
                            <h3 className="font-semibold truncate">
                              {match.name}, {match.age}
                            </h3>
                            <Badge variant="secondary" className="text-xs">
                              {match.compatibility}%
                            </Badge>
                          </div>
                          
                          <div className="flex items-center gap-1 text-sm text-muted-foreground mb-2">
                            <MapPin className="h-3 w-3" />
                            <span>{match.distance}</span>
                            <span>•</span>
                            <Clock className="h-3 w-3" />
                            <span>{match.matchDate}</span>
                          </div>

                          {match.lastMessage && (
                            <p className={`text-sm truncate ${match.unread ? 'font-medium text-foreground' : 'text-muted-foreground'}`}>
                              {match.lastMessage}
                            </p>
                          )}
                        </div>

                        {match.unread && (
                          <div className="w-3 h-3 bg-primary rounded-full flex-shrink-0" />
                        )}
                      </div>

                      <div className="flex gap-2 mt-4">
                        <Button 
                          variant="outline" 
                          size="sm" 
                          className="flex-1"
                          onClick={() => onNavigate('messages')}
                        >
                          <MessageCircle className="h-4 w-4 mr-1" />
                          Message
                        </Button>
                        <Button 
                          variant="outline" 
                          size="sm" 
                          className="flex-1"
                          onClick={() => onNavigate('user-profile')}
                        >
                          View Profile
                        </Button>
                      </div>
                    </div>
                  </Card>
                ))}
              </div>
            )}
          </TabsContent>

          {/* Likes Tab */}
          <TabsContent value="likes" className="mt-0">
            {likes.length === 0 ? (
              <div className="text-center py-12">
                <Heart className="h-12 w-12 text-muted-foreground mx-auto mb-4" />
                <h3 className="text-lg font-semibold mb-2">No likes yet</h3>
                <p className="text-muted-foreground">
                  People who like you will appear here
                </p>
              </div>
            ) : (
              <div className="grid grid-cols-2 gap-4">
                {likes.map((like) => (
                  <Card key={like.id} className="overflow-hidden">
                    <div className="aspect-[3/4] relative">
                      <img 
                        src={like.photo} 
                        alt={like.name}
                        className="w-full h-full object-cover"
                      />
                      <div className="absolute inset-0 bg-gradient-to-t from-black/60 to-transparent" />
                      
                      {like.isOnline && (
                        <div className="absolute top-2 left-2">
                          <div className="w-3 h-3 bg-green-500 rounded-full border-2 border-white" />
                        </div>
                      )}

                      <div className="absolute bottom-2 left-2 right-2 text-white">
                        <h3 className="font-semibold text-sm mb-1">
                          {like.name}, {like.age}
                        </h3>
                        <div className="flex items-center justify-between">
                          <span className="text-xs">{like.location}</span>
                          <Badge variant="secondary" className="text-xs bg-white/20 text-white">
                            {like.compatibility}%
                          </Badge>
                        </div>
                      </div>
                    </div>

                    <div className="p-3">
                      <div className="flex gap-2">
                        <Button variant="outline" size="sm" className="flex-1">
                          <X className="h-3 w-3 mr-1" />
                          Pass
                        </Button>
                        <Button size="sm" className="flex-1">
                          <Heart className="h-3 w-3 mr-1" />
                          Like
                        </Button>
                      </div>
                    </div>
                  </Card>
                ))}
              </div>
            )}
          </TabsContent>

          {/* Super Likes Tab */}
          <TabsContent value="super-likes" className="mt-0">
            {superLikes.length === 0 ? (
              <div className="text-center py-12">
                <Star className="h-12 w-12 text-yellow-500 mx-auto mb-4" />
                <h3 className="text-lg font-semibold mb-2">No super likes yet</h3>
                <p className="text-muted-foreground">
                  People who super liked you will appear here
                </p>
              </div>
            ) : (
              <div className="space-y-4">
                {superLikes.map((superLike) => (
                  <Card key={superLike.id} className="overflow-hidden border-yellow-500 border-2">
                    <div className="p-4">
                      <div className="flex items-center gap-4">
                        <div className="relative">
                          <Avatar className="h-20 w-20">
                            <AvatarImage src={superLike.photo} />
                            <AvatarFallback>{superLike.name[0]}</AvatarFallback>
                          </Avatar>
                          <div className="absolute -top-2 -right-2">
                            <div className="bg-yellow-500 text-white p-1 rounded-full">
                              <Star className="h-4 w-4" />
                            </div>
                          </div>
                          {superLike.isOnline && (
                            <div className="absolute -bottom-1 -left-1 w-5 h-5 bg-green-500 border-2 border-white rounded-full" />
                          )}
                        </div>

                        <div className="flex-1">
                          <div className="flex items-center gap-2 mb-2">
                            <h3 className="font-semibold text-lg">
                              {superLike.name}, {superLike.age}
                            </h3>
                            <Badge className="bg-yellow-500 text-black">
                              {superLike.compatibility}%
                            </Badge>
                          </div>
                          
                          <p className="text-muted-foreground text-sm mb-3">
                            {superLike.location}
                          </p>

                          <div className="bg-yellow-50 dark:bg-yellow-900/20 p-2 rounded-lg">
                            <p className="text-sm text-yellow-800 dark:text-yellow-200">
                              ⭐ This person super liked you!
                            </p>
                          </div>
                        </div>
                      </div>

                      <div className="flex gap-2 mt-4">
                        <Button variant="outline" size="sm" className="flex-1">
                          <X className="h-4 w-4 mr-1" />
                          Pass
                        </Button>
                        <Button size="sm" className="flex-1 bg-yellow-500 hover:bg-yellow-600 text-black">
                          <Heart className="h-4 w-4 mr-1" />
                          Like Back
                        </Button>
                      </div>
                    </div>
                  </Card>
                ))}
              </div>
            )}
          </TabsContent>
        </Tabs>
      </div>

      {/* Bottom Navigation */}
      <div className="fixed bottom-0 left-0 right-0 bg-background/95 backdrop-blur-sm border-t border-border">
        <div className="flex items-center justify-around py-2">
          <Button 
            variant="ghost" 
            size="sm" 
            className="flex flex-col gap-1 p-2"
            onClick={() => onNavigate('search')}
          >
            <Search className="h-4 w-4" />
            <span className="text-xs">Search</span>
          </Button>
          <Button 
            variant="ghost" 
            size="sm" 
            className="flex flex-col gap-1 p-2"
            onClick={() => onNavigate('discover')}
          >
            <Heart className="h-4 w-4" />
            <span className="text-xs">Discover</span>
          </Button>
          <Button 
            variant="ghost" 
            size="sm" 
            className="flex flex-col gap-1 p-2 text-primary"
          >
            <Heart className="h-4 w-4" />
            <span className="text-xs">Matches</span>
          </Button>
          <Button 
            variant="ghost" 
            size="sm" 
            className="flex flex-col gap-1 p-2"
            onClick={() => onNavigate('messages')}
          >
            <MessageCircle className="h-4 w-4" />
            <span className="text-xs">Messages</span>
          </Button>
          <Button 
            variant="ghost" 
            size="sm" 
            className="flex flex-col gap-1 p-2"
            onClick={() => onNavigate('profile-settings')}
          >
            <Avatar className="h-4 w-4">
              <AvatarImage src={user?.profilePicture} />
              <AvatarFallback>{user?.firstName?.[0]}</AvatarFallback>
            </Avatar>
            <span className="text-xs">Profile</span>
          </Button>
        </div>
      </div>
    </div>
  );
}