import React, { useState } from 'react';
import { 
  Video, 
  Mic, 
  MapPin, 
  Calendar, 
  Clock, 
  Users, 
  DollarSign, 
  Star,
  Search,
  Filter,
  Heart,
  Award,
  CheckCircle,
  ChevronRight,
  X
} from 'lucide-react';
import { Button } from '../ui/button';
import { Input } from '../ui/input';
import { Card } from '../ui/card';
import { Badge } from '../ui/badge';
import { Sheet, SheetContent, SheetHeader, SheetTitle, SheetDescription } from '../ui/sheet';
import { Avatar, AvatarImage, AvatarFallback } from '../ui/avatar';
import { Textarea } from '../ui/textarea';
import { Label } from '../ui/label';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '../ui/select';
import { toast } from 'sonner@2.0.3';

interface CoachSession {
  id: string;
  coachId: string;
  coachName: string;
  coachPhoto: string;
  title: string;
  description: string;
  sessionType: 'video' | 'audio';
  date: string;
  time: string;
  duration: number;
  maxParticipants: number;
  currentParticipants: number;
  price: number;
  location: string;
  country: string;
  category: string;
  rating: number;
  reviewCount: number;
  isVerified: boolean;
  tags: string[];
}

interface Coach {
  id: string;
  name: string;
  photo: string;
  bio: string;
  specialties: string[];
  rating: number;
  totalSessions: number;
  reviewCount: number;
  location: string;
  country: string;
  isVerified: boolean;
  yearsExperience: number;
}

// Mock data (same as desktop)
const mockSessions: CoachSession[] = [
  {
    id: '1',
    coachId: 'c1',
    coachName: 'Dr. Emily Roberts',
    coachPhoto: 'https://images.unsplash.com/photo-1494790108377-be9c29b29330?w=400',
    title: 'First Date Confidence Masterclass',
    description: 'Learn how to overcome first date anxiety and make a great impression. We\'ll cover conversation starters, body language, and building genuine connections.',
    sessionType: 'video',
    date: '2025-10-25',
    time: '18:00',
    duration: 60,
    maxParticipants: 20,
    currentParticipants: 12,
    price: 25,
    location: 'New York',
    country: 'USA',
    category: 'Confidence Building',
    rating: 4.8,
    reviewCount: 156,
    isVerified: true,
    tags: ['First Dates', 'Confidence', 'Communication']
  },
  {
    id: '2',
    coachId: 'c2',
    coachName: 'Marcus Johnson',
    coachPhoto: 'https://images.unsplash.com/photo-1507003211169-0a1dd7228f2d?w=400',
    title: 'Online Dating Profile Optimization',
    description: 'Transform your dating profile into a magnet! Get personalized feedback and learn proven strategies to stand out.',
    sessionType: 'video',
    date: '2025-10-23',
    time: '19:30',
    duration: 90,
    maxParticipants: 15,
    currentParticipants: 8,
    price: 0,
    location: 'London',
    country: 'UK',
    category: 'Profile Tips',
    rating: 4.9,
    reviewCount: 203,
    isVerified: true,
    tags: ['Profile', 'Photos', 'Bio Writing']
  },
  {
    id: '3',
    coachId: 'c3',
    coachName: 'Sofia Martinez',
    coachPhoto: 'https://images.unsplash.com/photo-1438761681033-6461ffad8d80?w=400',
    title: 'Building Emotional Intelligence in Relationships',
    description: 'Audio session focused on developing emotional awareness and communication skills for deeper connections.',
    sessionType: 'audio',
    date: '2025-10-24',
    time: '20:00',
    duration: 45,
    maxParticipants: 30,
    currentParticipants: 22,
    price: 15,
    location: 'Madrid',
    country: 'Spain',
    category: 'Emotional Intelligence',
    rating: 4.7,
    reviewCount: 89,
    isVerified: true,
    tags: ['Emotions', 'Communication', 'Self-Awareness']
  },
  {
    id: '4',
    coachId: 'c4',
    coachName: 'David Chen',
    coachPhoto: 'https://images.unsplash.com/photo-1500648767791-00dcc994a43e?w=400',
    title: 'Red Flags & Green Flags: What to Look For',
    description: 'Learn to identify healthy relationship patterns and warning signs early on. Interactive Q&A session included.',
    sessionType: 'video',
    date: '2025-10-26',
    time: '17:00',
    duration: 75,
    maxParticipants: 25,
    currentParticipants: 18,
    price: 30,
    location: 'Singapore',
    country: 'Singapore',
    category: 'Relationship Advice',
    rating: 4.9,
    reviewCount: 178,
    isVerified: true,
    tags: ['Red Flags', 'Dating Safety', 'Compatibility']
  }
];

const mockCoaches: Coach[] = [
  {
    id: 'c1',
    name: 'Dr. Emily Roberts',
    photo: 'https://images.unsplash.com/photo-1494790108377-be9c29b29330?w=400',
    bio: 'Licensed therapist and dating coach with over 10 years of experience helping singles build confidence and find meaningful connections.',
    specialties: ['Confidence Building', 'First Dates', 'Communication Skills'],
    rating: 4.8,
    totalSessions: 234,
    reviewCount: 156,
    location: 'New York',
    country: 'USA',
    isVerified: true,
    yearsExperience: 10
  },
  {
    id: 'c2',
    name: 'Marcus Johnson',
    photo: 'https://images.unsplash.com/photo-1507003211169-0a1dd7228f2d?w=400',
    bio: 'Former matchmaker turned dating coach. Specializing in online dating strategies and profile optimization.',
    specialties: ['Profile Tips', 'Online Dating', 'First Messages'],
    rating: 4.9,
    totalSessions: 312,
    reviewCount: 203,
    location: 'London',
    country: 'UK',
    isVerified: true,
    yearsExperience: 8
  }
];

const categories = [
  'All Categories',
  'Confidence Building',
  'Profile Tips',
  'First Dates',
  'Communication',
  'Emotional Intelligence',
  'Relationship Advice',
  'Dating Safety'
];

export default function MobileDatingCoachesPage() {
  const [activeView, setActiveView] = useState<'sessions' | 'coaches'>('sessions');
  const [searchQuery, setSearchQuery] = useState('');
  const [showFilters, setShowFilters] = useState(false);
  const [selectedCategory, setSelectedCategory] = useState('All Categories');
  const [sessionType, setSessionType] = useState<'all' | 'video' | 'audio'>('all');
  const [priceFilter, setPriceFilter] = useState<'all' | 'free' | 'paid'>('all');
  const [selectedSession, setSelectedSession] = useState<CoachSession | null>(null);
  const [selectedCoach, setSelectedCoach] = useState<Coach | null>(null);
  const [showRSVPSheet, setShowRSVPSheet] = useState(false);
  const [showApplicationSheet, setShowApplicationSheet] = useState(false);
  const [applicationForm, setApplicationForm] = useState({
    fullName: '',
    email: '',
    phone: '',
    location: '',
    country: '',
    yearsExperience: '',
    specialties: '',
    bio: '',
    certifications: '',
    why: ''
  });

  const filteredSessions = mockSessions.filter(session => {
    const matchesSearch = session.title.toLowerCase().includes(searchQuery.toLowerCase()) ||
                         session.coachName.toLowerCase().includes(searchQuery.toLowerCase()) ||
                         session.location.toLowerCase().includes(searchQuery.toLowerCase());
    const matchesCategory = selectedCategory === 'All Categories' || session.category === selectedCategory;
    const matchesType = sessionType === 'all' || session.sessionType === sessionType;
    const matchesPrice = priceFilter === 'all' || 
                        (priceFilter === 'free' && session.price === 0) ||
                        (priceFilter === 'paid' && session.price > 0);
    
    return matchesSearch && matchesCategory && matchesType && matchesPrice;
  });

  const filteredCoaches = mockCoaches.filter(coach => {
    const matchesSearch = coach.name.toLowerCase().includes(searchQuery.toLowerCase()) ||
                         coach.location.toLowerCase().includes(searchQuery.toLowerCase()) ||
                         coach.specialties.some(s => s.toLowerCase().includes(searchQuery.toLowerCase()));
    
    return matchesSearch;
  });

  const handleRSVP = (session: CoachSession) => {
    setSelectedSession(session);
    setShowRSVPSheet(true);
  };

  const confirmRSVP = () => {
    if (selectedSession) {
      if (selectedSession.price > 0) {
        toast.success(`Payment of $${selectedSession.price} processed. You're registered!`);
      } else {
        toast.success(`You're registered for "${selectedSession.title}"!`);
      }
      setShowRSVPSheet(false);
      setSelectedSession(null);
    }
  };

  const handleApplicationSubmit = (e: React.FormEvent) => {
    e.preventDefault();
    toast.success('Application submitted! We\'ll review it within 2-3 business days.');
    setShowApplicationSheet(false);
    setApplicationForm({
      fullName: '',
      email: '',
      phone: '',
      location: '',
      country: '',
      yearsExperience: '',
      specialties: '',
      bio: '',
      certifications: '',
      why: ''
    });
  };

  return (
    <div className="min-h-screen bg-gradient-to-br from-pink-50 via-purple-50 to-blue-50 dark:from-gray-900 dark:via-purple-900/20 dark:to-gray-900 pb-20">
      {/* Header */}
      <div className="bg-white/80 dark:bg-gray-800/80 backdrop-blur-sm border-b dark:border-gray-700 sticky top-0 z-10">
        <div className="p-4">
          <div className="flex items-center justify-between mb-3">
            <div>
              <h1 className="text-xl text-pink-600 dark:text-pink-400">Dating Coaches</h1>
              <p className="text-sm text-gray-600 dark:text-gray-400">Expert guidance for your love life</p>
            </div>
            <Button 
              onClick={() => setShowApplicationSheet(true)}
              size="sm"
              className="bg-gradient-to-r from-pink-500 to-purple-500"
            >
              <Award className="w-4 h-4 mr-1" />
              Apply
            </Button>
          </div>

          {/* Search */}
          <div className="relative mb-3">
            <Search className="absolute left-3 top-1/2 -translate-y-1/2 w-4 h-4 text-gray-400" />
            <Input
              placeholder="Search coaches or topics..."
              value={searchQuery}
              onChange={(e) => setSearchQuery(e.target.value)}
              className="pl-10 pr-10"
            />
            <Button
              size="sm"
              variant="ghost"
              onClick={() => setShowFilters(true)}
              className="absolute right-1 top-1/2 -translate-y-1/2"
            >
              <Filter className="w-4 h-4" />
            </Button>
          </div>

          {/* View Toggle */}
          <div className="flex gap-2">
            <Button
              size="sm"
              variant={activeView === 'sessions' ? 'default' : 'outline'}
              onClick={() => setActiveView('sessions')}
              className={activeView === 'sessions' ? 'flex-1 bg-gradient-to-r from-pink-500 to-purple-500' : 'flex-1'}
            >
              <Calendar className="w-4 h-4 mr-1" />
              Sessions
            </Button>
            <Button
              size="sm"
              variant={activeView === 'coaches' ? 'default' : 'outline'}
              onClick={() => setActiveView('coaches')}
              className={activeView === 'coaches' ? 'flex-1 bg-gradient-to-r from-pink-500 to-purple-500' : 'flex-1'}
            >
              <Users className="w-4 h-4 mr-1" />
              Coaches
            </Button>
          </div>
        </div>

        {/* Stats Bar */}
        <div className="grid grid-cols-4 gap-2 px-4 pb-4">
          <div className="text-center">
            <div className="text-lg font-semibold text-pink-600 dark:text-pink-400">127</div>
            <div className="text-xs text-gray-600 dark:text-gray-400">Coaches</div>
          </div>
          <div className="text-center">
            <div className="text-lg font-semibold text-purple-600 dark:text-purple-400">248</div>
            <div className="text-xs text-gray-600 dark:text-gray-400">Sessions</div>
          </div>
          <div className="text-center">
            <div className="text-lg font-semibold text-blue-600 dark:text-blue-400">4.8</div>
            <div className="text-xs text-gray-600 dark:text-gray-400">Rating</div>
          </div>
          <div className="text-center">
            <div className="text-lg font-semibold text-green-600 dark:text-green-400">1.8k</div>
            <div className="text-xs text-gray-600 dark:text-gray-400">Success</div>
          </div>
        </div>
      </div>

      {/* Content */}
      <div className="p-4 space-y-4">
        {activeView === 'sessions' ? (
          filteredSessions.length === 0 ? (
            <Card className="p-8 text-center bg-white/80 dark:bg-gray-800/80 backdrop-blur-sm">
              <Calendar className="w-12 h-12 mx-auto mb-3 text-gray-400" />
              <p className="text-gray-600 dark:text-gray-400">No sessions found</p>
              <Button
                size="sm"
                variant="link"
                onClick={() => setShowFilters(true)}
                className="mt-2"
              >
                Adjust filters
              </Button>
            </Card>
          ) : (
            filteredSessions.map(session => (
              <Card 
                key={session.id} 
                className="p-4 bg-white/80 dark:bg-gray-800/80 backdrop-blur-sm active:scale-98 transition-transform"
              >
                {/* Coach Info */}
                <div className="flex items-start gap-3 mb-3">
                  <Avatar className="w-12 h-12 ring-2 ring-pink-200 dark:ring-pink-800">
                    <AvatarImage src={session.coachPhoto} alt={session.coachName} />
                    <AvatarFallback>{session.coachName[0]}</AvatarFallback>
                  </Avatar>
                  <div className="flex-1 min-w-0">
                    <div className="flex items-center gap-1 mb-1">
                      <span className="font-medium truncate">{session.coachName}</span>
                      {session.isVerified && (
                        <CheckCircle className="w-3.5 h-3.5 text-blue-500 flex-shrink-0" />
                      )}
                    </div>
                    <div className="flex items-center gap-2 text-xs text-gray-600 dark:text-gray-400">
                      <div className="flex items-center gap-0.5">
                        <Star className="w-3 h-3 text-yellow-500 fill-yellow-500" />
                        <span>{session.rating}</span>
                      </div>
                      <span>•</span>
                      <div className="flex items-center gap-0.5 truncate">
                        <MapPin className="w-3 h-3 flex-shrink-0" />
                        <span className="truncate">{session.location}</span>
                      </div>
                    </div>
                  </div>
                </div>

                {/* Session Details */}
                <div className="mb-3">
                  <h3 className="text-base mb-1">{session.title}</h3>
                  <p className="text-sm text-gray-600 dark:text-gray-400 line-clamp-2 mb-2">
                    {session.description}
                  </p>
                  
                  {/* Tags */}
                  <div className="flex flex-wrap gap-1.5 mb-3">
                    {session.tags.slice(0, 3).map(tag => (
                      <Badge key={tag} variant="secondary" className="text-xs">
                        {tag}
                      </Badge>
                    ))}
                  </div>

                  {/* Session Info */}
                  <div className="grid grid-cols-2 gap-2 p-2.5 bg-gray-50 dark:bg-gray-900/50 rounded-lg text-xs">
                    <div className="flex items-center gap-1.5">
                      {session.sessionType === 'video' ? (
                        <Video className="w-3.5 h-3.5 text-purple-500 flex-shrink-0" />
                      ) : (
                        <Mic className="w-3.5 h-3.5 text-blue-500 flex-shrink-0" />
                      )}
                      <span className="capitalize truncate">{session.sessionType}</span>
                    </div>
                    <div className="flex items-center gap-1.5">
                      <Calendar className="w-3.5 h-3.5 text-pink-500 flex-shrink-0" />
                      <span className="truncate">{new Date(session.date).toLocaleDateString()}</span>
                    </div>
                    <div className="flex items-center gap-1.5">
                      <Clock className="w-3.5 h-3.5 text-orange-500 flex-shrink-0" />
                      <span className="truncate">{session.time}</span>
                    </div>
                    <div className="flex items-center gap-1.5">
                      <Users className="w-3.5 h-3.5 text-green-500 flex-shrink-0" />
                      <span className="truncate">{session.currentParticipants}/{session.maxParticipants}</span>
                    </div>
                  </div>
                </div>

                {/* Footer */}
                <div className="flex items-center justify-between pt-3 border-t dark:border-gray-700">
                  <div>
                    {session.price === 0 ? (
                      <Badge className="bg-green-500 text-white">Free</Badge>
                    ) : (
                      <div className="flex items-center gap-0.5">
                        <DollarSign className="w-4 h-4 text-gray-600 dark:text-gray-400" />
                        <span className="text-lg font-semibold">{session.price}</span>
                      </div>
                    )}
                  </div>
                  <Button
                    size="sm"
                    onClick={() => handleRSVP(session)}
                    disabled={session.currentParticipants >= session.maxParticipants}
                    className="bg-gradient-to-r from-pink-500 to-purple-500"
                  >
                    {session.currentParticipants >= session.maxParticipants ? 'Full' : 'Reserve'}
                  </Button>
                </div>
              </Card>
            ))
          )
        ) : (
          // Coaches View
          filteredCoaches.length === 0 ? (
            <Card className="p-8 text-center bg-white/80 dark:bg-gray-800/80 backdrop-blur-sm">
              <Users className="w-12 h-12 mx-auto mb-3 text-gray-400" />
              <p className="text-gray-600 dark:text-gray-400">No coaches found</p>
            </Card>
          ) : (
            filteredCoaches.map(coach => (
              <Card 
                key={coach.id} 
                className="p-4 bg-white/80 dark:bg-gray-800/80 backdrop-blur-sm active:scale-98 transition-transform"
                onClick={() => setSelectedCoach(coach)}
              >
                <div className="flex items-start gap-4">
                  <Avatar className="w-16 h-16 ring-2 ring-pink-200 dark:ring-pink-800">
                    <AvatarImage src={coach.photo} alt={coach.name} />
                    <AvatarFallback>{coach.name[0]}</AvatarFallback>
                  </Avatar>
                  <div className="flex-1 min-w-0">
                    <div className="flex items-center gap-1.5 mb-1">
                      <h3 className="text-base truncate">{coach.name}</h3>
                      {coach.isVerified && (
                        <CheckCircle className="w-4 h-4 text-blue-500 flex-shrink-0" />
                      )}
                    </div>
                    <div className="flex items-center gap-2 text-xs text-gray-600 dark:text-gray-400 mb-2">
                      <div className="flex items-center gap-0.5">
                        <Star className="w-3 h-3 text-yellow-500 fill-yellow-500" />
                        <span>{coach.rating}</span>
                      </div>
                      <span>•</span>
                      <span>{coach.totalSessions} sessions</span>
                    </div>
                    <p className="text-sm text-gray-600 dark:text-gray-400 line-clamp-2 mb-2">
                      {coach.bio}
                    </p>
                    <div className="flex flex-wrap gap-1.5">
                      {coach.specialties.slice(0, 2).map(specialty => (
                        <Badge key={specialty} variant="secondary" className="text-xs">
                          {specialty}
                        </Badge>
                      ))}
                      {coach.specialties.length > 2 && (
                        <Badge variant="secondary" className="text-xs">
                          +{coach.specialties.length - 2}
                        </Badge>
                      )}
                    </div>
                  </div>
                  <ChevronRight className="w-5 h-5 text-gray-400 flex-shrink-0" />
                </div>
              </Card>
            ))
          )
        )}
      </div>

      {/* Filters Sheet */}
      <Sheet open={showFilters} onOpenChange={setShowFilters}>
        <SheetContent side="bottom" className="h-[80vh]">
          <SheetHeader>
            <SheetTitle>Filters</SheetTitle>
          </SheetHeader>
          <div className="space-y-4 mt-6">
            <div>
              <Label>Category</Label>
              <Select value={selectedCategory} onValueChange={setSelectedCategory}>
                <SelectTrigger>
                  <SelectValue />
                </SelectTrigger>
                <SelectContent>
                  {categories.map(cat => (
                    <SelectItem key={cat} value={cat}>{cat}</SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>
            <div>
              <Label>Session Type</Label>
              <Select value={sessionType} onValueChange={(v: any) => setSessionType(v)}>
                <SelectTrigger>
                  <SelectValue />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="all">All Types</SelectItem>
                  <SelectItem value="video">Video Only</SelectItem>
                  <SelectItem value="audio">Audio Only</SelectItem>
                </SelectContent>
              </Select>
            </div>
            <div>
              <Label>Price</Label>
              <Select value={priceFilter} onValueChange={(v: any) => setPriceFilter(v)}>
                <SelectTrigger>
                  <SelectValue />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="all">All Sessions</SelectItem>
                  <SelectItem value="free">Free Only</SelectItem>
                  <SelectItem value="paid">Paid Only</SelectItem>
                </SelectContent>
              </Select>
            </div>
            <Button 
              onClick={() => setShowFilters(false)}
              className="w-full bg-gradient-to-r from-pink-500 to-purple-500"
            >
              Apply Filters
            </Button>
          </div>
        </SheetContent>
      </Sheet>

      {/* RSVP Sheet */}
      <Sheet open={showRSVPSheet} onOpenChange={setShowRSVPSheet}>
        <SheetContent side="bottom" className="h-[90vh] overflow-y-auto">
          <SheetHeader>
            <SheetTitle>Reserve Your Spot</SheetTitle>
            <SheetDescription>Confirm your registration</SheetDescription>
          </SheetHeader>
          {selectedSession && (
            <div className="space-y-4 mt-6">
              <Card className="p-4 bg-gray-50 dark:bg-gray-900/50">
                <div className="flex items-start gap-3 mb-3">
                  <Avatar className="w-12 h-12">
                    <AvatarImage src={selectedSession.coachPhoto} alt={selectedSession.coachName} />
                    <AvatarFallback>{selectedSession.coachName[0]}</AvatarFallback>
                  </Avatar>
                  <div className="flex-1">
                    <div className="font-medium mb-1">{selectedSession.title}</div>
                    <div className="text-sm text-gray-600 dark:text-gray-400">{selectedSession.coachName}</div>
                  </div>
                </div>
                <div className="grid grid-cols-2 gap-2 text-sm">
                  <div className="flex items-center gap-1">
                    <Calendar className="w-4 h-4" />
                    <span>{new Date(selectedSession.date).toLocaleDateString()}</span>
                  </div>
                  <div className="flex items-center gap-1">
                    <Clock className="w-4 h-4" />
                    <span>{selectedSession.time}</span>
                  </div>
                </div>
              </Card>

              {selectedSession.price > 0 && (
                <Card className="p-4 bg-blue-50 dark:bg-blue-900/20">
                  <div className="flex items-center justify-between mb-2">
                    <span className="text-gray-600 dark:text-gray-400">Session Fee</span>
                    <span className="text-2xl font-semibold">${selectedSession.price}</span>
                  </div>
                  <p className="text-sm text-gray-600 dark:text-gray-400">
                    Secure payment. Confirmation email with session link will be sent.
                  </p>
                </Card>
              )}

              <Card className="p-4 bg-yellow-50 dark:bg-yellow-900/20">
                <p className="text-sm text-yellow-800 dark:text-yellow-200">
                  <strong>Important:</strong> You'll receive the session link 30 minutes before start time.
                </p>
              </Card>

              <div className="flex gap-3">
                <Button
                  variant="outline"
                  onClick={() => setShowRSVPSheet(false)}
                  className="flex-1"
                >
                  Cancel
                </Button>
                <Button
                  onClick={confirmRSVP}
                  className="flex-1 bg-gradient-to-r from-pink-500 to-purple-500"
                >
                  {selectedSession.price > 0 ? `Pay $${selectedSession.price}` : 'Register Free'}
                </Button>
              </div>
            </div>
          )}
        </SheetContent>
      </Sheet>

      {/* Coach Application Sheet */}
      <Sheet open={showApplicationSheet} onOpenChange={setShowApplicationSheet}>
        <SheetContent side="bottom" className="h-[95vh] overflow-y-auto">
          <SheetHeader>
            <SheetTitle>Apply to Become a Coach</SheetTitle>
            <SheetDescription>We'll review your application within 2-3 business days</SheetDescription>
          </SheetHeader>
          <form onSubmit={handleApplicationSubmit} className="space-y-4 mt-6">
            <div>
              <Label htmlFor="fullName">Full Name *</Label>
              <Input
                id="fullName"
                required
                value={applicationForm.fullName}
                onChange={(e) => setApplicationForm({...applicationForm, fullName: e.target.value})}
              />
            </div>
            <div>
              <Label htmlFor="email">Email *</Label>
              <Input
                id="email"
                type="email"
                required
                value={applicationForm.email}
                onChange={(e) => setApplicationForm({...applicationForm, email: e.target.value})}
              />
            </div>
            <div>
              <Label htmlFor="phone">Phone Number *</Label>
              <Input
                id="phone"
                required
                value={applicationForm.phone}
                onChange={(e) => setApplicationForm({...applicationForm, phone: e.target.value})}
              />
            </div>
            <div>
              <Label htmlFor="yearsExperience">Years of Experience *</Label>
              <Input
                id="yearsExperience"
                type="number"
                required
                value={applicationForm.yearsExperience}
                onChange={(e) => setApplicationForm({...applicationForm, yearsExperience: e.target.value})}
              />
            </div>
            <div>
              <Label htmlFor="location">City/Location *</Label>
              <Input
                id="location"
                required
                value={applicationForm.location}
                onChange={(e) => setApplicationForm({...applicationForm, location: e.target.value})}
              />
            </div>
            <div>
              <Label htmlFor="country">Country *</Label>
              <Input
                id="country"
                required
                value={applicationForm.country}
                onChange={(e) => setApplicationForm({...applicationForm, country: e.target.value})}
              />
            </div>
            <div>
              <Label htmlFor="specialties">Areas of Expertise *</Label>
              <Input
                id="specialties"
                placeholder="e.g., First Dates, Communication"
                required
                value={applicationForm.specialties}
                onChange={(e) => setApplicationForm({...applicationForm, specialties: e.target.value})}
              />
            </div>
            <div>
              <Label htmlFor="certifications">Certifications & Credentials</Label>
              <Textarea
                id="certifications"
                rows={3}
                value={applicationForm.certifications}
                onChange={(e) => setApplicationForm({...applicationForm, certifications: e.target.value})}
              />
            </div>
            <div>
              <Label htmlFor="bio">Professional Bio * (min 100 characters)</Label>
              <Textarea
                id="bio"
                rows={4}
                required
                value={applicationForm.bio}
                onChange={(e) => setApplicationForm({...applicationForm, bio: e.target.value})}
              />
            </div>
            <div>
              <Label htmlFor="why">Why do you want to be a coach? *</Label>
              <Textarea
                id="why"
                rows={4}
                required
                value={applicationForm.why}
                onChange={(e) => setApplicationForm({...applicationForm, why: e.target.value})}
              />
            </div>

            <Card className="p-4 bg-blue-50 dark:bg-blue-900/20">
              <h4 className="font-medium mb-2 text-sm">What happens next?</h4>
              <ul className="text-xs text-gray-600 dark:text-gray-400 space-y-1">
                <li>• Application review (2-3 days)</li>
                <li>• Email with setup instructions</li>
                <li>• Complete profile & schedule sessions</li>
                <li>• Start helping singles!</li>
              </ul>
            </Card>

            <div className="flex gap-3 pb-6">
              <Button
                type="button"
                variant="outline"
                onClick={() => setShowApplicationSheet(false)}
                className="flex-1"
              >
                Cancel
              </Button>
              <Button
                type="submit"
                className="flex-1 bg-gradient-to-r from-pink-500 to-purple-500"
              >
                Submit
              </Button>
            </div>
          </form>
        </SheetContent>
      </Sheet>

      {/* Coach Details Sheet */}
      <Sheet open={selectedCoach !== null} onOpenChange={() => setSelectedCoach(null)}>
        <SheetContent side="bottom" className="h-[90vh] overflow-y-auto">
          {selectedCoach && (
            <div className="space-y-6">
              <div className="flex items-start gap-4">
                <Avatar className="w-20 h-20 ring-4 ring-pink-200 dark:ring-pink-800">
                  <AvatarImage src={selectedCoach.photo} alt={selectedCoach.name} />
                  <AvatarFallback>{selectedCoach.name[0]}</AvatarFallback>
                </Avatar>
                <div className="flex-1">
                  <div className="flex items-center gap-2 mb-2">
                    <h2 className="text-xl">{selectedCoach.name}</h2>
                    {selectedCoach.isVerified && (
                      <CheckCircle className="w-5 h-5 text-blue-500" />
                    )}
                  </div>
                  <div className="flex items-center gap-3 text-sm mb-2">
                    <div className="flex items-center gap-1">
                      <Star className="w-4 h-4 text-yellow-500 fill-yellow-500" />
                      <span>{selectedCoach.rating}</span>
                      <span className="text-gray-400">({selectedCoach.reviewCount})</span>
                    </div>
                    <div className="flex items-center gap-1">
                      <MapPin className="w-4 h-4" />
                      <span>{selectedCoach.location}</span>
                    </div>
                  </div>
                  <div className="flex items-center gap-4 text-sm">
                    <div>
                      <span className="text-gray-500">Sessions: </span>
                      <span className="font-medium">{selectedCoach.totalSessions}</span>
                    </div>
                    <div>
                      <span className="text-gray-500">Experience: </span>
                      <span className="font-medium">{selectedCoach.yearsExperience}y</span>
                    </div>
                  </div>
                </div>
              </div>

              <div>
                <h3 className="mb-2">About</h3>
                <p className="text-sm text-gray-600 dark:text-gray-400">{selectedCoach.bio}</p>
              </div>

              <div>
                <h3 className="mb-3">Specialties</h3>
                <div className="flex flex-wrap gap-2">
                  {selectedCoach.specialties.map(specialty => (
                    <Badge key={specialty} className="bg-gradient-to-r from-pink-500 to-purple-500 text-white">
                      {specialty}
                    </Badge>
                  ))}
                </div>
              </div>

              <div>
                <h3 className="mb-3">Upcoming Sessions</h3>
                <div className="space-y-3">
                  {mockSessions
                    .filter(s => s.coachId === selectedCoach.id)
                    .map(session => (
                      <Card key={session.id} className="p-3 bg-gray-50 dark:bg-gray-900/50">
                        <div className="flex items-start justify-between gap-3">
                          <div className="flex-1">
                            <h4 className="text-sm font-medium mb-2">{session.title}</h4>
                            <div className="flex items-center gap-3 text-xs text-gray-600 dark:text-gray-400">
                              <div className="flex items-center gap-1">
                                <Calendar className="w-3 h-3" />
                                <span>{new Date(session.date).toLocaleDateString()}</span>
                              </div>
                              <div className="flex items-center gap-1">
                                <Clock className="w-3 h-3" />
                                <span>{session.time}</span>
                              </div>
                            </div>
                          </div>
                          <Button
                            size="sm"
                            onClick={() => {
                              setSelectedCoach(null);
                              handleRSVP(session);
                            }}
                            className="bg-gradient-to-r from-pink-500 to-purple-500"
                          >
                            {session.price === 0 ? 'Free' : `$${session.price}`}
                          </Button>
                        </div>
                      </Card>
                    ))}
                </div>
              </div>
            </div>
          )}
        </SheetContent>
      </Sheet>
    </div>
  );
}
