import React, { useState } from 'react';
import { Plus, Filter, Search, MapPin, Clock, Star, Heart, MessageCircle, Share2 } from 'lucide-react';
import { Button } from '../ui/button';
import { Card } from '../ui/card';
import { Avatar, AvatarImage, AvatarFallback } from '../ui/avatar';
import { Badge } from '../ui/badge';
import { Input } from '../ui/input';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '../ui/tabs';
import { Sheet, SheetContent, SheetTrigger } from '../ui/sheet';
import { useTranslation } from '../TranslationProvider';
import { useUser } from '../UserProvider';

interface MobileAdsPageProps {
  onNavigate: (view: string) => void;
  onOpenCookieConsent: () => void;
}

export function MobileAdsPage({ onNavigate, onOpenCookieConsent }: MobileAdsPageProps) {
  const { t } = useTranslation();
  const { user } = useUser();
  const [searchQuery, setSearchQuery] = useState('');
  const [selectedCategory, setSelectedCategory] = useState('all');
  const [showFilters, setShowFilters] = useState(false);

  // Mock ads data
  const ads = [
    {
      id: 1,
      title: 'Coffee Date Anyone?',
      description: 'Looking for someone to join me for coffee this weekend. I love trying new cafes around the city!',
      author: {
        name: 'Emma',
        age: 28,
        photo: 'https://images.unsplash.com/photo-1494790108755-2616b612b786?w=400',
        location: 'New York'
      },
      category: 'meetup',
      location: 'Manhattan, NY',
      timestamp: '2 hours ago',
      responses: 15,
      likes: 23,
      isPremium: true,
      photos: ['https://images.unsplash.com/photo-1495474472287-4d71bcdd2085?w=400']
    },
    {
      id: 2,
      title: 'Concert Buddy Needed',
      description: 'Have an extra ticket to the indie concert this Friday. Would love to go with someone who enjoys good music!',
      author: {
        name: 'Sofia',
        age: 25,
        photo: 'https://images.unsplash.com/photo-1534528741775-53994a69daeb?w=400',
        location: 'Los Angeles'
      },
      category: 'events',
      location: 'Downtown LA',
      timestamp: '5 hours ago',
      responses: 8,
      likes: 32,
      isPremium: false,
      photos: ['https://images.unsplash.com/photo-1459749411175-04bf5292ceea?w=400']
    },
    {
      id: 3,
      title: 'Hiking Adventure',
      description: 'Planning a hiking trip this Sunday morning. Looking for fellow nature lovers to join the adventure!',
      author: {
        name: 'Isabella',
        age: 30,
        photo: 'https://images.unsplash.com/photo-1524504388940-b1c1722653e1?w=400',
        location: 'Miami'
      },
      category: 'activities',
      location: 'Everglades National Park',
      timestamp: '1 day ago',
      responses: 12,
      likes: 45,
      isPremium: true,
      photos: ['https://images.unsplash.com/photo-1551632811-561732d1e306?w=400']
    }
  ];

  const categories = [
    { id: 'all', label: 'All', count: ads.length },
    { id: 'meetup', label: 'Meetups', count: 1 },
    { id: 'events', label: 'Events', count: 1 },
    { id: 'activities', label: 'Activities', count: 1 }
  ];

  const filteredAds = selectedCategory === 'all' 
    ? ads 
    : ads.filter(ad => ad.category === selectedCategory);

  return (
    <div className="min-h-screen bg-background">
      {/* Mobile Header */}
      <div className="sticky top-0 z-10 bg-background/95 backdrop-blur-sm border-b border-border">
        <div className="p-4">
          <div className="flex items-center justify-between mb-4">
            <h1 className="text-xl font-bold">Member Ads</h1>
            <div className="flex gap-2">
              <Button
                variant="ghost"
                size="sm"
                className="p-2"
                onClick={() => setShowFilters(true)}
              >
                <Filter className="h-4 w-4" />
              </Button>
              <Button
                size="sm"
                className="px-4"
                onClick={() => {
                  // Handle create ad
                }}
              >
                <Plus className="h-4 w-4 mr-1" />
                Post Ad
              </Button>
            </div>
          </div>

          {/* Search Bar */}
          <div className="relative mb-4">
            <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 h-4 w-4 text-muted-foreground" />
            <Input
              placeholder="Search ads..."
              value={searchQuery}
              onChange={(e) => setSearchQuery(e.target.value)}
              className="pl-10"
            />
          </div>

          {/* Categories */}
          <div className="flex gap-2 overflow-x-auto scrollbar-hide">
            {categories.map((category) => (
              <Button
                key={category.id}
                variant={selectedCategory === category.id ? "default" : "outline"}
                size="sm"
                className="whitespace-nowrap"
                onClick={() => setSelectedCategory(category.id)}
              >
                {category.label} ({category.count})
              </Button>
            ))}
          </div>
        </div>
      </div>

      {/* Ads List */}
      <div className="p-4 pb-20 space-y-4">
        {filteredAds.map((ad) => (
          <Card key={ad.id} className="overflow-hidden">
            {/* Ad Header */}
            <div className="p-4 pb-0">
              <div className="flex items-center justify-between mb-3">
                <div className="flex items-center gap-3">
                  <Avatar className="h-10 w-10">
                    <AvatarImage src={ad.author.photo} />
                    <AvatarFallback>{ad.author.name[0]}</AvatarFallback>
                  </Avatar>
                  <div>
                    <div className="flex items-center gap-2">
                      <h3 className="font-semibold text-sm">{ad.author.name}, {ad.author.age}</h3>
                      {ad.isPremium && (
                        <Badge className="bg-yellow-500 text-black text-xs px-1">
                          Premium
                        </Badge>
                      )}
                    </div>
                    <div className="flex items-center gap-1 text-xs text-muted-foreground">
                      <MapPin className="h-3 w-3" />
                      <span>{ad.author.location}</span>
                      <span>•</span>
                      <Clock className="h-3 w-3" />
                      <span>{ad.timestamp}</span>
                    </div>
                  </div>
                </div>
                
                <Badge variant="secondary" className="text-xs">
                  {categories.find(c => c.id === ad.category)?.label}
                </Badge>
              </div>

              {/* Ad Content */}
              <div className="mb-3">
                <h2 className="font-bold text-lg mb-2">{ad.title}</h2>
                <p className="text-sm text-muted-foreground mb-2">{ad.description}</p>
                <div className="flex items-center gap-1 text-sm text-muted-foreground">
                  <MapPin className="h-3 w-3" />
                  <span>{ad.location}</span>
                </div>
              </div>
            </div>

            {/* Ad Photos */}
            {ad.photos && ad.photos.length > 0 && (
              <div className="px-4 mb-3">
                <div className="grid grid-cols-1 gap-2">
                  {ad.photos.map((photo, index) => (
                    <img
                      key={index}
                      src={photo}
                      alt={`${ad.title} photo`}
                      className="w-full h-48 object-cover rounded-lg"
                    />
                  ))}
                </div>
              </div>
            )}

            {/* Ad Actions */}
            <div className="border-t border-border p-3">
              <div className="flex items-center justify-between">
                <div className="flex items-center gap-4">
                  <button className="flex items-center gap-1 text-sm text-muted-foreground hover:text-foreground">
                    <Heart className="h-4 w-4" />
                    <span>{ad.likes}</span>
                  </button>
                  <button className="flex items-center gap-1 text-sm text-muted-foreground hover:text-foreground">
                    <MessageCircle className="h-4 w-4" />
                    <span>{ad.responses}</span>
                  </button>
                  <button className="flex items-center gap-1 text-sm text-muted-foreground hover:text-foreground">
                    <Share2 className="h-4 w-4" />
                    <span>Share</span>
                  </button>
                </div>

                <Button size="sm" variant="outline">
                  Contact
                </Button>
              </div>
            </div>
          </Card>
        ))}

        {filteredAds.length === 0 && (
          <div className="text-center py-12">
            <MessageCircle className="h-12 w-12 text-muted-foreground mx-auto mb-4" />
            <h3 className="text-lg font-semibold mb-2">No ads found</h3>
            <p className="text-muted-foreground mb-6">
              Be the first to post an ad in this category!
            </p>
            <Button>
              <Plus className="h-4 w-4 mr-2" />
              Create Ad
            </Button>
          </div>
        )}
      </div>

      {/* Filters Sheet */}
      <Sheet open={showFilters} onOpenChange={setShowFilters}>
        <SheetContent side="bottom" className="h-[60vh]">
          <div className="p-4">
            <h2 className="text-lg font-semibold mb-6">Filters</h2>
            
            <div className="space-y-6">
              <div>
                <h3 className="font-medium mb-3">Category</h3>
                <div className="space-y-2">
                  {categories.map((category) => (
                    <button
                      key={category.id}
                      className={`w-full text-left p-3 rounded-lg border transition-colors ${
                        selectedCategory === category.id
                          ? 'border-primary bg-primary/10'
                          : 'border-border hover:bg-muted/50'
                      }`}
                      onClick={() => {
                        setSelectedCategory(category.id);
                        setShowFilters(false);
                      }}
                    >
                      <div className="flex items-center justify-between">
                        <span>{category.label}</span>
                        <Badge variant="secondary">{category.count}</Badge>
                      </div>
                    </button>
                  ))}
                </div>
              </div>

              <div className="flex gap-3 pt-4">
                <Button
                  variant="outline"
                  className="flex-1"
                  onClick={() => setShowFilters(false)}
                >
                  Cancel
                </Button>
                <Button
                  className="flex-1"
                  onClick={() => setShowFilters(false)}
                >
                  Apply Filters
                </Button>
              </div>
            </div>
          </div>
        </SheetContent>
      </Sheet>

      {/* Bottom Navigation */}
      <div className="fixed bottom-0 left-0 right-0 bg-background/95 backdrop-blur-sm border-t border-border">
        <div className="flex items-center justify-around py-2">
          <Button 
            variant="ghost" 
            size="sm" 
            className="flex flex-col gap-1 p-2"
            onClick={() => onNavigate('search')}
          >
            <Search className="h-4 w-4" />
            <span className="text-xs">Search</span>
          </Button>
          <Button 
            variant="ghost" 
            size="sm" 
            className="flex flex-col gap-1 p-2"
            onClick={() => onNavigate('discover')}
          >
            <Heart className="h-4 w-4" />
            <span className="text-xs">Discover</span>
          </Button>
          <Button 
            variant="ghost" 
            size="sm" 
            className="flex flex-col gap-1 p-2"
            onClick={() => onNavigate('matches')}
          >
            <Heart className="h-4 w-4" />
            <span className="text-xs">Matches</span>
          </Button>
          <Button 
            variant="ghost" 
            size="sm" 
            className="flex flex-col gap-1 p-2"
            onClick={() => onNavigate('messages')}
          >
            <MessageCircle className="h-4 w-4" />
            <span className="text-xs">Messages</span>
          </Button>
          <Button 
            variant="ghost" 
            size="sm" 
            className="flex flex-col gap-1 p-2 text-primary"
          >
            <div className="h-4 w-4" />
            <span className="text-xs">Ads</span>
          </Button>
        </div>
      </div>
    </div>
  );
}