import { useState, useEffect } from "react";
import { Button } from "./ui/button";
import { Input } from "./ui/input";
import { Label } from "./ui/label";
import { Alert, AlertDescription } from "./ui/alert";
import { Badge } from "./ui/badge";
import { Card, CardContent, CardHeader, CardTitle } from "./ui/card";
import { Check, AlertTriangle, RefreshCw, Sparkles, User, Heart, Star, Zap } from "lucide-react";

interface UsernameHelperProps {
  value: string;
  onChange: (value: string) => void;
  firstName?: string;
  interests?: string[];
  className?: string;
  onValidationChange?: (isValid: boolean) => void;
}

// Mock database of existing usernames (in real app, this would be from backend)
const existingUsernames = new Set([
  // Popular dating app usernames
  "musiclover", "adventureseeker", "coffeeaddict", "bookwormbella", "traveler123",
  "dancequeen", "fitnessfan", "artlover", "techguru", "naturelover", "foodie",
  "photographer", "gamer", "moviebuff", "sports_fan", "chef_mike", "yoga_girl",
  "rock_climber", "beach_lover", "city_explorer", "wine_enthusiast", "dog_lover",
  "cat_person", "hiking_buddy", "sunset_chaser", "coffee_shop", "book_club",
  "gym_rat", "travel_bug", "music_maker", "art_creator", "tech_savvy", "nature_walker",
  
  // Common romantic usernames
  "heartseeker", "lovebug", "soulmate123", "dreamlover", "cupid", "sweetheart",
  "romantic", "passionate", "tender", "gentle_soul", "caring_heart", "lover_boy",
  "princess", "charming", "handsome", "beautiful", "gorgeous", "stunning",
  
  // Trendy modern usernames
  "digitalnomad", "wanderlust", "bohemian", "hipster", "trendsetter", "influencer",
  "lifestyle", "minimalist", "wellness", "mindfulness", "zen_master", "guru",
  "explorer", "wanderer", "dreamer", "creator", "innovator", "entrepreneur",
  
  // Common first names with numbers
  "john123", "sarah21", "mike99", "anna88", "david77", "lisa55", "chris24",
  "emma19", "alex_23", "maria_18", "james_99", "kate_77", "tom_88", "lucy_55",
  
  // Single letter combinations that are often taken
  "a", "aa", "aaa", "test", "user", "admin", "hello", "world", "love", "life"
]);

// AI-powered username suggestion engine
class UsernameAI {
  static generateSuggestions(firstName: string = "", interests: string[] = []): string[] {
    const suggestions: string[] = [];
    const adjectives = [
      "Amazing", "Bright", "Creative", "Dynamic", "Energetic", "Fun", "Gentle",
      "Happy", "Inspired", "Joyful", "Kind", "Lovely", "Mystic", "Noble",
      "Optimistic", "Peaceful", "Quick", "Radiant", "Sweet", "Vibrant",
      "Charming", "Clever", "Dreamy", "Elegant", "Fierce", "Graceful",
      "Witty", "Zen", "Bold", "Cosmic", "Divine", "Epic", "Fresh", "Golden"
    ];
    
    const suffixes = [
      "Seeker", "Lover", "Explorer", "Dreamer", "Hunter", "Finder", "Chaser",
      "Walker", "Rider", "Dancer", "Singer", "Player", "Creator", "Maker",
      "Star", "Soul", "Heart", "Spirit", "Wizard", "Angel", "Knight", "Queen",
      "King", "Warrior", "Artist", "Poet", "Sage", "Phoenix", "Tiger", "Wolf"
    ];
    
    const numbers = ["", "123", "2024", "21", "99", "77", "55", "88", "24", "19"];
    
    // Generate suggestions based on first name
    if (firstName) {
      const cleanFirstName = firstName.toLowerCase().replace(/[^a-z]/g, '');
      
      // First name + adjective (both ways)
      adjectives.slice(0, 6).forEach(adj => {
        suggestions.push(`${adj}${cleanFirstName}`);
        suggestions.push(`${cleanFirstName}${adj}`);
        suggestions.push(`${adj}_${cleanFirstName}`);
      });
      
      // First name + suffix
      suffixes.slice(0, 8).forEach(suffix => {
        suggestions.push(`${cleanFirstName}${suffix}`);
        suggestions.push(`${cleanFirstName}_${suffix}`);
      });
      
      // First name + numbers
      numbers.slice(0, 4).forEach(num => {
        if (num) {
          suggestions.push(`${cleanFirstName}${num}`);
          suggestions.push(`${cleanFirstName}_${num}`);
        }
      });
    }
    
    // Generate suggestions based on interests
    interests.slice(0, 3).forEach(interest => {
      const cleanInterest = interest.toLowerCase().replace(/[^a-z]/g, '');
      
      // Interest + suffix
      suffixes.slice(0, 3).forEach(suffix => {
        suggestions.push(`${cleanInterest}${suffix}`);
      });
      
      // Adjective + interest
      adjectives.slice(0, 3).forEach(adj => {
        suggestions.push(`${adj}${cleanInterest}`);
      });
      
      // Interest + numbers
      numbers.slice(1, 3).forEach(num => {
        suggestions.push(`${cleanInterest}${num}`);
      });
    });
    
    // Generic creative suggestions for dating app context
    const romanticSuggestions = [
      "HeartWhisperer", "LoveSeeker", "SoulMate", "DreamLover", "HeartSong",
      "PassionQuest", "TenderSoul", "LoveBird", "HeartThrob", "CupidArrow",
      "RomanticDreamer", "SweetHeart", "LoveStory", "HeartFelt", "CharmingSmile"
    ];
    
    const adventureSuggestions = [
      "StarGazer", "MoonDancer", "SunChaser", "DreamWeaver", "WildSpirit", 
      "FreeFlyer", "JoyRider", "LightBearer", "WaveRider", "CloudWalker",
      "FireSoul", "WindDancer", "RainMaker", "SnowAngel", "ThunderHeart",
      "LightningBolt", "SkyDiver", "OceanExplorer", "MountainClimber", "CityWanderer"
    ];
    
    const modernSuggestions = [
      "DigitalNomad", "CoffeeConnoisseur", "BookwormChic", "FitnessGuru", "ArtisticSoul",
      "TechSavvy", "NatureLover", "FoodieFun", "TravelBug", "MusicMaven",
      "PhotoPro", "YogaZen", "BeachVibes", "CityLife", "CountryCharm"
    ];
    
    // Mix different types of suggestions
    suggestions.push(...romanticSuggestions.slice(0, 4));
    suggestions.push(...adventureSuggestions.slice(0, 4));
    suggestions.push(...modernSuggestions.slice(0, 4));
    
    // Filter out existing usernames and return unique available suggestions
    return [...new Set(suggestions)]
      .filter(suggestion => !existingUsernames.has(suggestion.toLowerCase()))
      .slice(0, 12);
  }
  
  static checkAvailability(username: string): boolean {
    return !existingUsernames.has(username.toLowerCase());
  }
  
  static addToDatabase(username: string): void {
    existingUsernames.add(username.toLowerCase());
  }
}

export function UsernameHelper({ value, onChange, firstName, interests, className, onValidationChange }: UsernameHelperProps) {
  const [isChecking, setIsChecking] = useState(false);
  const [isAvailable, setIsAvailable] = useState<boolean | null>(null);
  const [error, setError] = useState("");
  const [suggestions, setSuggestions] = useState<string[]>([]);
  const [showSuggestions, setShowSuggestions] = useState(false);
  
  const validateUsername = (username: string): string => {
    const cleanUsername = username.trim();
    
    if (cleanUsername.length < 2) {
      return "Username must be at least 2 characters long.";
    }
    if (cleanUsername.length > 20) {
      return "Username must be 20 characters or less.";
    }
    if (!/^[a-zA-Z0-9_\s-]+$/.test(cleanUsername)) {
      return "Username can only contain letters, numbers, spaces, hyphens, and underscores.";
    }
    return "";
  };
  
  const checkUsernameAvailability = async (username: string) => {
    if (!username || validateUsername(username)) {
      setIsAvailable(null);
      return;
    }
    
    setIsChecking(true);
    
    // Simulate API call delay
    await new Promise(resolve => setTimeout(resolve, 500));
    
    const available = UsernameAI.checkAvailability(username);
    setIsAvailable(available);
    setIsChecking(false);
    
    // Notify parent component of validation status
    onValidationChange?.(available);
  };
  
  const generateSuggestions = () => {
    const newSuggestions = UsernameAI.generateSuggestions(firstName, interests);
    setSuggestions(newSuggestions);
    setShowSuggestions(true);
  };
  
  const selectSuggestion = (suggestion: string) => {
    onChange(suggestion);
    setShowSuggestions(false);
  };
  
  useEffect(() => {
    const validationError = validateUsername(value);
    setError(validationError);
    
    if (!validationError && value) {
      checkUsernameAvailability(value);
    } else {
      setIsAvailable(null);
      onValidationChange?.(false);
    }
  }, [value]);
  
  const getInputClassName = () => {
    let baseClass = className || "";
    if (error) return `${baseClass} border-red-500 focus-visible:ring-red-500`;
    if (value && !error && isAvailable === true) return `${baseClass} border-green-500 focus-visible:ring-green-500`;
    if (value && !error && isAvailable === false) return `${baseClass} border-red-500 focus-visible:ring-red-500`;
    return baseClass;
  };
  
  return (
    <div className="space-y-3">
      <div className="space-y-2">
        <Label htmlFor="profileName">Profile Name (Username)</Label>
        <div className="relative">
          <Input
            id="profileName"
            value={value}
            onChange={(e) => onChange(e.target.value)}
            placeholder="Choose your unique username"
            className={getInputClassName()}
          />
          <div className="absolute right-3 top-1/2 transform -translate-y-1/2 flex items-center gap-2">
            {isChecking && (
              <RefreshCw className="w-4 h-4 animate-spin text-muted-foreground" />
            )}
            {value && !error && isAvailable === true && (
              <Check className="w-4 h-4 text-green-500" />
            )}
            {value && !error && isAvailable === false && (
              <AlertTriangle className="w-4 h-4 text-red-500" />
            )}
          </div>
        </div>
        
        {/* Error State */}
        {error && (
          <Alert variant="destructive">
            <AlertTriangle className="h-4 w-4" />
            <AlertDescription>{error}</AlertDescription>
          </Alert>
        )}
        
        {/* Success State */}
        {value && !error && isAvailable === true && (
          <Alert className="border-green-200 bg-green-50 dark:bg-green-950 dark:border-green-800">
            <Check className="h-4 w-4 text-green-600 dark:text-green-400" />
            <AlertDescription className="text-green-700 dark:text-green-300">
              Great! "{value}" is available and ready to use.
            </AlertDescription>
          </Alert>
        )}
        
        {/* Unavailable State */}
        {value && !error && isAvailable === false && (
          <Alert variant="destructive">
            <AlertTriangle className="h-4 w-4" />
            <AlertDescription>
              Sorry, "{value}" is already taken. Try one of our AI suggestions below!
            </AlertDescription>
          </Alert>
        )}
        
        <div className="space-y-2">
          <p className="text-xs text-muted-foreground">
            🔒 Your real name stays private. Only your username will be visible to other members.
          </p>
          <div className="flex items-center justify-between">
            <p className="text-xs text-muted-foreground">
              💡 Need inspiration? Let our AI help you find the perfect username!
            </p>
            <Button
              type="button"
              variant="outline"
              size="sm"
              onClick={generateSuggestions}
              className="text-xs"
            >
              <Sparkles className="w-3 h-3 mr-1" />
              AI Suggestions
            </Button>
          </div>
        </div>
      </div>
      
      {/* AI Suggestions */}
      {showSuggestions && suggestions.length > 0 && (
        <Card className="border-blue-200 bg-blue-50/50 dark:bg-blue-950/20 dark:border-blue-800">
          <CardHeader className="pb-3">
            <CardTitle className="flex items-center gap-2 text-sm">
              <Sparkles className="w-4 h-4 text-blue-600" />
              AI-Powered Username Suggestions
            </CardTitle>
          </CardHeader>
          <CardContent className="pt-0">
            <div className="grid grid-cols-2 sm:grid-cols-3 gap-2 mb-3">
              {suggestions.map((suggestion, index) => (
                <Button
                  key={suggestion}
                  variant="outline"
                  size="sm"
                  onClick={() => selectSuggestion(suggestion)}
                  className="justify-start text-xs h-8 hover:bg-blue-100 dark:hover:bg-blue-900/40"
                >
                  <div className="flex items-center gap-1">
                    {index < 3 && <Star className="w-3 h-3 text-yellow-500" />}
                    {index >= 3 && index < 6 && <Heart className="w-3 h-3 text-red-500" />}
                    {index >= 6 && <Zap className="w-3 h-3 text-blue-500" />}
                    {suggestion}
                  </div>
                </Button>
              ))}
            </div>
            <div className="flex justify-between items-center">
              <p className="text-xs text-muted-foreground">
                ✨ All suggestions are verified as available
              </p>
              <Button
                type="button"
                variant="ghost"
                size="sm"
                onClick={() => setShowSuggestions(false)}
                className="text-xs"
              >
                Hide Suggestions
              </Button>
            </div>
          </CardContent>
        </Card>
      )}
      
      {/* No Suggestions Available */}
      {showSuggestions && suggestions.length === 0 && (
        <Alert>
          <User className="h-4 w-4" />
          <AlertDescription>
            Wow! It looks like you've found a very unique combination. 
            All our AI suggestions are taken, but keep trying different variations!
          </AlertDescription>
        </Alert>
      )}
    </div>
  );
}

// Export the AI class for potential use elsewhere
export { UsernameAI };