import { useState } from "react";
import { Button } from "./ui/button";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "./ui/card";
import { Input } from "./ui/input";
import { Label } from "./ui/label";
import { Textarea } from "./ui/textarea";
import { Alert, AlertDescription } from "./ui/alert";
import { Badge } from "./ui/badge";
import { Separator } from "./ui/separator";
import { 
  User, 
  AlertTriangle, 
  Clock, 
  CheckCircle, 
  XCircle, 
  Edit3,
  Send,
  History
} from "lucide-react";
import { UsernameAI } from "./UsernameHelper";

interface UsernameChangeRequestProps {
  currentUsername: string;
  onRequestSubmitted?: (newUsername: string, reason: string) => void;
}

interface ChangeRequest {
  id: string;
  requestedUsername: string;
  reason: string;
  status: 'pending' | 'approved' | 'rejected';
  requestDate: string;
  reviewDate?: string;
  reviewedBy?: string;
  adminComment?: string;
}

// Mock data for existing requests
const mockRequests: ChangeRequest[] = [
  {
    id: "req-1",
    requestedUsername: "emma_photographer",
    reason: "Want to reflect my profession in my username",
    status: "approved",
    requestDate: "2025-01-10",
    reviewDate: "2025-01-11",
    reviewedBy: "Admin Sarah",
    adminComment: "Approved - username reflects user's profession"
  },
  {
    id: "req-2", 
    requestedUsername: "emmaJ_photos",
    reason: "Previous request variation",
    status: "rejected",
    requestDate: "2025-01-05",
    reviewDate: "2025-01-06",
    reviewedBy: "Admin Mike",
    adminComment: "Too similar to existing username. Please choose something more distinct."
  }
];

export function UsernameChangeRequest({ currentUsername, onRequestSubmitted }: UsernameChangeRequestProps) {
  const [newUsername, setNewUsername] = useState("");
  const [reason, setReason] = useState("");
  const [isSubmitting, setIsSubmitting] = useState(false);
  const [showSuggestions, setShowSuggestions] = useState(false);
  const [requests, setRequests] = useState<ChangeRequest[]>(mockRequests);
  const [showHistory, setShowHistory] = useState(false);
  const [usernameError, setUsernameError] = useState("");
  
  // Check if there's a pending request
  const pendingRequest = requests.find(req => req.status === 'pending');
  const hasRecentRejection = requests.some(req => 
    req.status === 'rejected' && 
    new Date(req.requestDate) > new Date(Date.now() - 7 * 24 * 60 * 60 * 1000) // Within 7 days
  );

  const validateUsername = (username: string) => {
    const trimmed = username.trim();
    
    if (trimmed.length < 2) {
      return "Username must be at least 2 characters long";
    }
    if (trimmed.length > 20) {
      return "Username must be less than 20 characters long";
    }
    if (!/^[a-zA-Z0-9_\s-]+$/.test(trimmed)) {
      return "Username can only contain letters, numbers, spaces, hyphens, and underscores";
    }
    if (trimmed.toLowerCase() === currentUsername.toLowerCase()) {
      return "New username must be different from current username";
    }
    if (!UsernameAI.checkAvailability(trimmed)) {
      return "This username is already taken";
    }
    
    return "";
  };

  const handleUsernameChange = (value: string) => {
    setNewUsername(value);
    const error = validateUsername(value);
    setUsernameError(error);
  };

  const handleSubmitRequest = async () => {
    if (!newUsername.trim() || !reason.trim()) {
      return;
    }

    const validationError = validateUsername(newUsername);
    if (validationError) {
      setUsernameError(validationError);
      return;
    }

    setIsSubmitting(true);
    
    // Simulate API call
    await new Promise(resolve => setTimeout(resolve, 1000));
    
    const newRequest: ChangeRequest = {
      id: `req-${Date.now()}`,
      requestedUsername: newUsername.trim(),
      reason: reason.trim(),
      status: 'pending',
      requestDate: new Date().toISOString().split('T')[0]
    };
    
    setRequests([newRequest, ...requests]);
    setNewUsername("");
    setReason("");
    setIsSubmitting(false);
    
    if (onRequestSubmitted) {
      onRequestSubmitted(newRequest.requestedUsername, newRequest.reason);
    }
  };

  const generateSuggestions = () => {
    if (!currentUsername) return [];
    
    // Use the existing AI system to generate suggestions based on current username
    const suggestions = [];
    const baseName = currentUsername.replace(/[^a-zA-Z]/g, '');
    
    // Generate variations
    suggestions.push(`${baseName}_2025`);
    suggestions.push(`${baseName}_official`);
    suggestions.push(`${baseName}_user`);
    suggestions.push(`new_${baseName}`);
    suggestions.push(`${baseName}123`);
    
    // Filter out unavailable ones
    return suggestions.filter(suggestion => 
      suggestion !== currentUsername && UsernameAI.checkAvailability(suggestion)
    ).slice(0, 4);
  };

  const getStatusColor = (status: string) => {
    switch (status) {
      case 'pending': return 'bg-yellow-100 text-yellow-800 dark:bg-yellow-900/20 dark:text-yellow-400';
      case 'approved': return 'bg-green-100 text-green-800 dark:bg-green-900/20 dark:text-green-400';
      case 'rejected': return 'bg-red-100 text-red-800 dark:bg-red-900/20 dark:text-red-400';
      default: return 'bg-gray-100 text-gray-800 dark:bg-gray-900/20 dark:text-gray-400';
    }
  };

  const getStatusIcon = (status: string) => {
    switch (status) {
      case 'pending': return <Clock className="w-4 h-4" />;
      case 'approved': return <CheckCircle className="w-4 h-4" />;
      case 'rejected': return <XCircle className="w-4 h-4" />;
      default: return null;
    }
  };

  return (
    <div className="space-y-6">
      {/* Current Username Display */}
      <Card>
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <User className="w-5 h-5" />
            Current Username
          </CardTitle>
        </CardHeader>
        <CardContent>
          <div className="flex items-center justify-between">
            <div>
              <p className="text-lg font-medium">@{currentUsername}</p>
              <p className="text-sm text-muted-foreground">This is how other users see you</p>
            </div>
            <Badge variant="outline">Active</Badge>
          </div>
        </CardContent>
      </Card>

      {/* Pending Request Alert */}
      {pendingRequest && (
        <Alert>
          <Clock className="w-4 h-4" />
          <AlertDescription>
            You have a pending username change request for "<strong>{pendingRequest.requestedUsername}</strong>". 
            Please wait for admin approval before submitting a new request.
          </AlertDescription>
        </Alert>
      )}

      {/* Recent Rejection Warning */}
      {hasRecentRejection && !pendingRequest && (
        <Alert variant="destructive">
          <AlertTriangle className="w-4 h-4" />
          <AlertDescription>
            Your recent username change request was rejected. Please review the admin feedback before submitting a new request.
          </AlertDescription>
        </Alert>
      )}

      {/* Username Change Request Form */}
      {!pendingRequest && (
        <Card>
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <Edit3 className="w-5 h-5" />
              Request Username Change
            </CardTitle>
            <CardDescription>
              Submit a request to change your username. All requests must be approved by an administrator.
            </CardDescription>
          </CardHeader>
          <CardContent className="space-y-4">
            <div className="space-y-2">
              <Label htmlFor="newUsername">New Username</Label>
              <Input
                id="newUsername"
                value={newUsername}
                onChange={(e) => handleUsernameChange(e.target.value)}
                placeholder="Enter your desired username"
                className={usernameError ? "border-red-500" : ""}
              />
              {usernameError && (
                <p className="text-sm text-red-600">{usernameError}</p>
              )}
              <p className="text-sm text-muted-foreground">
                2-20 characters. Letters, numbers, spaces, hyphens, and underscores only.
              </p>
            </div>

            {/* Username Suggestions */}
            <div>
              <div className="flex items-center justify-between mb-2">
                <Label className="text-sm">Need inspiration?</Label>
                <Button
                  variant="link"
                  size="sm"
                  onClick={() => setShowSuggestions(!showSuggestions)}
                  className="h-auto p-0 text-sm"
                >
                  {showSuggestions ? 'Hide' : 'Show'} suggestions
                </Button>
              </div>
              {showSuggestions && (
                <div className="flex flex-wrap gap-2">
                  {generateSuggestions().map((suggestion) => (
                    <Button
                      key={suggestion}
                      variant="outline"
                      size="sm"
                      onClick={() => handleUsernameChange(suggestion)}
                      className="text-sm"
                    >
                      {suggestion}
                    </Button>
                  ))}
                </div>
              )}
            </div>

            <div className="space-y-2">
              <Label htmlFor="reason">Reason for Change</Label>
              <Textarea
                id="reason"
                value={reason}
                onChange={(e) => setReason(e.target.value)}
                placeholder="Please explain why you want to change your username..."
                className="min-h-[80px]"
                maxLength={500}
              />
              <p className="text-sm text-muted-foreground">
                {reason.length}/500 characters
              </p>
            </div>

            <Alert>
              <AlertTriangle className="w-4 h-4" />
              <AlertDescription>
                Username changes are subject to approval. Please choose wisely as frequent requests may be denied.
              </AlertDescription>
            </Alert>

            <Button 
              onClick={handleSubmitRequest}
              disabled={!newUsername.trim() || !reason.trim() || !!usernameError || isSubmitting}
              className="w-full"
            >
              {isSubmitting ? (
                <>
                  <Clock className="w-4 h-4 mr-2 animate-spin" />
                  Submitting Request...
                </>
              ) : (
                <>
                  <Send className="w-4 h-4 mr-2" />
                  Submit Request
                </>
              )}
            </Button>
          </CardContent>
        </Card>
      )}

      {/* Request History */}
      <Card>
        <CardHeader>
          <div className="flex items-center justify-between">
            <CardTitle className="flex items-center gap-2">
              <History className="w-5 h-5" />
              Request History
            </CardTitle>
            <Button
              variant="outline"
              size="sm"
              onClick={() => setShowHistory(!showHistory)}
            >
              {showHistory ? 'Hide' : 'Show'} History
            </Button>
          </div>
        </CardHeader>
        {showHistory && (
          <CardContent>
            {requests.length === 0 ? (
              <p className="text-muted-foreground text-center py-4">
                No username change requests yet.
              </p>
            ) : (
              <div className="space-y-4">
                {requests.map((request) => (
                  <div key={request.id} className="border rounded-lg p-4">
                    <div className="flex items-center justify-between mb-2">
                      <div className="flex items-center gap-2">
                        <span className="font-medium">@{request.requestedUsername}</span>
                        <Badge className={getStatusColor(request.status)}>
                          {getStatusIcon(request.status)}
                          <span className="ml-1 capitalize">{request.status}</span>
                        </Badge>
                      </div>
                      <span className="text-sm text-muted-foreground">
                        {new Date(request.requestDate).toLocaleDateString()}
                      </span>
                    </div>
                    
                    <p className="text-sm text-muted-foreground mb-2">
                      <strong>Reason:</strong> {request.reason}
                    </p>
                    
                    {request.reviewDate && (
                      <>
                        <Separator className="my-2" />
                        <div className="space-y-1">
                          <p className="text-sm">
                            <strong>Reviewed by:</strong> {request.reviewedBy} on {new Date(request.reviewDate).toLocaleDateString()}
                          </p>
                          {request.adminComment && (
                            <p className="text-sm">
                              <strong>Admin comment:</strong> {request.adminComment}
                            </p>
                          )}
                        </div>
                      </>
                    )}
                  </div>
                ))}
              </div>
            )}
          </CardContent>
        )}
      </Card>
    </div>
  );
}