import React, { createContext, useContext, useState, useEffect } from 'react';

export type UserRole = 'super-admin' | 'admin' | 'moderator' | 'premium' | 'user';
export type SubscriptionPlan = 'free' | 'premium' | 'platinum';

export interface SubscriptionOverride {
  id: string;
  plan: SubscriptionPlan;
  startDate: string;
  expirationDate?: string;
  isActive: boolean;
  reason: string;
  grantedBy: string;
}

export interface User {
  id: string;
  name: string;
  email: string;
  role: UserRole;
  permissions: string[];
  status: 'active' | 'suspended' | 'pending';
  subscriptionPlan: SubscriptionPlan;
  membershipTier: SubscriptionPlan; // For groups access control
  subscriptionOverride?: SubscriptionOverride;
}

interface UserContextType {
  user: User | null;
  setUser: (user: User | null) => void;
  hasPermission: (permission: string) => boolean;
  isRole: (role: UserRole | UserRole[]) => boolean;
  canModifyUser: (targetRole: UserRole) => boolean;
  getEffectiveSubscriptionPlan: () => SubscriptionPlan;
  grantSubscriptionOverride: (userId: string, plan: SubscriptionPlan, expirationDate?: string, reason?: string) => void;
  revokeSubscriptionOverride: (userId: string) => void;
  checkSubscriptionExpiry: () => void;
}

const UserContext = createContext<UserContextType | undefined>(undefined);

// Role hierarchy (higher number = more power)
const roleHierarchy: Record<UserRole, number> = {
  'super-admin': 100,
  'admin': 50,
  'moderator': 25,
  'premium': 10,
  'user': 1
};

// Default permissions for each role
const defaultPermissions: Record<UserRole, string[]> = {
  'super-admin': ['all'],
  'admin': ['user-management', 'content-moderation', 'reports', 'analytics'],
  'moderator': ['content-moderation', 'reports'],
  'premium': ['premium-features'],
  'user': ['basic-features']
};

// Function to get default subscription plan based on role
const getDefaultSubscriptionPlan = (role: UserRole): SubscriptionPlan => {
  switch (role) {
    case 'super-admin':
    case 'admin':
    case 'moderator':
      return 'platinum'; // All staff get platinum automatically
    case 'premium':
      return 'premium';
    case 'user':
    default:
      return 'free';
  }
};

export function UserProvider({ children }: { children: React.ReactNode }) {
  const [user, setUser] = useState<User | null>(null);

  // Initialize with Super Admin for demo purposes
  useEffect(() => {
    const defaultUser: User = {
      id: 'super-1',
      name: 'Super Administrator',
      email: 'superadmin@singleszone.com',
      role: 'super-admin',
      permissions: ['all'],
      status: 'active',
      subscriptionPlan: 'platinum', // Super admin gets platinum automatically
      membershipTier: 'platinum'
    };
    setUser(defaultUser);
  }, []);

  // Check for expired subscription overrides
  useEffect(() => {
    const checkExpiry = () => {
      if (user?.subscriptionOverride?.isActive && user.subscriptionOverride.expirationDate) {
        const expiryDate = new Date(user.subscriptionOverride.expirationDate);
        const now = new Date();
        
        if (now > expiryDate) {
          // Override has expired, revert to default subscription
          const updatedUser = { 
            ...user, 
            subscriptionPlan: getDefaultSubscriptionPlan(user.role),
            subscriptionOverride: { 
              ...user.subscriptionOverride, 
              isActive: false 
            }
          };
          setUser(updatedUser);
          console.log('Subscription override expired for user:', user.name);
        }
      }
    };

    // Check expiry every minute
    const interval = setInterval(checkExpiry, 60000);
    checkExpiry(); // Check immediately

    return () => clearInterval(interval);
  }, [user]);

  const hasPermission = (permission: string): boolean => {
    if (!user) return false;
    if (user.permissions.includes('all')) return true;
    return user.permissions.includes(permission);
  };

  const isRole = (roles: UserRole | UserRole[]): boolean => {
    if (!user) return false;
    const roleArray = Array.isArray(roles) ? roles : [roles];
    return roleArray.includes(user.role);
  };

  const canModifyUser = (targetRole: UserRole): boolean => {
    if (!user) return false;
    
    // Super admin can modify anyone except other super admins
    if (user.role === 'super-admin') {
      return targetRole !== 'super-admin';
    }
    
    // Regular admins and moderators cannot modify super admins or users of equal/higher rank
    const userLevel = roleHierarchy[user.role];
    const targetLevel = roleHierarchy[targetRole];
    
    return userLevel > targetLevel;
  };

  const getEffectiveSubscriptionPlan = (): SubscriptionPlan => {
    if (!user) return 'free';
    
    // If user has an active override, use that
    if (user.subscriptionOverride?.isActive) {
      return user.subscriptionOverride.plan;
    }
    
    // Otherwise use their regular subscription plan
    return user.subscriptionPlan;
  };

  const grantSubscriptionOverride = (
    userId: string, 
    plan: SubscriptionPlan, 
    expirationDate?: string, 
    reason: string = 'Admin granted'
  ) => {
    if (!user || !hasPermission('user-management')) {
      console.error('Insufficient permissions to grant subscription override');
      return;
    }

    // In a real app, this would make an API call
    // For now, we'll just update the current user if it matches
    if (user.id === userId) {
      const override: SubscriptionOverride = {
        id: Date.now().toString(),
        plan,
        startDate: new Date().toISOString().split('T')[0],
        expirationDate,
        isActive: true,
        reason,
        grantedBy: user.name
      };

      setUser({
        ...user,
        subscriptionOverride: override
      });
    }

    console.log(`Subscription override granted: ${plan} to user ${userId}`);
  };

  const revokeSubscriptionOverride = (userId: string) => {
    if (!user || !hasPermission('user-management')) {
      console.error('Insufficient permissions to revoke subscription override');
      return;
    }

    // In a real app, this would make an API call
    // For now, we'll just update the current user if it matches
    if (user.id === userId && user.subscriptionOverride) {
      setUser({
        ...user,
        subscriptionPlan: getDefaultSubscriptionPlan(user.role),
        subscriptionOverride: {
          ...user.subscriptionOverride,
          isActive: false
        }
      });
    }

    console.log(`Subscription override revoked for user ${userId}`);
  };

  const checkSubscriptionExpiry = () => {
    if (user?.subscriptionOverride?.isActive && user.subscriptionOverride.expirationDate) {
      const expiryDate = new Date(user.subscriptionOverride.expirationDate);
      const now = new Date();
      
      if (now > expiryDate) {
        revokeSubscriptionOverride(user.id);
      }
    }
  };

  const value = {
    user,
    setUser,
    hasPermission,
    isRole,
    canModifyUser,
    getEffectiveSubscriptionPlan,
    grantSubscriptionOverride,
    revokeSubscriptionOverride,
    checkSubscriptionExpiry,
  };

  return (
    <UserContext.Provider value={value}>
      {children}
    </UserContext.Provider>
  );
}

export function useUser() {
  const context = useContext(UserContext);
  if (context === undefined) {
    throw new Error('useUser must be used within a UserProvider');
  }
  return context;
}

// Role switching utility for testing different user roles
export function useRoleSwitcher() {
  const { setUser } = useUser();
  
  const switchToRole = (role: UserRole) => {
    const mockUsers: Record<UserRole, User> = {
      'super-admin': {
        id: 'super-1',
        name: 'Super Administrator',
        email: 'superadmin@singleszone.com',
        role: 'super-admin',
        permissions: ['all'],
        status: 'active',
        subscriptionPlan: 'platinum',
        membershipTier: 'platinum'
      },
      'admin': {
        id: 'admin-1',
        name: 'Administrator',
        email: 'admin@singleszone.com',
        role: 'admin',
        permissions: defaultPermissions.admin,
        status: 'active',
        subscriptionPlan: 'platinum',
        membershipTier: 'platinum'
      },
      'moderator': {
        id: 'mod-1',
        name: 'Moderator',
        email: 'mod@singleszone.com',
        role: 'moderator',
        permissions: defaultPermissions.moderator,
        status: 'active',
        subscriptionPlan: 'platinum',
        membershipTier: 'platinum'
      },
      'premium': {
        id: 'premium-1',
        name: 'Premium User',
        email: 'premium@singleszone.com',
        role: 'premium',
        permissions: defaultPermissions.premium,
        status: 'active',
        subscriptionPlan: 'premium',
        membershipTier: 'premium'
      },
      'user': {
        id: 'user-1',
        name: 'Regular User',
        email: 'user@singleszone.com',
        role: 'user',
        permissions: defaultPermissions.user,
        status: 'active',
        subscriptionPlan: 'free',
        membershipTier: 'free'
      }
    };
    
    setUser(mockUsers[role]);
  };
  
  return { switchToRole };
}