import React, { createContext, useContext, useState, useEffect } from 'react';

type Language = 'en' | 'es' | 'fr' | 'de' | 'it' | 'pt' | 'ru' | 'zh' | 'ja' | 'ko' | 'ar' | 'hi' | 'tr' | 'pl' | 'nl' | 'sv' | 'da' | 'no' | 'fi' | 'he' | 'th' | 'vi' | 'id' | 'ms' | 'tl' | 'cs' | 'hu' | 'ro' | 'bg' | 'hr' | 'sk' | 'sl' | 'et' | 'lv' | 'lt' | 'uk' | 'be' | 'kk' | 'uz' | 'az' | 'hy' | 'ka' | 'mn' | 'bn' | 'ta' | 'te' | 'ml' | 'kn' | 'gu' | 'pa' | 'ur' | 'fa' | 'ps' | 'ne' | 'si' | 'my' | 'km' | 'lo' | 'am' | 'sw' | 'yo' | 'ig' | 'ha' | 'zu' | 'xh' | 'af';

interface TranslationContextType {
  language: Language;
  setLanguage: (lang: Language) => void;
  t: (key: string, params?: Record<string, string | number>) => string;
  isLoading: boolean;
}

const TranslationContext = createContext<TranslationContextType | undefined>(undefined);

// Translation dictionaries
const translations: Record<Language, Record<string, string>> = {
  en: {
    // Common
    'common.yes': 'Yes',
    'common.no': 'No',
    'common.cancel': 'Cancel',
    'common.confirm': 'Confirm',
    'common.save': 'Save',
    'common.delete': 'Delete',
    'common.edit': 'Edit',
    'common.search': 'Search',
    'common.loading': 'Loading...',
    'common.error': 'Error',
    'common.success': 'Success',
    'common.back': 'Back',
    'common.next': 'Next',
    'common.previous': 'Previous',
    'common.submit': 'Submit',
    'common.close': 'Close',
    'common.open': 'Open',
    'common.view': 'View',
    'common.more': 'More',
    'common.less': 'Less',
    'common.show': 'Show',
    'common.hide': 'Hide',
    'common.online': 'Online',
    'common.offline': 'Offline',
    'common.members': 'members',
    'common.member': 'member',
    
    // Navigation
    'nav.search': 'Search',
    'nav.discover': 'Discover',
    'nav.messages': 'Messages',
    'nav.groups': 'Groups',
    'nav.profile': 'Profile',
    'nav.settings': 'Settings',
    'nav.logout': 'Logout',
    'nav.login': 'Login',
    'nav.register': 'Register',
    
    // Auth
    'auth.login.title': 'Welcome Back to SinglesZonne',
    'auth.login.subtitle': 'Find your perfect match today',
    'auth.register.title': 'Join SinglesZonne',
    'auth.register.subtitle': 'Start your journey to find love',
    'auth.email': 'Email',
    'auth.password': 'Password',
    'auth.confirmPassword': 'Confirm Password',
    'auth.firstName': 'First Name',
    'auth.lastName': 'Last Name',
    'auth.dateOfBirth': 'Date of Birth',
    'auth.gender': 'Gender',
    'auth.location': 'Location',
    'auth.interests': 'Interests',
    'auth.photos': 'Photos',
    'auth.language': 'Preferred Language',
    'auth.forgotPassword': 'Forgot your password?',
    'auth.noAccount': "Don't have an account?",
    'auth.hasAccount': 'Already have an account?',
    'auth.signUp': 'Sign up',
    'auth.signIn': 'Sign in',
    'auth.termsAgree': 'I agree to the Terms of Service and Privacy Policy',
    
    // Search
    'search.title': 'Find Your Perfect Match',
    'search.subtitle': 'Discover amazing people near you',
    'search.filters': 'Filters',
    'search.age': 'Age',
    'search.distance': 'Distance',
    'search.interests': 'Interests',
    'search.lookingFor': 'Looking for',
    'search.clearFilters': 'Clear Filters',
    'search.applyFilters': 'Apply Filters',
    'search.noResults': 'No results found',
    'search.loadMore': 'Load More',
    
    // Profile
    'profile.age': 'years old',
    'profile.away': 'away',
    'profile.online': 'Online now',
    'profile.lastSeen': 'Last seen',
    'profile.match': 'Match',
    'profile.like': 'Like',
    'profile.pass': 'Pass',
    'profile.superLike': 'Super Like',
    'profile.message': 'Message',
    'profile.photos': 'Photos',
    'profile.about': 'About',
    'profile.interests': 'Interests',
    'profile.lookingFor': 'Looking for',
    'profile.education': 'Education',
    'profile.work': 'Work',
    'profile.height': 'Height',
    'profile.smoking': 'Smoking',
    'profile.drinking': 'Drinking',
    'profile.children': 'Children',
    'profile.pets': 'Pets',
    
    // Messages
    'messages.title': 'Messages',
    'messages.subtitle': 'Start meaningful conversations with your matches',
    'messages.all': 'All',
    'messages.matches': 'Matches',
    'messages.likes': 'Likes',
    'messages.unread': 'Unread',
    'messages.typeMessage': 'Type your message...',
    'messages.send': 'Send',
    'messages.selectConversation': 'Select a conversation',
    'messages.selectConversationText': 'Choose a conversation from the left to start messaging',
    'messages.liveChat': 'Live Chat',
    'messages.emailSupport': 'Email Support',
    'messages.activeAgo': 'Active {time} ago',
    'messages.justNow': 'Just now',
    
    // Groups
    'groups.title': 'Groups & Communities',
    'groups.subtitle': 'Connect with like-minded people through shared interests and activities',
    'groups.discover': 'Discover Groups',
    'groups.myGroups': 'My Groups',
    'groups.searchPlaceholder': 'Search groups by name, interests, or tags...',
    'groups.interests': 'Interests',
    'groups.location': 'Location',
    'groups.events': 'Events',
    'groups.activities': 'Activities',
    'groups.join': 'Join',
    'groups.leave': 'Leave',
    'groups.chat': 'Chat',
    'groups.openChat': 'Open Chat',
    'groups.private': 'Private',
    'groups.noGroupsYet': 'No groups joined yet',
    'groups.exploreGroups': 'Explore Groups',
    'groups.discoverText': 'Discover and join groups that match your interests!',
    'groups.joined': 'Joined',
    'groups.memberCount': '{count} members',
    
    // Footer
    'footer.about': 'About SinglesZonne',
    'footer.careers': 'Careers',
    'footer.press': 'Press',
    'footer.blog': 'Blog',
    'footer.help': 'Help Center',
    'footer.safety': 'Safety',
    'footer.privacy': 'Privacy Policy',
    'footer.terms': 'Terms of Service',
    'footer.cookies': 'Cookie Policy',
    'footer.community': 'Community Guidelines',
    'footer.contact': 'Contact Us',
    'footer.support': 'Support',
    'footer.feedback': 'Feedback',
    'footer.social': 'Follow Us',
    'footer.app': 'Download App',
    'footer.copyright': '© 2024 SinglesZonne. All rights reserved.',
    'footer.description': 'The world\'s most popular dating app for meaningful connections.',
  },
  
  // Spanish
  es: {
    'nav.search': 'Buscar',
    'nav.discover': 'Descubrir',
    'nav.messages': 'Mensajes',
    'nav.groups': 'Grupos',
    'auth.login.title': 'Bienvenido de vuelta a SinglesZonne',
    'auth.language': 'Idioma preferido',
    'messages.title': 'Mensajes',
    'messages.subtitle': 'Comienza conversaciones significativas con tus matches',
    'footer.copyright': '© 2024 SinglesZonne. Todos los derechos reservados.',
  },
  
  // French
  fr: {
    'nav.search': 'Rechercher',
    'nav.discover': 'Découvrir',
    'nav.messages': 'Messages',
    'nav.groups': 'Groupes',
    'auth.login.title': 'Bienvenue sur SinglesZonne',
    'auth.language': 'Langue préférée',
    'messages.title': 'Messages',
    'messages.subtitle': 'Commencez des conversations significatives avec vos matches',
    'footer.copyright': '© 2024 SinglesZonne. Tous droits réservés.',
  },
  
  // German
  de: {
    'nav.search': 'Suchen',
    'nav.discover': 'Entdecken',
    'nav.messages': 'Nachrichten',
    'nav.groups': 'Gruppen',
    'auth.login.title': 'Willkommen zurück bei SinglesZonne',
    'auth.language': 'Bevorzugte Sprache',
    'messages.title': 'Nachrichten',
    'messages.subtitle': 'Beginnen Sie bedeutungsvolle Gespräche mit Ihren Matches',
    'footer.copyright': '© 2024 SinglesZonne. Alle Rechte vorbehalten.',
  },
  
  // Italian
  it: {
    'nav.search': 'Cerca',
    'nav.discover': 'Scopri',
    'nav.messages': 'Messaggi',
    'nav.groups': 'Gruppi',
    'auth.login.title': 'Bentornato su SinglesZonne',
    'auth.language': 'Lingua preferita',
    'messages.title': 'Messaggi',
    'messages.subtitle': 'Inizia conversazioni significative con i tuoi match',
    'footer.copyright': '© 2024 SinglesZonne. Tutti i diritti riservati.',
  },
  
  // Portuguese
  pt: {
    'nav.search': 'Pesquisar',
    'nav.discover': 'Descobrir',
    'nav.messages': 'Mensagens',
    'nav.groups': 'Grupos',
    'auth.login.title': 'Bem-vindo de volta ao SinglesZonne',
    'auth.language': 'Idioma preferido',
    'messages.title': 'Mensagens',
    'messages.subtitle': 'Inicie conversas significativas com seus matches',
    'footer.copyright': '© 2024 SinglesZonne. Todos os direitos reservados.',
  },
  
  // Russian
  ru: {
    'nav.search': 'Поиск',
    'nav.discover': 'Открыть',
    'nav.messages': 'Сообщения',
    'nav.groups': 'Группы',
    'auth.login.title': 'Добро пожаловать обратно в SinglesZonne',
    'auth.language': 'Предпочитаемый язык',
    'messages.title': 'Сообщения',
    'messages.subtitle': 'Начните содержательные разговоры со своими совпадениями',
    'footer.copyright': '© 2024 SinglesZonne. Все права защищены.',
  },
  
  // Chinese
  zh: {
    'nav.search': '搜索',
    'nav.discover': '发现',
    'nav.messages': '消息',
    'nav.groups': '群组',
    'auth.login.title': '欢迎回到 SinglesZonne',
    'auth.language': '首选语言',
    'messages.title': '消息',
    'messages.subtitle': '与您的匹配开始有意义的对话',
    'footer.copyright': '© 2024 SinglesZonne。保留所有权利。',
  },
  
  // Japanese
  ja: {
    'nav.search': '検索',
    'nav.discover': '発见',
    'nav.messages': 'メッセージ',
    'nav.groups': 'グループ',
    'auth.login.title': 'SinglesZonne へようこそ',
    'auth.language': '希望言語',
    'messages.title': 'メッセージ',
    'messages.subtitle': 'マッチした相手と意味のある会話を始めましょう',
    'footer.copyright': '© 2024 SinglesZonne。全著作権所有。',
  },
  
  // Korean
  ko: {
    'nav.search': '검색',
    'nav.discover': '발견',
    'nav.messages': '메시지',
    'nav.groups': '그룹',
    'auth.login.title': 'SinglesZonne에 다시 오신 것을 환영합니다',
    'auth.language': '선호 언어',
    'messages.title': '메시지',
    'messages.subtitle': '매치된 상대와 의미 있는 대화를 시작하세요',
    'footer.copyright': '© 2024 SinglesZonne. 모든 권리 보유.',
  },
  
  // Arabic
  ar: {
    'nav.search': 'بحث',
    'nav.discover': 'استكشاف',
    'nav.messages': 'الرسائل',
    'nav.groups': 'المجموعات',
    'auth.login.title': 'مرحباً بعودتك إلى SinglesZonne',
    'auth.language': 'اللغة المفضلة',
    'messages.title': 'الرسائل',
    'messages.subtitle': 'ابدأ محادثات مفيدة مع أشخاص تطابقت معهم',
    'footer.copyright': '© 2024 SinglesZonne. جميع الحقوق محفوظة.',
  },
  
  // Hindi
  hi: {
    'nav.search': 'खोजें',
    'nav.discover': 'खोजें',
    'nav.messages': 'संदेश',
    'nav.groups': 'समूह',
    'auth.login.title': 'SinglesZonne में आपका स्वागत है',
    'auth.language': 'पसंदीदा भाषा',
    'messages.title': 'संदेश',
    'messages.subtitle': 'अपने मैच के साथ अर्थपूर्ण बातचीत शुरू करें',
    'footer.copyright': '© 2024 SinglesZonne. सभी अधिकार सुरक्षित।',
  },
  
  // Turkish
  tr: {
    'nav.search': 'Ara',
    'nav.discover': 'Keşfet',
    'nav.messages': 'Mesajlar',
    'nav.groups': 'Gruplar',
    'auth.login.title': 'SinglesZonne\'a tekrar hoş geldiniz',
    'auth.language': 'Tercih edilen dil',
    'messages.title': 'Mesajlar',
    'messages.subtitle': 'Eşleştiklerinizle anlamlı sohbetler başlatın',
    'footer.copyright': '© 2024 SinglesZonne. Tüm hakları saklıdır.',
  },
  
  // Polish
  pl: {
    'nav.search': 'Szukaj',
    'nav.discover': 'Odkryj',
    'nav.messages': 'Wiadomości',
    'nav.groups': 'Grupy',
    'auth.login.title': 'Witamy ponownie w SinglesZonne',
    'auth.language': 'Preferowany język',
    'messages.title': 'Wiadomości',
    'messages.subtitle': 'Rozpocznij znaczące rozmowy z dopasowaniami',
    'footer.copyright': '© 2024 SinglesZonne. Wszelkie prawa zastrzeżone.',
  },
  
  // Dutch
  nl: {
    'nav.search': 'Zoeken',
    'nav.discover': 'Ontdekken',
    'nav.messages': 'Berichten',
    'nav.groups': 'Groepen',
    'auth.login.title': 'Welkom terug bij SinglesZonne',
    'auth.language': 'Voorkeurstaal',
    'messages.title': 'Berichten',
    'messages.subtitle': 'Begin betekenisvolle gesprekken met matches',
    'footer.copyright': '© 2024 SinglesZonne. Alle rechten voorbehouden.',
  },
  
  // Swedish
  sv: {
    'nav.search': 'Sök',
    'nav.discover': 'Upptäck',
    'nav.messages': 'Meddelanden',
    'nav.groups': 'Grupper',
    'auth.login.title': 'Välkommen tillbaka till SinglesZonne',
    'auth.language': 'Föredraget språk',
    'messages.title': 'Meddelanden',
    'messages.subtitle': 'Börja meningsfulla samtal med matchningar',
    'footer.copyright': '© 2024 SinglesZonne. Alla rättigheter förbehållna.',
  },
  
  // Danish
  da: {
    'nav.search': 'Søg',
    'nav.discover': 'Opdag',
    'nav.messages': 'Beskeder',
    'nav.groups': 'Grupper',
    'auth.login.title': 'Velkommen tilbage til SinglesZonne',
    'auth.language': 'Foretrukket sprog',
    'messages.title': 'Beskeder',
    'messages.subtitle': 'Start meningsfulde samtaler med matches',
    'footer.copyright': '© 2024 SinglesZonne. Alle rettigheder forbeholdt.',
  },
  
  // Norwegian
  no: {
    'nav.search': 'Søk',
    'nav.discover': 'Oppdag',
    'nav.messages': 'Meldinger',
    'nav.groups': 'Grupper',
    'auth.login.title': 'Velkommen tilbake til SinglesZonne',
    'auth.language': 'Foretrukket språk',
    'messages.title': 'Meldinger',
    'messages.subtitle': 'Start meningsfulle samtaler med matches',
    'footer.copyright': '© 2024 SinglesZonne. Alle rettigheter forbeholdt.',
  },
  
  // Finnish
  fi: {
    'nav.search': 'Haku',
    'nav.discover': 'Löydä',
    'nav.messages': 'Viestit',
    'nav.groups': 'Ryhmät',
    'auth.login.title': 'Tervetuloa takaisin SinglesZonneen',
    'auth.language': 'Ensisijainen kieli',
    'messages.title': 'Viestit',
    'messages.subtitle': 'Aloita merkityksellisiä keskusteluja matchien kanssa',
    'footer.copyright': '© 2024 SinglesZonne. Kaikki oikeudet pidätetään.',
  },
  
  // Hebrew
  he: {
    'nav.search': 'חיפוש',
    'nav.discover': 'גלה',
    'nav.messages': 'הודעות',
    'nav.groups': 'קבוצות',
    'auth.login.title': 'ברוכים השבים ל-SinglesZonne',
    'auth.language': 'שפה מועדפת',
    'messages.title': 'הודעות',
    'messages.subtitle': 'התחל שיחות משמעותיות עם התאמות',
    'footer.copyright': '© 2024 SinglesZonne. כל הזכויות שמורות.',
  },
  
  // Thai
  th: {
    'nav.search': 'ค้นหา',
    'nav.discover': 'ค้นพบ',
    'nav.messages': 'ข้อความ',
    'nav.groups': 'กลุ่ม',
    'auth.login.title': 'ยินดีต้อนรับกลับสู่ SinglesZonne',
    'auth.language': 'ภาษาที่ต้องการ',
    'messages.title': 'ข้อความ',
    'messages.subtitle': 'เริ่มต้นการสนทนาที่มีความหมายกับคู่ที่จับคู่',
    'footer.copyright': '© 2024 SinglesZonne สงวนลิขสิทธิ์',
  },
  
  // Vietnamese
  vi: {
    'nav.search': 'Tìm kiếm',
    'nav.discover': 'Khám phá',
    'nav.messages': 'Tin nhắn',
    'nav.groups': 'Nhóm',
    'auth.login.title': 'Chào mừng trở lại SinglesZonne',
    'auth.language': 'Ngôn ngữ ưa thích',
    'messages.title': 'Tin nhắn',
    'messages.subtitle': 'Bắt đầu cuộc trò chuyện ý nghĩa với các kết nối',
    'footer.copyright': '© 2024 SinglesZonne. Bảo lưu mọi quyền.',
  },
  
  // Indonesian
  id: {
    'nav.search': 'Cari',
    'nav.discover': 'Jelajahi',
    'nav.messages': 'Pesan',
    'nav.groups': 'Grup',
    'auth.login.title': 'Selamat datang kembali di SinglesZonne',
    'auth.language': 'Bahasa pilihan',
    'messages.title': 'Pesan',
    'messages.subtitle': 'Mulai percakapan bermakna dengan pasangan',
    'footer.copyright': '© 2024 SinglesZonne. Hak cipta dilindungi.',
  },
  
  // Malay
  ms: {
    'nav.search': 'Cari',
    'nav.discover': 'Terokai',
    'nav.messages': 'Mesej',
    'nav.groups': 'Kumpulan',
    'auth.login.title': 'Selamat kembali ke SinglesZonne',
    'auth.language': 'Bahasa pilihan',
    'messages.title': 'Mesej',
    'messages.subtitle': 'Mulakan perbualan bermakna dengan padanan',
    'footer.copyright': '© 2024 SinglesZonne. Hak cipta terpelihara.',
  },
  
  // Filipino/Tagalog
  tl: {
    'nav.search': 'Maghanap',
    'nav.discover': 'Tuklasin',
    'nav.messages': 'Mga Mensahe',
    'nav.groups': 'Mga Grupo',
    'auth.login.title': 'Maligayang pagbabalik sa SinglesZonne',
    'auth.language': 'Gustong wika',
    'messages.title': 'Mga Mensahe',
    'messages.subtitle': 'Magsimula ng makabuluhang pag-uusap sa mga tugma',
    'footer.copyright': '© 2024 SinglesZonne. Lahat ng karapatan ay nakalaan.',
  },
  
  // Czech
  cs: {
    'nav.search': 'Hledat',
    'nav.discover': 'Objevit',
    'nav.messages': 'Zprávy',
    'nav.groups': 'Skupiny',
    'auth.login.title': 'Vítejte zpět v SinglesZonne',
    'auth.language': 'Preferovaný jazyk',
    'messages.title': 'Zprávy',
    'messages.subtitle': 'Začněte smysluplné rozhovory s partnery',
    'footer.copyright': '© 2024 SinglesZonne. Všechna práva vyhrazena.',
  },
  
  // Hungarian
  hu: {
    'nav.search': 'Keresés',
    'nav.discover': 'Felfedezés',
    'nav.messages': 'Üzenetek',
    'nav.groups': 'Csoportok',
    'auth.login.title': 'Üdvözöljük vissza a SinglesZonne-ban',
    'auth.language': 'Előnyben részesített nyelv',
    'messages.title': 'Üzenetek',
    'messages.subtitle': 'Kezdjen értelmező beszélgetéseket a párokkal',
    'footer.copyright': '© 2024 SinglesZonne. Minden jog fenntartva.',
  },

  // Basic fallbacks for other languages (all fall back to English translations)
  ro: { 'auth.language': 'Limba preferată', 'footer.copyright': '© 2024 SinglesZonne. Toate drepturile rezervate.' },
  bg: { 'auth.language': 'Предпочитан език', 'footer.copyright': '© 2024 SinglesZonne. Всички права запазени.' },
  hr: { 'auth.language': 'Preferirani jezik', 'footer.copyright': '© 2024 SinglesZonne. Sva prava pridržana.' },
  sk: { 'auth.language': 'Preferovaný jazyk', 'footer.copyright': '© 2024 SinglesZonne. Všetky práva vyhradené.' },
  sl: { 'auth.language': 'Prednostni jezik', 'footer.copyright': '© 2024 SinglesZonne. Vse pravice pridržane.' },
  et: { 'auth.language': 'Eelistatud keel', 'footer.copyright': '© 2024 SinglesZonne. Kõik õigused kaitstud.' },
  lv: { 'auth.language': 'Vēlamā valoda', 'footer.copyright': '© 2024 SinglesZonne. Visas tiesības aizsargātas.' },
  lt: { 'auth.language': 'Pageidaujama kalba', 'footer.copyright': '© 2024 SinglesZonne. Visos teisės saugomos.' },
  uk: { 'auth.language': 'Бажана мова', 'footer.copyright': '© 2024 SinglesZonne. Усі права захищені.' },
  be: { 'auth.language': 'Жаданая мова', 'footer.copyright': '© 2024 SinglesZonne. Усе правы абаронены.' },
  kk: { 'auth.language': 'Қалаған тіл', 'footer.copyright': '© 2024 SinglesZonne. Барлық құқықтар қорғалған.' },
  uz: { 'auth.language': 'Afzal til', 'footer.copyright': '© 2024 SinglesZonne. Barcha huquqlar himoyalangan.' },
  az: { 'auth.language': 'Üstünlük verilən dil', 'footer.copyright': '© 2024 SinglesZonne. Bütün hüquqlar qorunur.' },
  hy: { 'auth.language': 'Նախընտրելի լեզու', 'footer.copyright': '© 2024 SinglesZonne. Բոլոր իրավունքները պահպանված են:' },
  ka: { 'auth.language': 'სასურველი ენა', 'footer.copyright': '© 2024 SinglesZonne. ყველა უფლება დაცულია.' },
  mn: { 'auth.language': 'Давуу хэл', 'footer.copyright': '© 2024 SinglesZonne. Бүх эрх хуулиар хамгаалагдсан.' },
  bn: { 'auth.language': 'পছন্দের ভাষা', 'footer.copyright': '© 2024 SinglesZonne. সমস্ত অধিকার সংরক্ষিত।' },
  ta: { 'auth.language': 'விருப்பமான மொழி', 'footer.copyright': '© 2024 SinglesZonne. அனைத்து உரிமைகளும் பாதுகாக்கப்பட்டுள்ளன.' },
  te: { 'auth.language': 'ప్రాధాన్యత భాష', 'footer.copyright': '© 2024 SinglesZonne. అన్ని హక్కులు సంరక్షించబడ్డాయి.' },
  ml: { 'auth.language': 'ആദ്യഗണനാ ഭാഷ', 'footer.copyright': '© 2024 SinglesZonne. എല്ലാ അവകാശങ്ങളും സംരക്ഷിതമാണ്.' },
  kn: { 'auth.language': 'ಆದ್ಯತೆಯ ಭಾಷೆ', 'footer.copyright': '© 2024 SinglesZonne. ಎಲ್ಲಾ ಹಕ್ಕುಗಳನ್ನು ಕಾಯ್ದಿರಿಸಲಾಗಿದೆ.' },
  gu: { 'auth.language': 'પ્રાથમિક ભાષા', 'footer.copyright': '© 2024 SinglesZonne. તમામ અધિકારો આરક્ષિત છે.' },
  pa: { 'auth.language': 'ਤਰਜੀਹੀ ਭਾਸ਼ਾ', 'footer.copyright': '© 2024 SinglesZonne. ਸਾਰੇ ਹੱਕ ਰਾਖਵੇਂ ਹਨ।' },
  ur: { 'auth.language': 'ترجیحی زبان', 'footer.copyright': '© 2024 SinglesZonne. تمام حقوق محفوظ ہیں۔' },
  fa: { 'auth.language': 'زبان مورد نظر', 'footer.copyright': '© 2024 SinglesZonne. کلیه حقوق محفوظ است.' },
  ps: { 'auth.language': 'د خوښې ژبه', 'footer.copyright': '© 2024 SinglesZonne. ټول حقونه ساتل شوي.' },
  ne: { 'auth.language': 'मनपर्ने भाषा', 'footer.copyright': '© 2024 SinglesZonne. सबै अधिकार सुरक्षित।' },
  si: { 'auth.language': 'කැමති භාෂාව', 'footer.copyright': '© 2024 SinglesZonne. සියලු හිමිකම් ඇවිරිණි.' },
  my: { 'auth.language': 'နှစ်သက်ရာဘာသာစကား', 'footer.copyright': '© 2024 SinglesZonne။ မူပိုင်ခွင့်များအားလုံး ကြေးစားသည်။' },
  km: { 'auth.language': 'ភាសាចំណូលចិត្ត', 'footer.copyright': '© 2024 SinglesZonne។ រក្សាសិទ្ធិគ្រប់យ៉ាង។' },
  lo: { 'auth.language': 'ພາສາທີ່ມັກ', 'footer.copyright': '© 2024 SinglesZonne. ຂໍສະຫງວນສິດທັງໝົດ.' },
  am: { 'auth.language': 'ተመራጭ ቋንቋ', 'footer.copyright': '© 2024 SinglesZonne። ሁሉም መብቶች የተጠበቁ ናቸው።' },
  sw: { 'auth.language': 'Lugha inayopendelewa', 'footer.copyright': '© 2024 SinglesZonne. Haki zote zimehifadhiwa.' },
  yo: { 'auth.language': 'Ede ti o fẹran', 'footer.copyright': '© 2024 SinglesZonne. Gbogbo ẹtọ ni a fi pamọ.' },
  ig: { 'auth.language': 'Asụsụ masịrị', 'footer.copyright': '© 2024 SinglesZonne. E chekwabara ikike nile.' },
  ha: { 'auth.language': 'Harshen da ake so', 'footer.copyright': '© 2024 SinglesZonne. An kiyaye dukkan haƙƙoƙin.' },
  zu: { 'auth.language': 'Ulimi olukhethiwe', 'footer.copyright': '© 2024 SinglesZonne. Wonke amalungelo agodliwe.' },
  xh: { 'auth.language': 'Ulwimi olukhethiweyo', 'footer.copyright': '© 2024 SinglesZonne. Onke amalungelo agciniwe.' },
  af: { 'auth.language': 'Voorkeur taal', 'footer.copyright': '© 2024 SinglesZonne. Alle regte voorbehou.' }
};

// Language options for the selector
export const languageOptions = [
  // Major World Languages
  { value: 'en', label: 'English', flag: '🇺🇸' },
  { value: 'es', label: 'Español', flag: '🇪🇸' },
  { value: 'fr', label: 'Français', flag: '🇫🇷' },
  { value: 'de', label: 'Deutsch', flag: '🇩🇪' },
  { value: 'it', label: 'Italiano', flag: '🇮🇹' },
  { value: 'pt', label: 'Português', flag: '🇧🇷' },
  { value: 'ru', label: 'Русский', flag: '🇷🇺' },
  { value: 'zh', label: '中文', flag: '🇨🇳' },
  { value: 'ja', label: '日本語', flag: '🇯🇵' },
  { value: 'ko', label: '한국어', flag: '🇰🇷' },
  { value: 'ar', label: 'العربية', flag: '🇸🇦' },
  { value: 'hi', label: 'हिंदी', flag: '🇮🇳' },
  { value: 'tr', label: 'Türkçe', flag: '🇹🇷' },
  { value: 'pl', label: 'Polski', flag: '🇵🇱' },
  { value: 'nl', label: 'Nederlands', flag: '🇳🇱' },

  // Nordic Languages
  { value: 'sv', label: 'Svenska', flag: '🇸🇪' },
  { value: 'da', label: 'Dansk', flag: '🇩🇰' },
  { value: 'no', label: 'Norsk', flag: '🇳🇴' },
  { value: 'fi', label: 'Suomi', flag: '🇫🇮' },

  // Other European Languages
  { value: 'he', label: 'עברית', flag: '🇮🇱' },
  { value: 'cs', label: 'Čeština', flag: '🇨🇿' },
  { value: 'hu', label: 'Magyar', flag: '🇭🇺' },
  { value: 'ro', label: 'Română', flag: '🇷🇴' },
  { value: 'bg', label: 'Български', flag: '🇧🇬' },
  { value: 'hr', label: 'Hrvatski', flag: '🇭🇷' },
  { value: 'sk', label: 'Slovenčina', flag: '🇸🇰' },
  { value: 'sl', label: 'Slovenščina', flag: '🇸🇮' },
  { value: 'et', label: 'Eesti', flag: '🇪🇪' },
  { value: 'lv', label: 'Latviešu', flag: '🇱🇻' },
  { value: 'lt', label: 'Lietuvių', flag: '🇱🇹' },

  // Southeast Asian Languages
  { value: 'th', label: 'ไทย', flag: '🇹🇭' },
  { value: 'vi', label: 'Tiếng Việt', flag: '🇻🇳' },
  { value: 'id', label: 'Bahasa Indonesia', flag: '🇮🇩' },
  { value: 'ms', label: 'Bahasa Melayu', flag: '🇲🇾' },
  { value: 'tl', label: 'Filipino', flag: '🇵🇭' },
  { value: 'my', label: 'မြန်မာ', flag: '🇲🇲' },
  { value: 'km', label: 'ខ្មែរ', flag: '🇰🇭' },
  { value: 'lo', label: 'ລາວ', flag: '🇱🇦' },

  // Eastern European & Central Asian
  { value: 'uk', label: 'Українська', flag: '🇺🇦' },
  { value: 'be', label: 'Беларуская', flag: '🇧🇾' },
  { value: 'kk', label: 'Қазақша', flag: '🇰🇿' },
  { value: 'uz', label: 'Oʻzbekcha', flag: '🇺🇿' },
  { value: 'az', label: 'Azərbaycan', flag: '🇦🇿' },
  { value: 'hy', label: 'Հայերեն', flag: '🇦🇲' },
  { value: 'ka', label: 'ქართული', flag: '🇬🇪' },
  { value: 'mn', label: 'Монгол', flag: '🇲🇳' },

  // South Asian Languages
  { value: 'bn', label: 'বাংলা', flag: '🇧🇩' },
  { value: 'ta', label: 'தமிழ்', flag: '🇱🇰' },
  { value: 'te', label: 'తెలుగు', flag: '🇮🇳' },
  { value: 'ml', label: 'മലയാളം', flag: '🇮🇳' },
  { value: 'kn', label: 'ಕನ್ನಡ', flag: '🇮🇳' },
  { value: 'gu', label: 'ગુજરાતી', flag: '🇮🇳' },
  { value: 'pa', label: 'ਪੰਜਾਬੀ', flag: '🇮🇳' },
  { value: 'ur', label: 'اردو', flag: '🇵🇰' },
  { value: 'ne', label: 'नेपाली', flag: '🇳🇵' },
  { value: 'si', label: 'සිංහල', flag: '🇱🇰' },

  // Middle Eastern Languages
  { value: 'fa', label: 'فارسی', flag: '🇮🇷' },
  { value: 'ps', label: 'پښتو', flag: '🇦🇫' },

  // African Languages
  { value: 'am', label: 'አማርኛ', flag: '🇪🇹' },
  { value: 'sw', label: 'Kiswahili', flag: '🇰🇪' },
  { value: 'yo', label: 'Yorùbá', flag: '🇳🇬' },
  { value: 'ig', label: 'Igbo', flag: '🇳🇬' },
  { value: 'ha', label: 'Hausa', flag: '🇳🇬' },
  { value: 'zu', label: 'isiZulu', flag: '🇿🇦' },
  { value: 'xh', label: 'isiXhosa', flag: '🇿🇦' },
  { value: 'af', label: 'Afrikaans', flag: '🇿🇦' }
];

export function TranslationProvider({ children }: { children: React.ReactNode }) {
  const [language, setLanguage] = useState<Language>('en');
  const [isLoading, setIsLoading] = useState(false);

  // Load saved language preference
  useEffect(() => {
    const savedLanguage = localStorage.getItem('preferredLanguage') as Language;
    if (savedLanguage && savedLanguage in translations) {
      setLanguage(savedLanguage);
    }
  }, []);

  // Save language preference
  useEffect(() => {
    localStorage.setItem('preferredLanguage', language);
  }, [language]);

  const t = (key: string, params?: Record<string, string | number>) => {
    const translation = translations[language]?.[key] || translations['en'][key] || key;
    
    if (params) {
      return Object.entries(params).reduce((acc, [paramKey, paramValue]) => {
        return acc.replace(new RegExp(`{${paramKey}}`, 'g'), String(paramValue));
      }, translation);
    }
    
    return translation;
  };

  const value = {
    language,
    setLanguage,
    t,
    isLoading,
  };

  return (
    <TranslationContext.Provider value={value}>
      {children}
    </TranslationContext.Provider>
  );
}

export function useTranslation() {
  const context = useContext(TranslationContext);
  if (context === undefined) {
    throw new Error('useTranslation must be used within a TranslationProvider');
  }
  return context;
}

export function TranslationLoader() {
  return null;
}