import { useState } from "react";
import { Header } from "./Header";
import { Footer } from "./Footer";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "./ui/card";
import { Button } from "./ui/button";
import { Badge } from "./ui/badge";
import { Alert, AlertDescription } from "./ui/alert";
import { Input } from "./ui/input";
import { Label } from "./ui/label";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "./ui/select";
import { Separator } from "./ui/separator";
import { Dialog, DialogContent, DialogDescription, DialogHeader, DialogTitle, DialogTrigger } from "./ui/dialog";
import {
  Users,
  Settings,
  Shield,
  Crown,
  UserCheck,
  UserX,
  Edit,
  Trash2,
  Plus,
  Search,
  Filter,
  MoreHorizontal,
  AlertTriangle,
  CheckCircle,
  Ban,
  Mail,
  Calendar,
  Activity,
  Database,
  Lock,
  Unlock,
  Key,
  Star,
  Zap
} from "lucide-react";
import { useTranslation } from "./TranslationProvider";
import { SubscriptionManager } from "./SubscriptionManager";
import { 
  DropdownMenu,
  DropdownMenuContent,
  DropdownMenuItem,
  DropdownMenuTrigger,
} from "./ui/dropdown-menu";

interface SuperAdminPanelProps {
  onNavigate: (view: string) => void;
  onOpenCookieConsent?: () => void;
}

// User role types
type UserRole = 'super-admin' | 'admin' | 'moderator' | 'premium' | 'user';

interface AdminUser {
  id: string;
  name: string;
  email: string;
  role: UserRole;
  status: 'active' | 'suspended' | 'pending';
  lastActive: string;
  createdBy: string;
  createdAt: string;
  permissions: string[];
  actionsCount: number;
}

export function SuperAdminPanel({ onNavigate, onOpenCookieConsent }: SuperAdminPanelProps) {
  const { t } = useTranslation();
  const [activeSection, setActiveSection] = useState("admins");
  const [searchQuery, setSearchQuery] = useState("");
  const [roleFilter, setRoleFilter] = useState<UserRole | 'all'>('all');
  const [showAddAdminDialog, setShowAddAdminDialog] = useState(false);
  const [showEditDialog, setShowEditDialog] = useState(false);
  const [selectedAdmin, setSelectedAdmin] = useState<AdminUser | null>(null);
  
  // Mock admin users data
  const [adminUsers, setAdminUsers] = useState<AdminUser[]>([
    {
      id: 'super-1',
      name: 'Super Administrator',
      email: 'superadmin@singleszone.com',
      role: 'super-admin',
      status: 'active',
      lastActive: '2024-01-15 14:30',
      createdBy: 'System',
      createdAt: '2020-01-01',
      permissions: ['all'],
      actionsCount: 0
    },
    {
      id: 'admin-1',
      name: 'John Smith',
      email: 'john.admin@singleszone.com',
      role: 'admin',
      status: 'active',
      lastActive: '2024-01-15 12:15',
      createdBy: 'Super Administrator',
      createdAt: '2023-06-15',
      permissions: ['user-management', 'content-moderation', 'reports'],
      actionsCount: 234
    },
    {
      id: 'admin-2',
      name: 'Sarah Johnson',
      email: 'sarah.admin@singleszone.com',
      role: 'admin',
      status: 'active',
      lastActive: '2024-01-15 09:45',
      createdBy: 'Super Administrator',
      createdAt: '2023-08-22',
      permissions: ['user-management', 'content-moderation'],
      actionsCount: 156
    },
    {
      id: 'mod-1',
      name: 'Mike Wilson',
      email: 'mike.mod@singleszone.com',
      role: 'moderator',
      status: 'active',
      lastActive: '2024-01-15 11:20',
      createdBy: 'John Smith',
      createdAt: '2023-11-10',
      permissions: ['content-moderation'],
      actionsCount: 89
    },
    {
      id: 'admin-3',
      name: 'Emma Davis',
      email: 'emma.admin@singleszone.com',
      role: 'admin',
      status: 'suspended',
      lastActive: '2024-01-10 16:30',
      createdBy: 'Super Administrator',
      createdAt: '2023-05-05',
      permissions: ['user-management'],
      actionsCount: 67
    }
  ]);

  const [newAdmin, setNewAdmin] = useState({
    name: '',
    email: '',
    role: 'moderator' as UserRole,
    permissions: [] as string[]
  });

  // Super Admin stats
  const superAdminStats = {
    totalAdmins: adminUsers.filter(u => u.role !== 'super-admin').length,
    activeAdmins: adminUsers.filter(u => u.role !== 'super-admin' && u.status === 'active').length,
    suspendedAdmins: adminUsers.filter(u => u.status === 'suspended').length,
    totalActions: adminUsers.reduce((sum, admin) => sum + admin.actionsCount, 0)
  };

  const getRoleBadge = (role: UserRole) => {
    switch (role) {
      case 'super-admin':
        return <Badge className="bg-gradient-to-r from-red-500 to-pink-600 text-white"><Crown className="w-3 h-3 mr-1" />Super Admin</Badge>;
      case 'admin':
        return <Badge className="bg-purple-100 text-purple-800"><Shield className="w-3 h-3 mr-1" />Admin</Badge>;
      case 'moderator':
        return <Badge className="bg-blue-100 text-blue-800"><UserCheck className="w-3 h-3 mr-1" />Moderator</Badge>;
      default:
        return <Badge variant="outline">{role}</Badge>;
    }
  };

  const getStatusBadge = (status: string) => {
    switch (status) {
      case 'active':
        return <Badge className="bg-green-100 text-green-800">Active</Badge>;
      case 'suspended':
        return <Badge className="bg-red-100 text-red-800">Suspended</Badge>;
      case 'pending':
        return <Badge className="bg-yellow-100 text-yellow-800">Pending</Badge>;
      default:
        return <Badge>{status}</Badge>;
    }
  };

  const filteredAdmins = adminUsers.filter(admin => {
    const matchesSearch = admin.name.toLowerCase().includes(searchQuery.toLowerCase()) ||
                         admin.email.toLowerCase().includes(searchQuery.toLowerCase());
    const matchesRole = roleFilter === 'all' || admin.role === roleFilter;
    return matchesSearch && matchesRole;
  });

  const handlePromoteUser = (adminId: string, newRole: UserRole) => {
    setAdminUsers(prev => prev.map(admin => 
      admin.id === adminId ? { ...admin, role: newRole } : admin
    ));
  };

  const handleSuspendAdmin = (adminId: string) => {
    setAdminUsers(prev => prev.map(admin => 
      admin.id === adminId ? { ...admin, status: admin.status === 'active' ? 'suspended' : 'active' } : admin
    ));
  };

  const handleDeleteAdmin = (adminId: string) => {
    setAdminUsers(prev => prev.filter(admin => admin.id !== adminId));
  };

  const handleAddAdmin = () => {
    if (newAdmin.name && newAdmin.email) {
      const newAdminUser: AdminUser = {
        id: `admin-${Date.now()}`,
        name: newAdmin.name,
        email: newAdmin.email,
        role: newAdmin.role,
        status: 'active',
        lastActive: new Date().toLocaleString(),
        createdBy: 'Super Administrator',
        createdAt: new Date().toISOString().split('T')[0],
        permissions: newAdmin.permissions,
        actionsCount: 0
      };
      
      setAdminUsers(prev => [...prev, newAdminUser]);
      setNewAdmin({ name: '', email: '', role: 'moderator', permissions: [] });
      setShowAddAdminDialog(false);
    }
  };

  const availablePermissions = [
    'user-management',
    'content-moderation',
    'reports',
    'analytics',
    'settings',
    'billing'
  ];

  // Navigation menu items
  const navigationItems = [
    {
      id: 'admins',
      title: 'Admin Users',
      description: 'Manage administrators',
      icon: Users,
      color: 'text-purple-600'
    },
    {
      id: 'subscriptions',
      title: 'Subscriptions',
      description: 'Manage user subscriptions',
      icon: Crown,
      color: 'text-yellow-600'
    },
    {
      id: 'permissions',
      title: 'Permissions',
      description: 'Configure role permissions',
      icon: Shield,
      color: 'text-blue-600'
    },
    {
      id: 'activity',
      title: 'Activity Log',
      description: 'Monitor admin actions',
      icon: Activity,
      color: 'text-green-600'
    },
    {
      id: 'system',
      title: 'System',
      description: 'Platform-wide controls',
      icon: Settings,
      color: 'text-gray-600'
    }
  ];

  return (
    <div className="min-h-screen bg-background">
      <Header currentView="super-admin" onNavigate={onNavigate} />
      
      <div className="flex">
        {/* Left Sidebar Navigation */}
        <div className="w-80 min-h-screen bg-card border-r border-border">
          <div className="p-6 border-b border-border">
            <div className="flex items-center gap-3 mb-2">
              <div className="p-2 bg-gradient-to-r from-red-500 to-pink-600 rounded-lg">
                <Crown className="w-6 h-6 text-white" />
              </div>
              <div>
                <h1 className="text-xl font-bold">Super Admin</h1>
                <Badge className="bg-gradient-to-r from-red-500 to-pink-600 text-white text-xs">
                  <Zap className="w-3 h-3 mr-1" />
                  Ultimate Access
                </Badge>
              </div>
            </div>
          </div>

          {/* Super Admin Stats */}
          <div className="p-4 border-b border-border">
            <div className="grid grid-cols-2 gap-3">
              <div className="text-center p-3 bg-purple-50 dark:bg-purple-900/20 rounded-lg">
                <Shield className="w-6 h-6 text-purple-600 mx-auto mb-1" />
                <p className="text-lg font-bold text-purple-600">{superAdminStats.totalAdmins}</p>
                <p className="text-xs text-muted-foreground">Admins</p>
              </div>
              <div className="text-center p-3 bg-green-50 dark:bg-green-900/20 rounded-lg">
                <UserCheck className="w-6 h-6 text-green-600 mx-auto mb-1" />
                <p className="text-lg font-bold text-green-600">{superAdminStats.activeAdmins}</p>
                <p className="text-xs text-muted-foreground">Active</p>
              </div>
              <div className="text-center p-3 bg-red-50 dark:bg-red-900/20 rounded-lg">
                <UserX className="w-6 h-6 text-red-600 mx-auto mb-1" />
                <p className="text-lg font-bold text-red-600">{superAdminStats.suspendedAdmins}</p>
                <p className="text-xs text-muted-foreground">Suspended</p>
              </div>
              <div className="text-center p-3 bg-blue-50 dark:bg-blue-900/20 rounded-lg">
                <Activity className="w-6 h-6 text-blue-600 mx-auto mb-1" />
                <p className="text-lg font-bold text-blue-600">{superAdminStats.totalActions}</p>
                <p className="text-xs text-muted-foreground">Actions</p>
              </div>
            </div>
          </div>

          {/* Navigation Menu */}
          <nav className="p-4">
            <div className="space-y-1">
              {navigationItems.map((item) => {
                const Icon = item.icon;
                const isActive = activeSection === item.id;
                
                return (
                  <button
                    key={item.id}
                    onClick={() => setActiveSection(item.id)}
                    className={`w-full flex items-center gap-3 px-4 py-3 rounded-lg text-left transition-colors ${
                      isActive
                        ? 'bg-primary text-primary-foreground'
                        : 'hover:bg-accent text-muted-foreground hover:text-foreground'
                    }`}
                  >
                    <Icon className={`w-5 h-5 ${isActive ? 'text-primary-foreground' : item.color}`} />
                    <div>
                      <p className={`font-medium ${isActive ? 'text-primary-foreground' : 'text-foreground'}`}>
                        {item.title}
                      </p>
                      <p className={`text-xs ${isActive ? 'text-primary-foreground/70' : 'text-muted-foreground'}`}>
                        {item.description}
                      </p>
                    </div>
                  </button>
                );
              })}
            </div>
          </nav>
        </div>

        {/* Main Content Area */}
        <div className="flex-1 p-8">
          {/* Page Header */}
          <div className="mb-8">
            {(() => {
              const currentItem = navigationItems.find(item => item.id === activeSection);
              const Icon = currentItem?.icon || Settings;
              
              return (
                <div>
                  <div className="flex items-center gap-3 mb-2">
                    <Icon className={`w-8 h-8 ${currentItem?.color || 'text-gray-600'}`} />
                    <div>
                      <h2 className="text-3xl font-bold">{currentItem?.title || 'Super Admin Panel'}</h2>
                      <p className="text-muted-foreground">{currentItem?.description || 'Manage the platform'}</p>
                    </div>
                  </div>
                </div>
              );
            })()}
          </div>

          {/* Content Sections */}
          {activeSection === 'admins' && (
            <Card>
              <CardHeader>
                <div className="flex items-center justify-between">
                  <div>
                    <CardTitle className="flex items-center gap-2">
                      <Users className="w-5 h-5" />
                      Administrator Management
                    </CardTitle>
                    <CardDescription>
                      Manage admin users and their roles
                    </CardDescription>
                  </div>
                  <Button onClick={() => setShowAddAdminDialog(true)}>
                    <Plus className="w-4 h-4 mr-2" />
                    Add Admin
                  </Button>
                </div>
              </CardHeader>
              <CardContent>
                {/* Search and Filter */}
                <div className="flex gap-4 mb-6">
                  <div className="flex-1">
                    <Input
                      placeholder="Search admins..."
                      value={searchQuery}
                      onChange={(e) => setSearchQuery(e.target.value)}
                      className="w-full"
                    />
                  </div>
                  <Select value={roleFilter} onValueChange={(value) => setRoleFilter(value as UserRole | 'all')}>
                    <SelectTrigger className="w-48">
                      <SelectValue placeholder="Filter by role" />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="all">All Roles</SelectItem>
                      <SelectItem value="super-admin">Super Admin</SelectItem>
                      <SelectItem value="admin">Admin</SelectItem>
                      <SelectItem value="moderator">Moderator</SelectItem>
                    </SelectContent>
                  </Select>
                </div>

                {/* Admin List */}
                <div className="space-y-4">
                  {filteredAdmins.map((admin) => (
                    <div key={admin.id} className="flex items-center justify-between p-4 border rounded-lg bg-card">
                      <div className="flex items-center gap-4">
                        <div className="w-10 h-10 bg-gradient-to-br from-purple-500 to-pink-600 rounded-full flex items-center justify-center text-white font-bold">
                          {admin.name.charAt(0)}
                        </div>
                        <div>
                          <div className="flex items-center gap-2 mb-1">
                            <p className="font-medium">{admin.name}</p>
                            {getRoleBadge(admin.role)}
                            {admin.role === 'super-admin' && (
                              <Badge className="bg-gradient-to-r from-yellow-400 to-orange-500 text-white">
                                <Lock className="w-3 h-3 mr-1" />
                                Protected
                              </Badge>
                            )}
                          </div>
                          <p className="text-sm text-muted-foreground">{admin.email}</p>
                          <div className="flex items-center gap-4 text-xs text-muted-foreground">
                            <span>Last active: {admin.lastActive}</span>
                            <span>Actions: {admin.actionsCount}</span>
                            <span>Created: {admin.createdAt}</span>
                          </div>
                        </div>
                      </div>
                      
                      <div className="flex items-center gap-3">
                        {getStatusBadge(admin.status)}
                        
                        {admin.role !== 'super-admin' && (
                          <DropdownMenu>
                            <DropdownMenuTrigger asChild>
                              <Button variant="ghost" size="sm">
                                <MoreHorizontal className="w-4 h-4" />
                              </Button>
                            </DropdownMenuTrigger>
                            <DropdownMenuContent align="end">
                              <DropdownMenuItem onClick={() => {
                                setSelectedAdmin(admin);
                                setShowEditDialog(true);
                              }}>
                                <Edit className="w-4 h-4 mr-2" />
                                Edit Permissions
                              </DropdownMenuItem>
                              
                              {admin.role === 'moderator' && (
                                <DropdownMenuItem onClick={() => handlePromoteUser(admin.id, 'admin')}>
                                  <Crown className="w-4 h-4 mr-2" />
                                  Promote to Admin
                                </DropdownMenuItem>
                              )}
                              
                              {admin.role === 'admin' && (
                                <DropdownMenuItem onClick={() => handlePromoteUser(admin.id, 'moderator')}>
                                  <UserCheck className="w-4 h-4 mr-2" />
                                  Demote to Moderator
                                </DropdownMenuItem>
                              )}
                              
                              <DropdownMenuItem onClick={() => handleSuspendAdmin(admin.id)}>
                                {admin.status === 'active' ? (
                                  <>
                                    <Ban className="w-4 h-4 mr-2" />
                                    Suspend Admin
                                  </>
                                ) : (
                                  <>
                                    <CheckCircle className="w-4 h-4 mr-2" />
                                    Reactivate Admin
                                  </>
                                )}
                              </DropdownMenuItem>
                              
                              <DropdownMenuItem 
                                onClick={() => handleDeleteAdmin(admin.id)}
                                className="text-red-600"
                              >
                                <Trash2 className="w-4 h-4 mr-2" />
                                Remove Admin
                              </DropdownMenuItem>
                            </DropdownMenuContent>
                          </DropdownMenu>
                        )}
                      </div>
                    </div>
                  ))}
                </div>
              </CardContent>
            </Card>
          )}

          {/* Subscription Management Section */}
          {activeSection === 'subscriptions' && (
            <SubscriptionManager />
          )}

          {/* Permissions Section */}
          {activeSection === 'permissions' && (
            <Card>
              <CardHeader>
                <CardTitle>Permission Management</CardTitle>
                <CardDescription>Configure role-based permissions</CardDescription>
              </CardHeader>
              <CardContent>
                <p className="text-muted-foreground">Permission management interface would be implemented here.</p>
              </CardContent>
            </Card>
          )}

          {/* Activity Log Section */}
          {activeSection === 'activity' && (
            <Card>
              <CardHeader>
                <CardTitle>Admin Activity Log</CardTitle>
                <CardDescription>Monitor all administrative actions</CardDescription>
              </CardHeader>
              <CardContent>
                <p className="text-muted-foreground">Activity monitoring interface would be implemented here.</p>
              </CardContent>
            </Card>
          )}

          {/* System Section */}
          {activeSection === 'system' && (
            <Card>
              <CardHeader>
                <CardTitle>System Management</CardTitle>
                <CardDescription>Platform-wide system controls</CardDescription>
              </CardHeader>
              <CardContent>
                <div className="space-y-4">
                  <Button className="w-full justify-start" variant="outline">
                    <Database className="w-4 h-4 mr-2" />
                    System Backup & Recovery
                  </Button>
                  <Button className="w-full justify-start" variant="outline">
                    <Settings className="w-4 h-4 mr-2" />
                    Global Platform Settings
                  </Button>
                  <Button className="w-full justify-start" variant="outline">
                    <Key className="w-4 h-4 mr-2" />
                    Security Configuration
                  </Button>
                </div>
              </CardContent>
            </Card>
          )}
        </div>
      </div>

      {/* Add Admin Dialog */}
      <Dialog open={showAddAdminDialog} onOpenChange={setShowAddAdminDialog}>
          <DialogContent>
            <DialogHeader>
              <DialogTitle>Add New Administrator</DialogTitle>
              <DialogDescription>
                Create a new admin account with specific permissions
              </DialogDescription>
            </DialogHeader>
            <div className="space-y-4">
              <div>
                <Label htmlFor="name">Full Name</Label>
                <Input
                  id="name"
                  value={newAdmin.name}
                  onChange={(e) => setNewAdmin(prev => ({ ...prev, name: e.target.value }))}
                  placeholder="Enter full name"
                />
              </div>
              <div>
                <Label htmlFor="email">Email Address</Label>
                <Input
                  id="email"
                  type="email"
                  value={newAdmin.email}
                  onChange={(e) => setNewAdmin(prev => ({ ...prev, email: e.target.value }))}
                  placeholder="Enter email address"
                />
              </div>
              <div>
                <Label htmlFor="role">Role</Label>
                <Select value={newAdmin.role} onValueChange={(value) => setNewAdmin(prev => ({ ...prev, role: value as UserRole }))}>
                  <SelectTrigger>
                    <SelectValue />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="moderator">Moderator</SelectItem>
                    <SelectItem value="admin">Admin</SelectItem>
                  </SelectContent>
                </Select>
              </div>
              <div className="flex gap-3 pt-4">
                <Button onClick={handleAddAdmin} className="flex-1">
                  Create Admin
                </Button>
                <Button variant="outline" onClick={() => setShowAddAdminDialog(false)} className="flex-1">
                  Cancel
                </Button>
              </div>
            </div>
          </DialogContent>
        </Dialog>

      <Footer onNavigate={onNavigate} onOpenCookieConsent={onOpenCookieConsent} />
    </div>
  );
}