import { useState } from "react";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "./ui/card";
import { Button } from "./ui/button";
import { Badge } from "./ui/badge";
import { Input } from "./ui/input";
import { Label } from "./ui/label";
import { Switch } from "./ui/switch";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "./ui/select";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "./ui/tabs";
import { Alert, AlertDescription } from "./ui/alert";
import { Separator } from "./ui/separator";
import { Textarea } from "./ui/textarea";
import { Dialog, DialogContent, DialogDescription, DialogHeader, DialogTitle, DialogTrigger } from "./ui/dialog";
import { motion } from "motion/react";
import {
  Crown,
  Calendar,
  Users,
  Gift,
  AlertTriangle,
  Clock,
  CheckCircle,
  XCircle,
  Star,
  Sparkles,
  UserCheck,
  Shield,
  Zap,
  Settings,
  Eye,
  LogIn,
  Edit,
  Trash2,
  Plus,
  Filter,
  Search,
  Download,
  Upload,
  RefreshCw
} from "lucide-react";
import { toast } from "sonner@2.0.3";
import { useTranslation } from "./TranslationProvider";
import { useUser } from "./UserProvider";
import { LoginSwitcher } from "./LoginSwitcher";

interface SubscriptionOverride {
  id: string;
  userId: string;
  userName: string;
  originalPlan: string;
  overridePlan: string;
  startDate: string;
  expirationDate?: string;
  isActive: boolean;
  createdBy: string;
  reason: string;
}

interface User {
  id: string;
  name: string;
  email: string;
  role: 'user' | 'premium' | 'moderator' | 'admin' | 'super-admin';
  subscriptionPlan: 'free' | 'premium' | 'platinum';
  subscriptionStatus: 'active' | 'cancelled' | 'expired';
  joinDate: string;
  lastActive: string;
  avatar?: string;
}

interface BulkAction {
  type: 'grant' | 'revoke' | 'extend';
  plan: 'premium' | 'platinum';
  duration?: string;
  userIds: string[];
  reason: string;
}

export function SubscriptionManager() {
  const { t } = useTranslation();
  const { user: currentUser, isRole } = useUser();
  const [activeTab, setActiveTab] = useState('individual');
  const [selectedUsers, setSelectedUsers] = useState<string[]>([]);
  const [bulkActionOpen, setBulkActionOpen] = useState(false);
  const [searchTerm, setSearchTerm] = useState('');
  const [filterRole, setFilterRole] = useState<string>('all');
  const [filterPlan, setFilterPlan] = useState<string>('all');
  const [showLoginSwitcher, setShowLoginSwitcher] = useState(false);
  const [selectedUserForLogin, setSelectedUserForLogin] = useState<User | null>(null);

  // Mock data - in real app, this would come from backend
  const [subscriptionOverrides, setSubscriptionOverrides] = useState<SubscriptionOverride[]>([
    {
      id: '1',
      userId: 'user1',
      userName: 'Alice Johnson',
      originalPlan: 'free',
      overridePlan: 'premium',
      startDate: '2024-01-15',
      expirationDate: '2024-03-15',
      isActive: true,
      createdBy: 'super-admin',
      reason: 'Beta tester reward'
    },
    {
      id: '2',
      userId: 'user2',
      userName: 'Bob Smith',
      originalPlan: 'free',
      overridePlan: 'platinum',
      startDate: '2024-01-10',
      isActive: true,
      createdBy: 'super-admin',
      reason: 'Community contributor'
    }
  ]);

  const [users] = useState<User[]>([
    {
      id: 'user1',
      name: 'Alice Johnson',
      email: 'alice@example.com',
      role: 'user',
      subscriptionPlan: 'premium',
      subscriptionStatus: 'active',
      joinDate: '2024-01-15',
      lastActive: '2024-01-20',
      avatar: 'https://images.unsplash.com/photo-1494790108755-2616b612b647?w=100&h=100&fit=crop&crop=face'
    },
    {
      id: 'user2',
      name: 'Bob Smith',
      email: 'bob@example.com',
      role: 'user',
      subscriptionPlan: 'platinum',
      subscriptionStatus: 'active',
      joinDate: '2024-01-10',
      lastActive: '2024-01-20',
      avatar: 'https://images.unsplash.com/photo-1507003211169-0a1dd7228f2d?w=100&h=100&fit=crop&crop=face'
    },
    {
      id: 'user3',
      name: 'Carol Davis',
      email: 'carol@example.com',
      role: 'moderator',
      subscriptionPlan: 'platinum',
      subscriptionStatus: 'active',
      joinDate: '2024-01-05',
      lastActive: '2024-01-20'
    },
    {
      id: 'user4',
      name: 'David Wilson',
      email: 'david@example.com',
      role: 'admin',
      subscriptionPlan: 'platinum',
      subscriptionStatus: 'active',
      joinDate: '2023-12-20',
      lastActive: '2024-01-20'
    }
  ]);

  const [individualOverride, setIndividualOverride] = useState({
    userId: '',
    plan: 'premium' as 'premium' | 'platinum',
    duration: 'permanent',
    expirationDate: '',
    reason: ''
  });

  const [bulkOverride, setBulkOverride] = useState({
    plan: 'premium' as 'premium' | 'platinum',
    duration: 'permanent',
    expirationDate: '',
    reason: '',
    applyToAll: false
  });

  const handleIndividualGrant = () => {
    if (!individualOverride.userId || !individualOverride.reason) {
      toast.error('Please select a user and provide a reason');
      return;
    }

    const user = users.find(u => u.id === individualOverride.userId);
    if (!user) return;

    const newOverride: SubscriptionOverride = {
      id: Date.now().toString(),
      userId: individualOverride.userId,
      userName: user.name,
      originalPlan: user.subscriptionPlan,
      overridePlan: individualOverride.plan,
      startDate: new Date().toISOString().split('T')[0],
      expirationDate: individualOverride.duration === 'permanent' ? undefined : individualOverride.expirationDate,
      isActive: true,
      createdBy: currentUser?.name || 'Super Admin',
      reason: individualOverride.reason
    };

    setSubscriptionOverrides(prev => [...prev, newOverride]);
    
    toast.success(`${individualOverride.plan.charAt(0).toUpperCase() + individualOverride.plan.slice(1)} access granted to ${user.name}`, {
      description: individualOverride.duration === 'permanent' ? 'Permanent access' : `Expires on ${individualOverride.expirationDate}`
    });

    // Reset form
    setIndividualOverride({
      userId: '',
      plan: 'premium',
      duration: 'permanent',
      expirationDate: '',
      reason: ''
    });
  };

  const handleBulkGrant = () => {
    if (!bulkOverride.reason) {
      toast.error('Please provide a reason for bulk subscription override');
      return;
    }

    const targetUsers = bulkOverride.applyToAll ? users : users.filter(u => selectedUsers.includes(u.id));
    
    if (targetUsers.length === 0) {
      toast.error('No users selected');
      return;
    }

    const newOverrides = targetUsers.map(user => ({
      id: `${Date.now()}-${user.id}`,
      userId: user.id,
      userName: user.name,
      originalPlan: user.subscriptionPlan,
      overridePlan: bulkOverride.plan,
      startDate: new Date().toISOString().split('T')[0],
      expirationDate: bulkOverride.duration === 'permanent' ? undefined : bulkOverride.expirationDate,
      isActive: true,
      createdBy: currentUser?.name || 'Super Admin',
      reason: bulkOverride.reason
    }));

    setSubscriptionOverrides(prev => [...prev, ...newOverrides]);
    
    toast.success(`${bulkOverride.plan.charAt(0).toUpperCase() + bulkOverride.plan.slice(1)} access granted to ${targetUsers.length} users`, {
      description: bulkOverride.duration === 'permanent' ? 'Permanent access' : `Expires on ${bulkOverride.expirationDate}`
    });

    setBulkActionOpen(false);
    setSelectedUsers([]);
    setBulkOverride({
      plan: 'premium',
      duration: 'permanent',
      expirationDate: '',
      reason: '',
      applyToAll: false
    });
  };

  const handleRevokeOverride = (overrideId: string) => {
    const override = subscriptionOverrides.find(o => o.id === overrideId);
    if (!override) return;

    setSubscriptionOverrides(prev => 
      prev.map(o => o.id === overrideId ? { ...o, isActive: false } : o)
    );

    toast.success(`Subscription override revoked for ${override.userName}`, {
      description: 'User will revert to their original subscription plan'
    });
  };

  const handleLoginAsUser = (userId: string) => {
    const user = users.find(u => u.id === userId);
    if (!user) return;

    setSelectedUserForLogin(user);
    setShowLoginSwitcher(true);
  };

  const handleConfirmLoginSwitch = () => {
    if (selectedUserForLogin) {
      // In real app, this would handle the actual login switch
      console.log(`Login as user: ${selectedUserForLogin.id}`);
      // You could update the user context here or redirect
    }
  };

  const handleEditUser = (userId: string) => {
    const user = users.find(u => u.id === userId);
    if (!user) return;

    toast.info(`Opening edit modal for ${user.name}`);
    // In real app, this would open an edit modal
  };

  const filteredUsers = users.filter(user => {
    const matchesSearch = user.name.toLowerCase().includes(searchTerm.toLowerCase()) ||
                         user.email.toLowerCase().includes(searchTerm.toLowerCase());
    const matchesRole = filterRole === 'all' || user.role === filterRole;
    const matchesPlan = filterPlan === 'all' || user.subscriptionPlan === filterPlan;
    
    return matchesSearch && matchesRole && matchesPlan;
  });

  const activeOverrides = subscriptionOverrides.filter(o => o.isActive);
  const expiredOverrides = subscriptionOverrides.filter(o => !o.isActive);

  return (
    <div className="space-y-6">
      {/* Header */}
      <div className="flex items-center justify-between">
        <div>
          <h2 className="text-2xl font-bold flex items-center gap-2">
            <Crown className="w-6 h-6 text-yellow-600" />
            Subscription Manager
          </h2>
          <p className="text-muted-foreground">
            Manage member subscriptions and grant premium access
          </p>
        </div>
        <div className="flex items-center gap-2">
          <Badge variant="outline" className="flex items-center gap-1">
            <Users className="w-3 h-3" />
            {activeOverrides.length} Active Overrides
          </Badge>
          <Button onClick={() => setBulkActionOpen(true)} className="bg-gradient-to-r from-purple-600 to-pink-600">
            <Plus className="w-4 h-4 mr-2" />
            Bulk Action
          </Button>
        </div>
      </div>

      {/* Stats Cards */}
      <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
        <Card>
          <CardContent className="p-4">
            <div className="flex items-center gap-3">
              <div className="p-2 bg-green-100 rounded-lg">
                <Crown className="w-5 h-5 text-green-600" />
              </div>
              <div>
                <p className="text-sm text-muted-foreground">Premium Users</p>
                <p className="text-xl font-bold">{users.filter(u => u.subscriptionPlan === 'premium').length}</p>
              </div>
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardContent className="p-4">
            <div className="flex items-center gap-3">
              <div className="p-2 bg-purple-100 rounded-lg">
                <Sparkles className="w-5 h-5 text-purple-600" />
              </div>
              <div>
                <p className="text-sm text-muted-foreground">Platinum Users</p>
                <p className="text-xl font-bold">{users.filter(u => u.subscriptionPlan === 'platinum').length}</p>
              </div>
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardContent className="p-4">
            <div className="flex items-center gap-3">
              <div className="p-2 bg-blue-100 rounded-lg">
                <Gift className="w-5 h-5 text-blue-600" />
              </div>
              <div>
                <p className="text-sm text-muted-foreground">Active Overrides</p>
                <p className="text-xl font-bold">{activeOverrides.length}</p>
              </div>
            </div>
          </CardContent>
        </Card>

        <Card>
          <CardContent className="p-4">
            <div className="flex items-center gap-3">
              <div className="p-2 bg-orange-100 rounded-lg">
                <Clock className="w-5 h-5 text-orange-600" />
              </div>
              <div>
                <p className="text-sm text-muted-foreground">Expiring Soon</p>
                <p className="text-xl font-bold">
                  {activeOverrides.filter(o => 
                    o.expirationDate && new Date(o.expirationDate) <= new Date(Date.now() + 7 * 24 * 60 * 60 * 1000)
                  ).length}
                </p>
              </div>
            </div>
          </CardContent>
        </Card>
      </div>

      <Tabs value={activeTab} onValueChange={setActiveTab}>
        <TabsList className="grid w-full grid-cols-4">
          <TabsTrigger value="individual">Individual</TabsTrigger>
          <TabsTrigger value="users">User Management</TabsTrigger>
          <TabsTrigger value="overrides">Active Overrides</TabsTrigger>
          <TabsTrigger value="history">History</TabsTrigger>
        </TabsList>

        {/* Individual Subscription Management */}
        <TabsContent value="individual" className="space-y-6">
          <Card>
            <CardHeader>
              <CardTitle>Grant Individual Access</CardTitle>
              <CardDescription>
                Grant premium or platinum access to a specific user
              </CardDescription>
            </CardHeader>
            <CardContent className="space-y-4">
              <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                <div>
                  <Label htmlFor="user-select">Select User</Label>
                  <Select value={individualOverride.userId} onValueChange={(value) => 
                    setIndividualOverride(prev => ({ ...prev, userId: value }))
                  }>
                    <SelectTrigger>
                      <SelectValue placeholder="Choose a user" />
                    </SelectTrigger>
                    <SelectContent>
                      {users.filter(u => u.role === 'user' || u.role === 'premium').map(user => (
                        <SelectItem key={user.id} value={user.id}>
                          <div className="flex items-center gap-2">
                            <span>{user.name}</span>
                            <Badge variant="outline" className="text-xs">
                              {user.subscriptionPlan}
                            </Badge>
                          </div>
                        </SelectItem>
                      ))}
                    </SelectContent>
                  </Select>
                </div>

                <div>
                  <Label htmlFor="plan-select">Subscription Plan</Label>
                  <Select value={individualOverride.plan} onValueChange={(value: 'premium' | 'platinum') => 
                    setIndividualOverride(prev => ({ ...prev, plan: value }))
                  }>
                    <SelectTrigger>
                      <SelectValue />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="premium">
                        <div className="flex items-center gap-2">
                          <Crown className="w-4 h-4 text-blue-600" />
                          Premium
                        </div>
                      </SelectItem>
                      <SelectItem value="platinum">
                        <div className="flex items-center gap-2">
                          <Sparkles className="w-4 h-4 text-purple-600" />
                          Platinum
                        </div>
                      </SelectItem>
                    </SelectContent>
                  </Select>
                </div>

                <div>
                  <Label htmlFor="duration-select">Duration</Label>
                  <Select value={individualOverride.duration} onValueChange={(value) => 
                    setIndividualOverride(prev => ({ ...prev, duration: value }))
                  }>
                    <SelectTrigger>
                      <SelectValue />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="permanent">Permanent</SelectItem>
                      <SelectItem value="custom">Custom Date</SelectItem>
                    </SelectContent>
                  </Select>
                </div>

                {individualOverride.duration === 'custom' && (
                  <div>
                    <Label htmlFor="expiration-date">Expiration Date</Label>
                    <Input
                      id="expiration-date"
                      type="date"
                      value={individualOverride.expirationDate}
                      onChange={(e) => setIndividualOverride(prev => ({ ...prev, expirationDate: e.target.value }))}
                      min={new Date().toISOString().split('T')[0]}
                    />
                  </div>
                )}
              </div>

              <div>
                <Label htmlFor="reason">Reason</Label>
                <Textarea
                  id="reason"
                  placeholder="Provide a reason for granting this access..."
                  value={individualOverride.reason}
                  onChange={(e) => setIndividualOverride(prev => ({ ...prev, reason: e.target.value }))}
                  rows={3}
                />
              </div>

              <Button onClick={handleIndividualGrant} className="w-full">
                <Gift className="w-4 h-4 mr-2" />
                Grant Access
              </Button>
            </CardContent>
          </Card>
        </TabsContent>

        {/* User Management */}
        <TabsContent value="users" className="space-y-6">
          {/* Search and Filters */}
          <Card>
            <CardContent className="p-4">
              <div className="flex gap-4 items-center">
                <div className="flex-1">
                  <div className="relative">
                    <Search className="w-4 h-4 absolute left-3 top-1/2 transform -translate-y-1/2 text-muted-foreground" />
                    <Input
                      placeholder="Search users..."
                      value={searchTerm}
                      onChange={(e) => setSearchTerm(e.target.value)}
                      className="pl-10"
                    />
                  </div>
                </div>
                <Select value={filterRole} onValueChange={setFilterRole}>
                  <SelectTrigger className="w-48">
                    <SelectValue placeholder="Filter by role" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="all">All Roles</SelectItem>
                    <SelectItem value="user">User</SelectItem>
                    <SelectItem value="premium">Premium</SelectItem>
                    <SelectItem value="moderator">Moderator</SelectItem>
                    <SelectItem value="admin">Admin</SelectItem>
                  </SelectContent>
                </Select>
                <Select value={filterPlan} onValueChange={setFilterPlan}>
                  <SelectTrigger className="w-48">
                    <SelectValue placeholder="Filter by plan" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="all">All Plans</SelectItem>
                    <SelectItem value="free">Free</SelectItem>
                    <SelectItem value="premium">Premium</SelectItem>
                    <SelectItem value="platinum">Platinum</SelectItem>
                  </SelectContent>
                </Select>
              </div>
            </CardContent>
          </Card>

          {/* Users List */}
          <Card>
            <CardHeader>
              <div className="flex items-center justify-between">
                <div>
                  <CardTitle>User Management</CardTitle>
                  <CardDescription>
                    Manage user accounts and subscriptions
                  </CardDescription>
                </div>
                <div className="flex items-center gap-2">
                  <Button
                    variant="outline"
                    size="sm"
                    onClick={() => setSelectedUsers(selectedUsers.length === filteredUsers.length ? [] : filteredUsers.map(u => u.id))}
                  >
                    {selectedUsers.length === filteredUsers.length ? 'Deselect All' : 'Select All'}
                  </Button>
                  {selectedUsers.length > 0 && (
                    <Badge variant="secondary">
                      {selectedUsers.length} selected
                    </Badge>
                  )}
                </div>
              </div>
            </CardHeader>
            <CardContent>
              <div className="space-y-2">
                {filteredUsers.map(user => {
                  const hasOverride = activeOverrides.find(o => o.userId === user.id);
                  const canLoginAs = user.role !== 'admin' && user.role !== 'super-admin' || isRole('super-admin');
                  const canEdit = user.role !== 'super-admin' && (user.role !== 'admin' || isRole('super-admin'));
                  
                  return (
                    <div key={user.id} className="flex items-center gap-4 p-4 border rounded-lg hover:bg-muted/50">
                      <input
                        type="checkbox"
                        checked={selectedUsers.includes(user.id)}
                        onChange={(e) => {
                          if (e.target.checked) {
                            setSelectedUsers(prev => [...prev, user.id]);
                          } else {
                            setSelectedUsers(prev => prev.filter(id => id !== user.id));
                          }
                        }}
                        className="rounded"
                      />
                      
                      <div className="flex items-center gap-3 flex-1">
                        {user.avatar ? (
                          <img src={user.avatar} alt={user.name} className="w-10 h-10 rounded-full" />
                        ) : (
                          <div className="w-10 h-10 rounded-full bg-muted flex items-center justify-center">
                            {user.name.charAt(0)}
                          </div>
                        )}
                        
                        <div>
                          <div className="flex items-center gap-2">
                            <p className="font-medium">{user.name}</p>
                            <Badge variant="outline" className="text-xs">
                              {user.role}
                            </Badge>
                            <Badge className={
                              user.subscriptionPlan === 'platinum' ? 'bg-purple-100 text-purple-800' :
                              user.subscriptionPlan === 'premium' ? 'bg-blue-100 text-blue-800' :
                              'bg-gray-100 text-gray-800'
                            }>
                              {user.subscriptionPlan}
                            </Badge>
                            {hasOverride && (
                              <Badge className="bg-yellow-100 text-yellow-800">
                                <Gift className="w-3 h-3 mr-1" />
                                Override
                              </Badge>
                            )}
                          </div>
                          <p className="text-sm text-muted-foreground">{user.email}</p>
                        </div>
                      </div>

                      <div className="flex items-center gap-2">
                        {canLoginAs && (
                          <Button
                            variant="outline"
                            size="sm"
                            onClick={() => handleLoginAsUser(user.id)}
                          >
                            <LogIn className="w-4 h-4 mr-1" />
                            Login As
                          </Button>
                        )}
                        {canEdit && (
                          <Button
                            variant="outline"
                            size="sm"
                            onClick={() => handleEditUser(user.id)}
                          >
                            <Edit className="w-4 h-4 mr-1" />
                            Edit
                          </Button>
                        )}
                        <Button
                          variant="outline"
                          size="sm"
                        >
                          <Eye className="w-4 h-4 mr-1" />
                          View
                        </Button>
                      </div>
                    </div>
                  );
                })}
              </div>
            </CardContent>
          </Card>
        </TabsContent>

        {/* Active Overrides */}
        <TabsContent value="overrides" className="space-y-6">
          <Card>
            <CardHeader>
              <CardTitle>Active Subscription Overrides</CardTitle>
              <CardDescription>
                Users currently receiving free premium access
              </CardDescription>
            </CardHeader>
            <CardContent>
              {activeOverrides.length === 0 ? (
                <div className="text-center py-8 text-muted-foreground">
                  <Gift className="w-12 h-12 mx-auto mb-4 opacity-50" />
                  <p>No active subscription overrides</p>
                </div>
              ) : (
                <div className="space-y-4">
                  {activeOverrides.map(override => (
                    <div key={override.id} className="flex items-center justify-between p-4 border rounded-lg">
                      <div className="flex items-center gap-4">
                        <div className={`p-2 rounded-lg ${
                          override.overridePlan === 'platinum' ? 'bg-purple-100' : 'bg-blue-100'
                        }`}>
                          {override.overridePlan === 'platinum' ? (
                            <Sparkles className="w-5 h-5 text-purple-600" />
                          ) : (
                            <Crown className="w-5 h-5 text-blue-600" />
                          )}
                        </div>
                        <div>
                          <div className="flex items-center gap-2">
                            <p className="font-medium">{override.userName}</p>
                            <Badge className={
                              override.overridePlan === 'platinum' ? 'bg-purple-100 text-purple-800' : 'bg-blue-100 text-blue-800'
                            }>
                              {override.overridePlan}
                            </Badge>
                            {override.expirationDate && (
                              <Badge variant="outline" className="text-xs">
                                <Clock className="w-3 h-3 mr-1" />
                                Expires {override.expirationDate}
                              </Badge>
                            )}
                          </div>
                          <p className="text-sm text-muted-foreground">{override.reason}</p>
                          <p className="text-xs text-muted-foreground">
                            Granted by {override.createdBy} on {override.startDate}
                          </p>
                        </div>
                      </div>
                      <Button
                        variant="outline"
                        size="sm"
                        onClick={() => handleRevokeOverride(override.id)}
                        className="text-red-600 hover:text-red-700"
                      >
                        <XCircle className="w-4 h-4 mr-1" />
                        Revoke
                      </Button>
                    </div>
                  ))}
                </div>
              )}
            </CardContent>
          </Card>
        </TabsContent>

        {/* History */}
        <TabsContent value="history" className="space-y-6">
          <Card>
            <CardHeader>
              <CardTitle>Subscription History</CardTitle>
              <CardDescription>
                All subscription overrides and changes
              </CardDescription>
            </CardHeader>
            <CardContent>
              <div className="space-y-4">
                {[...activeOverrides, ...expiredOverrides].map(override => (
                  <div key={override.id} className="flex items-center justify-between p-4 border rounded-lg">
                    <div className="flex items-center gap-4">
                      <div className={`p-2 rounded-lg ${
                        override.isActive 
                          ? (override.overridePlan === 'platinum' ? 'bg-purple-100' : 'bg-blue-100')
                          : 'bg-gray-100'
                      }`}>
                        {override.isActive ? (
                          override.overridePlan === 'platinum' ? (
                            <Sparkles className="w-5 h-5 text-purple-600" />
                          ) : (
                            <Crown className="w-5 h-5 text-blue-600" />
                          )
                        ) : (
                          <XCircle className="w-5 h-5 text-gray-500" />
                        )}
                      </div>
                      <div>
                        <div className="flex items-center gap-2">
                          <p className="font-medium">{override.userName}</p>
                          <Badge className={
                            override.overridePlan === 'platinum' ? 'bg-purple-100 text-purple-800' : 'bg-blue-100 text-blue-800'
                          }>
                            {override.overridePlan}
                          </Badge>
                          <Badge variant={override.isActive ? 'default' : 'secondary'}>
                            {override.isActive ? 'Active' : 'Revoked'}
                          </Badge>
                        </div>
                        <p className="text-sm text-muted-foreground">{override.reason}</p>
                        <p className="text-xs text-muted-foreground">
                          {override.startDate} - {override.expirationDate || 'Permanent'}
                        </p>
                      </div>
                    </div>
                  </div>
                ))}
              </div>
            </CardContent>
          </Card>
        </TabsContent>
      </Tabs>

      {/* Login Switcher Modal */}
      {selectedUserForLogin && (
        <LoginSwitcher
          isOpen={showLoginSwitcher}
          onClose={() => {
            setShowLoginSwitcher(false);
            setSelectedUserForLogin(null);
          }}
          targetUser={{
            id: selectedUserForLogin.id,
            name: selectedUserForLogin.name,
            email: selectedUserForLogin.email,
            role: selectedUserForLogin.role,
            avatar: selectedUserForLogin.avatar,
            subscriptionPlan: selectedUserForLogin.subscriptionPlan
          }}
          onConfirm={handleConfirmLoginSwitch}
        />
      )}

      {/* Bulk Action Modal */}
      <Dialog open={bulkActionOpen} onOpenChange={setBulkActionOpen}>
        <DialogContent className="max-w-md">
          <DialogHeader>
            <DialogTitle>Bulk Subscription Action</DialogTitle>
            <DialogDescription>
              Grant subscription access to multiple users at once
            </DialogDescription>
          </DialogHeader>
          <div className="space-y-4">
            <div>
              <Label>Target Users</Label>
              <div className="space-y-2 mt-2">
                <div className="flex items-center gap-2">
                  <input
                    type="radio"
                    id="selected-users"
                    name="target"
                    checked={!bulkOverride.applyToAll}
                    onChange={() => setBulkOverride(prev => ({ ...prev, applyToAll: false }))}
                  />
                  <Label htmlFor="selected-users">
                    Selected users ({selectedUsers.length})
                  </Label>
                </div>
                <div className="flex items-center gap-2">
                  <input
                    type="radio"
                    id="all-users"
                    name="target"
                    checked={bulkOverride.applyToAll}
                    onChange={() => setBulkOverride(prev => ({ ...prev, applyToAll: true }))}
                  />
                  <Label htmlFor="all-users">
                    All users ({users.filter(u => u.role === 'user' || u.role === 'premium').length})
                  </Label>
                </div>
              </div>
            </div>

            <div>
              <Label>Subscription Plan</Label>
              <Select value={bulkOverride.plan} onValueChange={(value: 'premium' | 'platinum') => 
                setBulkOverride(prev => ({ ...prev, plan: value }))
              }>
                <SelectTrigger>
                  <SelectValue />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="premium">Premium</SelectItem>
                  <SelectItem value="platinum">Platinum</SelectItem>
                </SelectContent>
              </Select>
            </div>

            <div>
              <Label>Duration</Label>
              <Select value={bulkOverride.duration} onValueChange={(value) => 
                setBulkOverride(prev => ({ ...prev, duration: value }))
              }>
                <SelectTrigger>
                  <SelectValue />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="permanent">Permanent</SelectItem>
                  <SelectItem value="custom">Custom Date</SelectItem>
                </SelectContent>
              </Select>
            </div>

            {bulkOverride.duration === 'custom' && (
              <div>
                <Label>Expiration Date</Label>
                <Input
                  type="date"
                  value={bulkOverride.expirationDate}
                  onChange={(e) => setBulkOverride(prev => ({ ...prev, expirationDate: e.target.value }))}
                  min={new Date().toISOString().split('T')[0]}
                />
              </div>
            )}

            <div>
              <Label>Reason</Label>
              <Textarea
                placeholder="Provide a reason for bulk subscription override..."
                value={bulkOverride.reason}
                onChange={(e) => setBulkOverride(prev => ({ ...prev, reason: e.target.value }))}
                rows={3}
              />
            </div>

            <div className="flex gap-2 pt-4">
              <Button variant="outline" onClick={() => setBulkActionOpen(false)} className="flex-1">
                Cancel
              </Button>
              <Button onClick={handleBulkGrant} className="flex-1">
                Apply Changes
              </Button>
            </div>
          </div>
        </DialogContent>
      </Dialog>
    </div>
  );
}