import { useState } from "react";
import { Button } from "./ui/button";
import { Input } from "./ui/input";
import { Label } from "./ui/label";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "./ui/select";
import { Sparkles, Search, MapPin } from "lucide-react";
import { MapLocationPicker, useLocationManager } from "./MapLocationPicker";
import { getDistanceOptions, useUnitsPreference, UnitsSelector } from "./UnitsConverter";

interface SearchFiltersProps {
  onSearch: (filters: SearchFilters) => void;
}

interface SearchFilters {
  lookingFor: string;
  ageMin: number;
  ageMax: number;
  country: string;
  distance: number;
  location?: {
    address: string;
    city: string;
    country: string;
    coordinates: { lat: number; lng: number };
  };
}

export function SearchFilters({ onSearch }: SearchFiltersProps) {
  const { preferredUnits, setUnits } = useUnitsPreference();
  const distanceOptions = getDistanceOptions();
  
  const [filters, setFilters] = useState<SearchFilters>({
    lookingFor: "women",
    ageMin: 18,
    ageMax: 40,
    country: "any",
    distance: 50,
  });

  const {
    location,
    showLocationPicker,
    openLocationPicker,
    closeLocationPicker,
    handleLocationSelect
  } = useLocationManager();

  const countries = [
    "any",
    "united-states",
    "canada",
    "united-kingdom",
    "australia",
    "germany",
    "france",
    "spain",
    "italy",
    "japan",
    "brazil",
    "mexico",
    "india",
    "south-korea",
    "netherlands",
    "sweden",
    "norway",
    "denmark",
  ];

  const handleSearch = () => {
    onSearch({
      ...filters,
      location
    });
  };

  const handleLocationSelectComplete = (selectedLocation: {
    address: string;
    city: string;
    country: string;
    coordinates: { lat: number; lng: number };
  }) => {
    handleLocationSelect(selectedLocation);
    // Auto-update country filter if location is selected
    setFilters(prev => ({
      ...prev,
      country: selectedLocation.country.toLowerCase().replace(/\s+/g, '-')
    }));
  };

  const handleSurpriseMe = () => {
    // Randomize filters for surprise search
    const randomCountry = countries[Math.floor(Math.random() * countries.length)];
    const randomAge = Math.floor(Math.random() * 30) + 18;
    const surpriseFilters = {
      ...filters,
      country: randomCountry,
      ageMin: randomAge,
      ageMax: randomAge + 10,
    };
    setFilters(surpriseFilters);
    onSearch(surpriseFilters);
  };

  return (
    <>
      <div className="bg-card py-8 border-b border-border">
        <div className="max-w-7xl mx-auto px-6">
          <div className="flex justify-between items-center mb-4">
            <h3 className="text-lg font-semibold">Search Filters</h3>
            <UnitsSelector value={preferredUnits} onChange={setUnits} />
          </div>
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-7 gap-4 items-end">
          {/* Looking For */}
          <div className="space-y-2">
            <Label htmlFor="lookingFor">I'm looking for</Label>
            <Select
              value={filters.lookingFor}
              onValueChange={(value) => setFilters({ ...filters, lookingFor: value })}
            >
              <SelectTrigger>
                <SelectValue />
              </SelectTrigger>
              <SelectContent>
                <SelectItem value="women">Women</SelectItem>
                <SelectItem value="men">Men</SelectItem>
                <SelectItem value="everyone">Everyone</SelectItem>
              </SelectContent>
            </Select>
          </div>

          {/* Age Range */}
          <div className="space-y-2">
            <Label>Age</Label>
            <div className="flex items-center gap-2">
              <Input
                type="number"
                value={filters.ageMin}
                onChange={(e) => setFilters({ ...filters, ageMin: parseInt(e.target.value) || 18 })}
                min="18"
                max="99"
                className="w-16"
              />
              <span className="text-muted-foreground">to</span>
              <Input
                type="number"
                value={filters.ageMax}
                onChange={(e) => setFilters({ ...filters, ageMax: parseInt(e.target.value) || 40 })}
                min="18"
                max="99"
                className="w-16"
              />
            </div>
          </div>

          {/* Country */}
          <div className="space-y-2">
            <Label htmlFor="country">Country</Label>
            <Select
              value={filters.country}
              onValueChange={(value) => setFilters({ ...filters, country: value })}
            >
              <SelectTrigger>
                <SelectValue placeholder="Any" />
              </SelectTrigger>
              <SelectContent>
                <SelectItem value="any">Any</SelectItem>
                <SelectItem value="united-states">United States</SelectItem>
                <SelectItem value="canada">Canada</SelectItem>
                <SelectItem value="united-kingdom">United Kingdom</SelectItem>
                <SelectItem value="australia">Australia</SelectItem>
                <SelectItem value="germany">Germany</SelectItem>
                <SelectItem value="france">France</SelectItem>
                <SelectItem value="spain">Spain</SelectItem>
                <SelectItem value="italy">Italy</SelectItem>
                <SelectItem value="japan">Japan</SelectItem>
                <SelectItem value="brazil">Brazil</SelectItem>
                <SelectItem value="mexico">Mexico</SelectItem>
                <SelectItem value="india">India</SelectItem>
                <SelectItem value="south-korea">South Korea</SelectItem>
                <SelectItem value="netherlands">Netherlands</SelectItem>
                <SelectItem value="sweden">Sweden</SelectItem>
                <SelectItem value="norway">Norway</SelectItem>
                <SelectItem value="denmark">Denmark</SelectItem>
              </SelectContent>
            </Select>
          </div>

          {/* Location Selector */}
          <div className="space-y-2">
            <Label>Location</Label>
            <Button
              variant="outline"
              onClick={openLocationPicker}
              className="w-full justify-start"
            >
              <MapPin className="w-4 h-4 mr-2" />
              {location ? location.city : 'Select Location'}
            </Button>
          </div>

          {/* Distance */}
          <div className="space-y-2">
            <Label>Distance</Label>
            <Select
              value={filters.distance.toString()}
              onValueChange={(value) => setFilters({ ...filters, distance: parseInt(value) })}
            >
              <SelectTrigger>
                <SelectValue placeholder="Select distance" />
              </SelectTrigger>
              <SelectContent>
                {distanceOptions.map((distance) => (
                  <SelectItem key={distance.value} value={distance.value.toString()}>
                    {distance.label}
                  </SelectItem>
                ))}
              </SelectContent>
            </Select>
          </div>

          {/* Search Button */}
          <Button onClick={handleSearch} className="bg-purple-600 hover:bg-purple-700">
            <Search className="w-4 h-4 mr-2" />
            Smart Search
          </Button>

          {/* Surprise Me Button */}
          <Button
            onClick={handleSurpriseMe}
            variant="outline"
            className="border-purple-200 text-purple-600 hover:bg-purple-50 dark:border-purple-800 dark:text-purple-400 dark:hover:bg-purple-900/20"
          >
            <Sparkles className="w-4 h-4 mr-2" />
            Surprise Me
          </Button>
        </div>
      </div>
      </div>

      {/* Map Location Picker Modal */}
      <MapLocationPicker
        isOpen={showLocationPicker}
        onClose={closeLocationPicker}
        onLocationSelect={handleLocationSelectComplete}
        title="Select Search Location"
        description="Choose a location to search around"
      />
    </>
  );
}