import { ArrowLeft, Shield, AlertTriangle, Eye, MessageSquare, Heart, Users, Lock, Phone } from "lucide-react";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "./ui/card";
import { Button } from "./ui/button";
import { Alert, AlertDescription } from "./ui/alert";
import { Badge } from "./ui/badge";

interface SafetyTipsPageProps {
  onBack: () => void;
}

export function SafetyTipsPage({ onBack }: SafetyTipsPageProps) {
  const safetyCategories = [
    {
      id: "profile-safety",
      title: "Profile & Personal Information",
      icon: Lock,
      color: "blue",
      tips: [
        {
          title: "Protect Your Personal Information",
          description: "Never share your full name, address, phone number, or financial information in your profile or early conversations.",
          importance: "high"
        },
        {
          title: "Use Recent, Authentic Photos",
          description: "Use current photos that accurately represent you. Avoid using photos that could reveal your location or personal details.",
          importance: "medium"
        },
        {
          title: "Be Mindful of What You Share",
          description: "Avoid sharing your workplace, school, or favorite hangout spots until you've built trust with someone.",
          importance: "high"
        }
      ]
    },
    {
      id: "messaging-safety",
      title: "Messaging & Communication",
      icon: MessageSquare,
      color: "green",
      tips: [
        {
          title: "Keep Conversations on the Platform",
          description: "Stay on SinglesZonne's messaging system until you're comfortable moving to other platforms.",
          importance: "high"
        },
        {
          title: "Trust Your Instincts",
          description: "If someone makes you uncomfortable or asks inappropriate questions, trust your gut feeling.",
          importance: "high"
        },
        {
          title: "Watch for Red Flags",
          description: "Be cautious of users who profess love quickly, ask for money, or avoid phone/video calls.",
          importance: "high"
        }
      ]
    },
    {
      id: "meeting-safety",
      title: "Meeting in Person",
      icon: Users,
      color: "purple",
      tips: [
        {
          title: "Meet in Public Places",
          description: "Always meet in busy, public locations like cafes, restaurants, or parks for your first few dates.",
          importance: "high"
        },
        {
          title: "Tell Someone Your Plans",
          description: "Let a friend or family member know where you're going, who you're meeting, and when you expect to return.",
          importance: "high"
        },
        {
          title: "Arrange Your Own Transportation",
          description: "Drive yourself, take public transport, or use a rideshare service. Don't rely on your date for transportation.",
          importance: "medium"
        },
        {
          title: "Video Chat First",
          description: "Have a video call before meeting to verify the person matches their photos and build comfort.",
          importance: "medium"
        }
      ]
    },
    {
      id: "online-safety",
      title: "Online Safety & Privacy",
      icon: Shield,
      color: "orange",
      tips: [
        {
          title: "Use Strong Passwords",
          description: "Create a unique, strong password for your SinglesZonne account and enable two-factor authentication.",
          importance: "medium"
        },
        {
          title: "Be Cautious with Links",
          description: "Don't click on suspicious links or download files from people you've just met online.",
          importance: "high"
        },
        {
          title: "Verify Profile Authenticity",
          description: "Look for verified profiles and be cautious of profiles with limited photos or information.",
          importance: "medium"
        }
      ]
    }
  ];

  const warningSignsData = [
    "Asks for money, gifts, or financial information",
    "Professes love or strong feelings very quickly",
    "Refuses to talk on the phone or video chat",
    "Has very few photos or photos that look professional/fake",
    "Asks for explicit photos or content",
    "Becomes angry or aggressive when you set boundaries",
    "Claims to be traveling or living abroad",
    "Stories don't add up or keep changing",
    "Pressures you to move off the dating platform quickly",
    "Grammar and language seem inconsistent with their profile"
  ];

  const getImportanceColor = (importance: string) => {
    switch (importance) {
      case "high": return "destructive";
      case "medium": return "default";
      default: return "secondary";
    }
  };

  const getImportanceText = (importance: string) => {
    switch (importance) {
      case "high": return "Critical";
      case "medium": return "Important";
      default: return "Helpful";
    }
  };

  return (
    <div className="min-h-screen bg-background">
      {/* Header */}
      <div className="bg-gradient-to-r from-red-600 to-orange-600 text-white">
        <div className="max-w-6xl mx-auto px-6 py-8">
          <div className="flex items-center gap-4 mb-6">
            <Button
              variant="ghost"
              size="sm"
              onClick={onBack}
              className="text-white hover:bg-white/20"
            >
              <ArrowLeft className="w-4 h-4 mr-2" />
              Back
            </Button>
          </div>
          
          <div className="text-center max-w-2xl mx-auto">
            <Shield className="w-16 h-16 mx-auto mb-4 text-white" />
            <h1 className="text-4xl font-bold mb-4">Safety Tips</h1>
            <p className="text-red-100 text-lg">
              Your safety is our top priority. Follow these guidelines to have a safe and enjoyable experience on SinglesZonne.
            </p>
          </div>
        </div>
      </div>

      {/* Emergency Alert */}
      <div className="max-w-6xl mx-auto px-6 py-6">
        <Alert className="border-red-200 bg-red-50 dark:bg-red-900/20">
          <AlertTriangle className="w-5 h-5 text-red-600" />
          <AlertDescription className="text-red-800 dark:text-red-400">
            <strong>Emergency:</strong> If you ever feel unsafe or threatened, contact local emergency services immediately. 
            Report any concerning behavior to our support team at support@singleszonne.com or +1 (555) 123-4567.
          </AlertDescription>
        </Alert>
      </div>

      {/* Content */}
      <div className="max-w-6xl mx-auto px-6 pb-12">
        {/* Safety Categories */}
        <div className="grid gap-8">
          {safetyCategories.map((category) => {
            const IconComponent = category.icon;
            return (
              <Card key={category.id}>
                <CardHeader>
                  <div className="flex items-center gap-3">
                    <div className={`p-3 rounded-lg bg-${category.color}-100 dark:bg-${category.color}-900/20`}>
                      <IconComponent className={`w-6 h-6 text-${category.color}-600`} />
                    </div>
                    <div>
                      <CardTitle className="text-2xl">{category.title}</CardTitle>
                      <CardDescription>Essential safety practices for this area</CardDescription>
                    </div>
                  </div>
                </CardHeader>
                <CardContent>
                  <div className="grid gap-4">
                    {category.tips.map((tip, index) => (
                      <div key={index} className="border rounded-lg p-4 hover:bg-muted/50 transition-colors">
                        <div className="flex items-start justify-between gap-3 mb-2">
                          <h4 className="font-semibold">{tip.title}</h4>
                          <Badge variant={getImportanceColor(tip.importance)} className="shrink-0">
                            {getImportanceText(tip.importance)}
                          </Badge>
                        </div>
                        <p className="text-muted-foreground leading-relaxed">{tip.description}</p>
                      </div>
                    ))}
                  </div>
                </CardContent>
              </Card>
            );
          })}
        </div>

        {/* Warning Signs */}
        <div className="mt-12">
          <Card className="border-orange-200 bg-orange-50 dark:bg-orange-900/20">
            <CardHeader>
              <div className="flex items-center gap-3">
                <Eye className="w-6 h-6 text-orange-600" />
                <div>
                  <CardTitle className="text-orange-800 dark:text-orange-400">Warning Signs to Watch For</CardTitle>
                  <CardDescription className="text-orange-700 dark:text-orange-500">
                    Be cautious if someone exhibits these behaviors
                  </CardDescription>
                </div>
              </div>
            </CardHeader>
            <CardContent>
              <div className="grid md:grid-cols-2 gap-3">
                {warningSignsData.map((sign, index) => (
                  <div key={index} className="flex items-center gap-3 p-3 bg-white dark:bg-orange-900/30 rounded-lg">
                    <AlertTriangle className="w-5 h-5 text-orange-600 shrink-0" />
                    <span className="text-sm text-orange-800 dark:text-orange-300">{sign}</span>
                  </div>
                ))}
              </div>
            </CardContent>
          </Card>
        </div>

        {/* Quick Actions */}
        <div className="mt-12 grid md:grid-cols-3 gap-6">
          <Card className="text-center">
            <CardHeader>
              <Shield className="w-8 h-8 text-blue-600 mx-auto mb-2" />
              <CardTitle>Report a User</CardTitle>
              <CardDescription>Report suspicious or inappropriate behavior</CardDescription>
            </CardHeader>
            <CardContent>
              <Button variant="outline" className="w-full">
                Report Someone
              </Button>
            </CardContent>
          </Card>

          <Card className="text-center">
            <CardHeader>
              <Phone className="w-8 h-8 text-green-600 mx-auto mb-2" />
              <CardTitle>Emergency Resources</CardTitle>
              <CardDescription>Access emergency contacts and resources</CardDescription>
            </CardHeader>
            <CardContent>
              <Button variant="outline" className="w-full">
                View Resources
              </Button>
            </CardContent>
          </Card>

          <Card className="text-center">
            <CardHeader>
              <Heart className="w-8 h-8 text-purple-600 mx-auto mb-2" />
              <CardTitle>Safety Guide</CardTitle>
              <CardDescription>Download our complete safety guide</CardDescription>
            </CardHeader>
            <CardContent>
              <Button variant="outline" className="w-full">
                Download Guide
              </Button>
            </CardContent>
          </Card>
        </div>

        {/* Additional Resources */}
        <div className="mt-12">
          <Card className="bg-gradient-to-r from-purple-50 to-blue-50 dark:from-purple-900/20 dark:to-blue-900/20 border-purple-200 dark:border-purple-800">
            <CardContent className="pt-6 text-center">
              <h3 className="text-2xl font-bold mb-3">Need Help or Have Questions?</h3>
              <p className="text-muted-foreground mb-6">
                Our safety team is available 24/7 to help with any concerns or questions about online dating safety.
              </p>
              <div className="flex flex-col sm:flex-row gap-4 justify-center">
                <Button size="lg" className="bg-purple-600 hover:bg-purple-700">
                  Contact Safety Team
                </Button>
                <Button size="lg" variant="outline">
                  Visit Help Center
                </Button>
              </div>
            </CardContent>
          </Card>
        </div>
      </div>
    </div>
  );
}