import { useState, useEffect } from 'react';
import { Card } from './ui/card';
import { Button } from './ui/button';
import { Input } from './ui/input';
import { Label } from './ui/label';
import { Textarea } from './ui/textarea';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from './ui/select';
import { Badge } from './ui/badge';
import { Separator } from './ui/separator';
import { LocationSelector } from './LocationSelector';
import { AIWritingAssistant } from './AIWritingAssistant';
import { 
  getHeightOptions, 
  getWeightOptions, 
  getDistanceOptions, 
  formatDistance,
  UnitsSelector,
  useUnitsPreference
} from './UnitsConverter';
import { 
  User, 
  MapPin, 
  Briefcase, 
  GraduationCap, 
  Ruler, 
  Heart,
  Plus,
  X,
  Sparkles,
  Globe,
  Target,
  Info,
  Weight
} from 'lucide-react';
import { Alert, AlertDescription } from './ui/alert';

interface ProfileData {
  firstName: string;
  lastName: string;
  age: number;
  bio: string;
  occupation: string;
  education: string;
  height: number; // Changed to number (cm)
  weight: number; // Added weight in kg
  location: string;
  country: string;
  interests: string[];
  relationshipGoal: string;
  lookingForText: string;
  ageRangeMin: number;
  ageRangeMax: number;
  maxDistance: number;
  dealBreakers: string[];
  importantQualities: string[];
  countriesPreference: 'all' | 'specific' | 'multiple';
  selectedCountries: string[];
}

export function ProfileInfoTab() {
  const { preferredUnits, setUnits } = useUnitsPreference();
  const heightOptions = getHeightOptions();
  const weightOptions = getWeightOptions();
  const distanceOptions = getDistanceOptions();
  
  const [profileData, setProfileData] = useState<ProfileData>({
    firstName: 'Sarah',
    lastName: 'Johnson',
    age: 28,
    bio: 'Passionate about life and always looking for new adventures. I believe in authentic connections and meaningful conversations. Love exploring new places, trying different cuisines, and enjoying the simple moments that make life beautiful.',
    occupation: 'Marketing Manager',
    education: "Bachelor's Degree",
    height: 168, // 168cm (5'6")
    weight: 65, // 65kg (143 lbs)
    location: 'Manhattan, New York',
    country: 'United States',
    interests: ['Travel', 'Photography', 'Yoga', 'Cooking', 'Reading', 'Music'],
    relationshipGoal: 'Serious relationship',
    lookingForText: 'Looking for someone genuine who shares my values and wants to build something meaningful together. Someone who appreciates both adventure and quiet moments, has a good sense of humor, and believes in growing together as a team.',
    ageRangeMin: 25,
    ageRangeMax: 35,
    maxDistance: 30,
    dealBreakers: ['Smoking', 'Dishonesty', 'No ambition'],
    importantQualities: ['Honesty', 'Humor', 'Kindness', 'Ambition'],
    countriesPreference: 'specific',
    selectedCountries: ['United States']
  });

  const [newInterest, setNewInterest] = useState('');
  const [newDealBreaker, setNewDealBreaker] = useState('');
  const [newQuality, setNewQuality] = useState('');
  const [aiAssistantOpen, setAiAssistantOpen] = useState(false);
  const [aiLookingForOpen, setAiLookingForOpen] = useState(false);
  const [unsavedChanges, setUnsavedChanges] = useState(false);

  // Character limits
  const BIO_MAX_CHARS = 1000;
  const LOOKING_FOR_MAX_CHARS = 800;

  const educationLevels = [
    'High School',
    "Bachelor's Degree",
    "Master's Degree", 
    'PhD',
    'Trade School',
    'Some College',
    'Prefer not to say'
  ];

  const relationshipGoals = [
    'Casual dating',
    'Serious relationship',
    'Marriage',
    'Companionship',
    'Not sure yet'
  ];



  const countries = [
    'United States', 'Canada', 'United Kingdom', 'Australia', 'Germany',
    'France', 'Spain', 'Italy', 'Netherlands', 'Sweden', 'Norway', 'Denmark'
  ];

  const commonInterests = [
    'Travel', 'Photography', 'Music', 'Dancing', 'Cooking', 'Reading',
    'Movies', 'Sports', 'Fitness', 'Yoga', 'Art', 'Gaming', 'Technology',
    'Nature', 'Animals', 'Wine', 'Coffee', 'Fashion', 'Volunteering'
  ];

  const commonQualities = [
    'Honesty', 'Humor', 'Kindness', 'Intelligence', 'Ambition', 'Creativity',
    'Loyalty', 'Empathy', 'Confidence', 'Authenticity', 'Respect', 'Passion'
  ];

  const commonDealBreakers = [
    'Smoking', 'Dishonesty', 'No ambition', 'Disrespectful', 'Bad hygiene',
    'Excessive drinking', 'No sense of humor', 'Closed-mindedness'
  ];

  const updateProfileData = (field: keyof ProfileData, value: any) => {
    setProfileData(prev => ({ ...prev, [field]: value }));
    setUnsavedChanges(true);
  };

  const addInterest = (interest: string) => {
    if (interest && !profileData.interests.includes(interest)) {
      updateProfileData('interests', [...profileData.interests, interest]);
      setNewInterest('');
    }
  };

  const removeInterest = (interest: string) => {
    updateProfileData('interests', profileData.interests.filter(i => i !== interest));
  };

  const addDealBreaker = (dealBreaker: string) => {
    if (dealBreaker && !profileData.dealBreakers.includes(dealBreaker)) {
      updateProfileData('dealBreakers', [...profileData.dealBreakers, dealBreaker]);
      setNewDealBreaker('');
    }
  };

  const removeDealBreaker = (dealBreaker: string) => {
    updateProfileData('dealBreakers', profileData.dealBreakers.filter(db => db !== dealBreaker));
  };

  const addQuality = (quality: string) => {
    if (quality && !profileData.importantQualities.includes(quality)) {
      updateProfileData('importantQualities', [...profileData.importantQualities, quality]);
      setNewQuality('');
    }
  };

  const removeQuality = (quality: string) => {
    updateProfileData('importantQualities', profileData.importantQualities.filter(q => q !== quality));
  };

  const addCountry = (country: string) => {
    if (country && !profileData.selectedCountries.includes(country)) {
      updateProfileData('selectedCountries', [...profileData.selectedCountries, country]);
    }
  };

  const removeCountry = (country: string) => {
    updateProfileData('selectedCountries', profileData.selectedCountries.filter(c => c !== country));
  };

  const handleSave = () => {
    // In a real app, this would save to the backend
    console.log('Saving profile data:', profileData);
    setUnsavedChanges(false);
    // Show success toast
  };

  const handleAIBioUpdate = (newBio: string) => {
    updateProfileData('bio', newBio);
    setAiAssistantOpen(false);
  };

  const handleAILookingForUpdate = (newText: string) => {
    updateProfileData('lookingForText', newText);
    setAiLookingForOpen(false);
  };

  // Validate character limits
  const handleBioChange = (value: string) => {
    if (value.length <= BIO_MAX_CHARS) {
      updateProfileData('bio', value);
    }
  };

  const handleLookingForChange = (value: string) => {
    if (value.length <= LOOKING_FOR_MAX_CHARS) {
      updateProfileData('lookingForText', value);
    }
  };

  return (
    <div className="space-y-8">
      {/* Header */}
      <div className="flex items-center justify-between">
        <div className="flex items-center gap-3">
          <div className="p-2 rounded-full bg-gradient-to-r from-blue-500 to-purple-600">
            <User className="w-6 h-6 text-white" />
          </div>
          <div>
            <h2 className="text-2xl font-bold">Profile Information</h2>
            <p className="text-muted-foreground">Update your basic profile details</p>
          </div>
        </div>
        
        <div className="flex items-center gap-4">
          <UnitsSelector value={preferredUnits} onChange={setUnits} />
          {unsavedChanges && (
            <Button onClick={handleSave} className="flex items-center gap-2">
              <Sparkles className="w-4 h-4" />
              Save Changes
            </Button>
          )}
        </div>
      </div>

      {/* Unsaved Changes Alert */}
      {unsavedChanges && (
        <Alert>
          <Info className="h-4 w-4" />
          <AlertDescription>
            You have unsaved changes. Don't forget to save your updates!
          </AlertDescription>
        </Alert>
      )}

      {/* Basic Information */}
      <Card className="p-6">
        <h3 className="text-lg font-semibold mb-4 flex items-center gap-2">
          <User className="w-5 h-5" />
          Basic Information
        </h3>
        
        <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
          <div className="space-y-2">
            <Label htmlFor="firstName">First Name</Label>
            <Input
              id="firstName"
              value={profileData.firstName}
              onChange={(e) => updateProfileData('firstName', e.target.value)}
              placeholder="Enter your first name"
            />
          </div>
          
          <div className="space-y-2">
            <Label htmlFor="lastName">Last Name</Label>
            <Input
              id="lastName"
              value={profileData.lastName}
              onChange={(e) => updateProfileData('lastName', e.target.value)}
              placeholder="Enter your last name"
            />
          </div>
          
          <div className="space-y-2">
            <Label htmlFor="age">Age</Label>
            <Input
              id="age"
              type="number"
              min="18"
              max="100"
              value={profileData.age}
              onChange={(e) => updateProfileData('age', parseInt(e.target.value))}
            />
          </div>
          
          <div className="space-y-2">
            <Label>Height</Label>
            <Select value={profileData.height.toString()} onValueChange={(value) => updateProfileData('height', parseInt(value))}>
              <SelectTrigger>
                <SelectValue placeholder="Select your height" />
              </SelectTrigger>
              <SelectContent>
                {heightOptions.map((height) => (
                  <SelectItem key={height.value} value={height.value.toString()}>{height.label}</SelectItem>
                ))}
              </SelectContent>
            </Select>
          </div>

          <div className="space-y-2">
            <Label>Weight</Label>
            <Select value={profileData.weight.toString()} onValueChange={(value) => updateProfileData('weight', parseInt(value))}>
              <SelectTrigger>
                <SelectValue placeholder="Select your weight" />
              </SelectTrigger>
              <SelectContent>
                {weightOptions.map((weight) => (
                  <SelectItem key={weight.value} value={weight.value.toString()}>{weight.label}</SelectItem>
                ))}
              </SelectContent>
            </Select>
          </div>
        </div>
      </Card>

      {/* Bio Section */}
      <Card className="p-6">
        <div className="flex items-center justify-between mb-4">
          <h3 className="text-lg font-semibold flex items-center gap-2">
            <Heart className="w-5 h-5" />
            About Me
          </h3>
          <Button
            variant="outline"
            onClick={() => setAiAssistantOpen(true)}
            className="flex items-center gap-2"
          >
            <Sparkles className="w-4 h-4" />
            AI Writing Assistant
          </Button>
        </div>
        
        <div className="space-y-2">
          <Label htmlFor="bio">Bio</Label>
          <Textarea
            id="bio"
            value={profileData.bio}
            onChange={(e) => handleBioChange(e.target.value)}
            placeholder="Tell people about yourself..."
            rows={4}
            className="resize-none"
            maxLength={BIO_MAX_CHARS}
          />
          <div className="flex justify-between items-center">
            <p className={`text-sm ${profileData.bio.length > BIO_MAX_CHARS * 0.9 ? 'text-orange-500' : 'text-muted-foreground'}`}>
              {profileData.bio.length}/{BIO_MAX_CHARS} characters
            </p>
            {profileData.bio.length >= BIO_MAX_CHARS && (
              <span className="text-sm text-red-500">Character limit reached</span>
            )}
          </div>
        </div>
      </Card>

      {/* Professional Information */}
      <Card className="p-6">
        <h3 className="text-lg font-semibold mb-4 flex items-center gap-2">
          <Briefcase className="w-5 h-5" />
          Professional Information
        </h3>
        
        <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
          <div className="space-y-2">
            <Label htmlFor="occupation">Occupation</Label>
            <Input
              id="occupation"
              value={profileData.occupation}
              onChange={(e) => updateProfileData('occupation', e.target.value)}
              placeholder="What do you do for work?"
            />
          </div>
          
          <div className="space-y-2">
            <Label>Education</Label>
            <Select value={profileData.education} onValueChange={(value) => updateProfileData('education', value)}>
              <SelectTrigger>
                <SelectValue placeholder="Select your education level" />
              </SelectTrigger>
              <SelectContent>
                {educationLevels.map((level) => (
                  <SelectItem key={level} value={level}>{level}</SelectItem>
                ))}
              </SelectContent>
            </Select>
          </div>
        </div>
      </Card>

      {/* Location */}
      <Card className="p-6">
        <h3 className="text-lg font-semibold mb-4 flex items-center gap-2">
          <MapPin className="w-5 h-5" />
          Location
        </h3>
        
        <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
          <div className="space-y-2">
            <Label htmlFor="location">City/Location</Label>
            <Input
              id="location"
              value={profileData.location}
              onChange={(e) => updateProfileData('location', e.target.value)}
              placeholder="e.g., Manhattan, New York"
            />
          </div>
          
          <div className="space-y-2">
            <Label>Country</Label>
            <Select value={profileData.country} onValueChange={(value) => updateProfileData('country', value)}>
              <SelectTrigger>
                <SelectValue placeholder="Select your country" />
              </SelectTrigger>
              <SelectContent>
                {countries.map((country) => (
                  <SelectItem key={country} value={country}>{country}</SelectItem>
                ))}
              </SelectContent>
            </Select>
          </div>
        </div>
      </Card>

      {/* Interests */}
      <Card className="p-6">
        <h3 className="text-lg font-semibold mb-4">Interests & Hobbies</h3>
        
        <div className="space-y-4">
          <div className="flex flex-wrap gap-2">
            {profileData.interests.map((interest) => (
              <Badge key={interest} variant="secondary" className="flex items-center gap-1">
                {interest}
                <X 
                  className="w-3 h-3 cursor-pointer hover:text-destructive" 
                  onClick={() => removeInterest(interest)}
                />
              </Badge>
            ))}
          </div>
          
          <div className="flex gap-2">
            <Input
              value={newInterest}
              onChange={(e) => setNewInterest(e.target.value)}
              placeholder="Add an interest..."
              onKeyPress={(e) => e.key === 'Enter' && addInterest(newInterest)}
            />
            <Button onClick={() => addInterest(newInterest)} variant="outline" size="sm">
              <Plus className="w-4 h-4" />
            </Button>
          </div>
          
          <div className="flex flex-wrap gap-2">
            <span className="text-sm text-muted-foreground">Quick add:</span>
            {commonInterests.filter(interest => !profileData.interests.includes(interest)).slice(0, 6).map((interest) => (
              <Button
                key={interest}
                variant="outline"
                size="sm"
                onClick={() => addInterest(interest)}
                className="text-xs"
              >
                {interest}
              </Button>
            ))}
          </div>
        </div>
      </Card>

      {/* What I'm Looking For */}
      <Card className="p-6">
        <h3 className="text-lg font-semibold mb-4 flex items-center gap-2">
          <Target className="w-5 h-5" />
          What I'm Looking For
        </h3>
        
        <div className="space-y-6">
          <div className="space-y-2">
            <Label>Relationship Goal</Label>
            <Select value={profileData.relationshipGoal} onValueChange={(value) => updateProfileData('relationshipGoal', value)}>
              <SelectTrigger>
                <SelectValue placeholder="What are you looking for?" />
              </SelectTrigger>
              <SelectContent>
                {relationshipGoals.map((goal) => (
                  <SelectItem key={goal} value={goal}>{goal}</SelectItem>
                ))}
              </SelectContent>
            </Select>
          </div>
          
          <div className="space-y-2">
            <div className="flex items-center justify-between">
              <Label htmlFor="lookingForText">Detailed Description</Label>
              <Button
                variant="outline"
                size="sm"
                onClick={() => setAiLookingForOpen(true)}
                className="flex items-center gap-2"
              >
                <Sparkles className="w-3 h-3" />
                AI Assist
              </Button>
            </div>
            <Textarea
              id="lookingForText"
              value={profileData.lookingForText}
              onChange={(e) => handleLookingForChange(e.target.value)}
              placeholder="Describe your ideal partner and relationship..."
              rows={3}
              className="resize-none"
              maxLength={LOOKING_FOR_MAX_CHARS}
            />
            <div className="flex justify-between items-center">
              <p className={`text-sm ${profileData.lookingForText.length > LOOKING_FOR_MAX_CHARS * 0.9 ? 'text-orange-500' : 'text-muted-foreground'}`}>
                {profileData.lookingForText.length}/{LOOKING_FOR_MAX_CHARS} characters
              </p>
              {profileData.lookingForText.length >= LOOKING_FOR_MAX_CHARS && (
                <span className="text-sm text-red-500">Character limit reached</span>
              )}
            </div>
          </div>
          
          <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
            <div className="space-y-2">
              <Label>Age Range</Label>
              <div className="flex items-center gap-2">
                <Input
                  type="number"
                  min="18"
                  max="100"
                  value={profileData.ageRangeMin}
                  onChange={(e) => updateProfileData('ageRangeMin', parseInt(e.target.value))}
                  className="w-20"
                />
                <span>to</span>
                <Input
                  type="number"
                  min="18"
                  max="100"
                  value={profileData.ageRangeMax}
                  onChange={(e) => updateProfileData('ageRangeMax', parseInt(e.target.value))}
                  className="w-20"
                />
              </div>
            </div>
            
            <div className="space-y-2">
              <Label>Maximum Distance</Label>
              <Select value={profileData.maxDistance.toString()} onValueChange={(value) => updateProfileData('maxDistance', parseInt(value))}>
                <SelectTrigger>
                  <SelectValue placeholder="Select maximum distance" />
                </SelectTrigger>
                <SelectContent>
                  {distanceOptions.map((distance) => (
                    <SelectItem key={distance.value} value={distance.value.toString()}>
                      {distance.label}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>
          </div>
        </div>
      </Card>

      {/* Important Qualities */}
      <Card className="p-6">
        <h3 className="text-lg font-semibold mb-4">Important Qualities</h3>
        
        <div className="space-y-4">
          <div className="flex flex-wrap gap-2">
            {profileData.importantQualities.map((quality) => (
              <Badge key={quality} variant="secondary" className="flex items-center gap-1 bg-green-100 text-green-800 dark:bg-green-900/30 dark:text-green-300">
                {quality}
                <X 
                  className="w-3 h-3 cursor-pointer hover:text-destructive" 
                  onClick={() => removeQuality(quality)}
                />
              </Badge>
            ))}
          </div>
          
          <div className="flex gap-2">
            <Input
              value={newQuality}
              onChange={(e) => setNewQuality(e.target.value)}
              placeholder="Add an important quality..."
              onKeyPress={(e) => e.key === 'Enter' && addQuality(newQuality)}
            />
            <Button onClick={() => addQuality(newQuality)} variant="outline" size="sm">
              <Plus className="w-4 h-4" />
            </Button>
          </div>
          
          <div className="flex flex-wrap gap-2">
            <span className="text-sm text-muted-foreground">Suggestions:</span>
            {commonQualities.filter(quality => !profileData.importantQualities.includes(quality)).slice(0, 4).map((quality) => (
              <Button
                key={quality}
                variant="outline"
                size="sm"
                onClick={() => addQuality(quality)}
                className="text-xs"
              >
                {quality}
              </Button>
            ))}
          </div>
        </div>
      </Card>

      {/* Deal Breakers */}
      <Card className="p-6">
        <h3 className="text-lg font-semibold mb-4">Deal Breakers</h3>
        
        <div className="space-y-4">
          <div className="flex flex-wrap gap-2">
            {profileData.dealBreakers.map((dealBreaker) => (
              <Badge key={dealBreaker} variant="secondary" className="flex items-center gap-1 bg-red-100 text-red-800 dark:bg-red-900/30 dark:text-red-300">
                {dealBreaker}
                <X 
                  className="w-3 h-3 cursor-pointer hover:text-destructive" 
                  onClick={() => removeDealBreaker(dealBreaker)}
                />
              </Badge>
            ))}
          </div>
          
          <div className="flex gap-2">
            <Input
              value={newDealBreaker}
              onChange={(e) => setNewDealBreaker(e.target.value)}
              placeholder="Add a deal breaker..."
              onKeyPress={(e) => e.key === 'Enter' && addDealBreaker(newDealBreaker)}
            />
            <Button onClick={() => addDealBreaker(newDealBreaker)} variant="outline" size="sm">
              <Plus className="w-4 h-4" />
            </Button>
          </div>
          
          <div className="flex flex-wrap gap-2">
            <span className="text-sm text-muted-foreground">Common:</span>
            {commonDealBreakers.filter(db => !profileData.dealBreakers.includes(db)).slice(0, 4).map((dealBreaker) => (
              <Button
                key={dealBreaker}
                variant="outline"
                size="sm"
                onClick={() => addDealBreaker(dealBreaker)}
                className="text-xs"
              >
                {dealBreaker}
              </Button>
            ))}
          </div>
        </div>
      </Card>

      {/* Country Preferences */}
      <Card className="p-6">
        <h3 className="text-lg font-semibold mb-4 flex items-center gap-2">
          <Globe className="w-5 h-5" />
          Country Preferences
        </h3>
        
        <div className="space-y-4">
          <div className="space-y-3">
            <div className="flex items-center space-x-2">
              <input
                type="radio"
                id="all-countries"
                name="countries"
                checked={profileData.countriesPreference === 'all'}
                onChange={() => updateProfileData('countriesPreference', 'all')}
                className="w-4 h-4"
              />
              <label htmlFor="all-countries" className="text-sm font-medium">
                Open to all countries
              </label>
            </div>
            
            <div className="flex items-center space-x-2">
              <input
                type="radio"
                id="specific-countries"
                name="countries"
                checked={profileData.countriesPreference === 'specific'}
                onChange={() => updateProfileData('countriesPreference', 'specific')}
                className="w-4 h-4"
              />
              <label htmlFor="specific-countries" className="text-sm font-medium">
                Specific countries only
              </label>
            </div>
            
            <div className="flex items-center space-x-2">
              <input
                type="radio"
                id="multiple-countries"
                name="countries"
                checked={profileData.countriesPreference === 'multiple'}
                onChange={() => updateProfileData('countriesPreference', 'multiple')}
                className="w-4 h-4"
              />
              <label htmlFor="multiple-countries" className="text-sm font-medium">
                Multiple preferred countries
              </label>
            </div>
          </div>
          
          {profileData.countriesPreference !== 'all' && (
            <div className="space-y-4">
              <div className="flex flex-wrap gap-2">
                {profileData.selectedCountries.map((country) => (
                  <Badge key={country} variant="secondary" className="flex items-center gap-1">
                    {country}
                    <X 
                      className="w-3 h-3 cursor-pointer hover:text-destructive" 
                      onClick={() => removeCountry(country)}
                    />
                  </Badge>
                ))}
              </div>
              
              <Select onValueChange={addCountry}>
                <SelectTrigger>
                  <SelectValue placeholder="Add a country" />
                </SelectTrigger>
                <SelectContent>
                  {countries.filter(country => !profileData.selectedCountries.includes(country)).map((country) => (
                    <SelectItem key={country} value={country}>{country}</SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>
          )}
        </div>
      </Card>

      {/* AI Writing Assistant Modals */}
      {aiAssistantOpen && (
        <AIWritingAssistant
          currentText={profileData.bio}
          onUpdate={handleAIBioUpdate}
          onClose={() => setAiAssistantOpen(false)}
          isOpen={aiAssistantOpen}
        />
      )}

      {aiLookingForOpen && (
        <AIWritingAssistant
          currentText={profileData.lookingForText}
          onUpdate={handleAILookingForUpdate}
          onClose={() => setAiLookingForOpen(false)}
          isOpen={aiLookingForOpen}
        />
      )}
    </div>
  );
}