import { useState } from 'react';
import { Card } from './ui/card';
import { Button } from './ui/button';
import { Label } from './ui/label';
import { Switch } from './ui/switch';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from './ui/select';
import { Alert, AlertDescription } from './ui/alert';
import { Separator } from './ui/separator';
import { 
  Shield, 
  Eye, 
  EyeOff, 
  Lock, 
  MapPin, 
  MessageCircle, 
  Phone,
  Heart,
  UserCheck,
  Globe,
  Info,
  Settings,
  AlertTriangle,
  Bell
} from 'lucide-react';

export function PrivacySettingsTab() {
  const [profileVisibility, setProfileVisibility] = useState('public');
  const [photoVisibility, setPhotoVisibility] = useState('all');
  const [lastSeenVisibility, setLastSeenVisibility] = useState(true);
  const [onlineStatus, setOnlineStatus] = useState(true);
  const [locationVisibility, setLocationVisibility] = useState('city');
  const [messagePermissions, setMessagePermissions] = useState('verified');
  const [phoneVisibility, setPhoneVisibility] = useState(false);
  const [likeNotifications, setLikeNotifications] = useState(true);
  const [viewNotifications, setViewNotifications] = useState(false);
  const [matchNotifications, setMatchNotifications] = useState(true);
  const [searchable, setSearchable] = useState(true);
  const [recommendToFriends, setRecommendToFriends] = useState(false);
  const [showInNearby, setShowInNearby] = useState(true);
  const [allowScreenshots, setAllowScreenshots] = useState(false);
  const [twoFactorAuth, setTwoFactorAuth] = useState(false);

  const handleSavePrivacySettings = () => {
    // In a real app, this would save to the backend
    console.log('Saving privacy settings:', {
      profileVisibility,
      photoVisibility,
      lastSeenVisibility,
      onlineStatus,
      locationVisibility,
      messagePermissions,
      phoneVisibility,
      likeNotifications,
      viewNotifications,
      matchNotifications,
      searchable,
      recommendToFriends,
      showInNearby,
      allowScreenshots,
      twoFactorAuth
    });
    // Show success toast
  };

  return (
    <div className="space-y-6">
      {/* Header */}
      <div className="flex items-center justify-between">
        <div className="flex items-center gap-3">
          <div className="p-2 rounded-full bg-gradient-to-r from-green-500 to-teal-600">
            <Shield className="w-6 h-6 text-white" />
          </div>
          <div>
            <h2 className="text-2xl font-bold">Privacy Settings</h2>
            <p className="text-muted-foreground">Control who can see your information and contact you</p>
          </div>
        </div>
        
        <Button onClick={handleSavePrivacySettings} className="flex items-center gap-2">
          <Shield className="w-4 h-4" />
          Save Privacy Settings
        </Button>
      </div>

      {/* Privacy Alert */}
      <Alert>
        <Shield className="h-4 w-4" />
        <AlertDescription>
          <strong>Your Privacy Matters:</strong> These settings control how other members can interact with you and what information they can see. 
          You can change these settings anytime.
        </AlertDescription>
      </Alert>

      {/* Profile Visibility */}
      <Card className="p-6">
        <h3 className="text-lg font-semibold mb-4 flex items-center gap-2">
          <Eye className="w-5 h-5" />
          Profile Visibility
        </h3>
        
        <div className="space-y-4">
          <div className="space-y-2">
            <Label>Who can see your profile</Label>
            <Select value={profileVisibility} onValueChange={setProfileVisibility}>
              <SelectTrigger>
                <SelectValue placeholder="Select visibility level" />
              </SelectTrigger>
              <SelectContent>
                <SelectItem value="public">Everyone</SelectItem>
                <SelectItem value="verified">Verified members only</SelectItem>
                <SelectItem value="premium">Premium members only</SelectItem>
                <SelectItem value="private">Private (invisible)</SelectItem>
              </SelectContent>
            </Select>
            <p className="text-sm text-muted-foreground">
              {profileVisibility === 'public' && 'Your profile is visible to all members'}
              {profileVisibility === 'verified' && 'Only verified members can see your profile'}
              {profileVisibility === 'premium' && 'Only premium members can see your profile'}
              {profileVisibility === 'private' && 'Your profile is hidden from search and discovery'}
            </p>
          </div>

          <Separator />

          <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
            <div className="flex items-center justify-between">
              <div>
                <Label htmlFor="searchable">Appear in search results</Label>
                <p className="text-sm text-muted-foreground">Allow others to find you through search</p>
              </div>
              <Switch 
                id="searchable" 
                checked={searchable}
                onCheckedChange={setSearchable}
              />
            </div>

            <div className="flex items-center justify-between">
              <div>
                <Label htmlFor="show-nearby">Show in nearby members</Label>
                <p className="text-sm text-muted-foreground">Appear in location-based discovery</p>
              </div>
              <Switch 
                id="show-nearby" 
                checked={showInNearby}
                onCheckedChange={setShowInNearby}
              />
            </div>
          </div>
        </div>
      </Card>

      {/* Photo Privacy */}
      <Card className="p-6">
        <h3 className="text-lg font-semibold mb-4 flex items-center gap-2">
          <Eye className="w-5 h-5" />
          Photo Privacy
        </h3>
        
        <div className="space-y-4">
          <div className="space-y-2">
            <Label>Who can see your photos</Label>
            <Select value={photoVisibility} onValueChange={setPhotoVisibility}>
              <SelectTrigger>
                <SelectValue placeholder="Select photo visibility" />
              </SelectTrigger>
              <SelectContent>
                <SelectItem value="all">All members</SelectItem>
                <SelectItem value="verified">Verified members only</SelectItem>
                <SelectItem value="premium">Premium members only</SelectItem>
                <SelectItem value="matches">Matches only</SelectItem>
              </SelectContent>
            </Select>
          </div>

          <div className="flex items-center justify-between">
            <div>
              <Label htmlFor="allow-screenshots">Allow photo screenshots</Label>
              <p className="text-sm text-muted-foreground">Disable to prevent screenshot notifications</p>
            </div>
            <Switch 
              id="allow-screenshots" 
              checked={allowScreenshots}
              onCheckedChange={setAllowScreenshots}
            />
          </div>
        </div>
      </Card>

      {/* Activity & Status */}
      <Card className="p-6">
        <h3 className="text-lg font-semibold mb-4 flex items-center gap-2">
          <UserCheck className="w-5 h-5" />
          Activity & Status
        </h3>
        
        <div className="space-y-4">
          <div className="flex items-center justify-between">
            <div>
              <Label htmlFor="online-status">Show online status</Label>
              <p className="text-sm text-muted-foreground">Let others see when you're online</p>
            </div>
            <Switch 
              id="online-status" 
              checked={onlineStatus}
              onCheckedChange={setOnlineStatus}
            />
          </div>

          <div className="flex items-center justify-between">
            <div>
              <Label htmlFor="last-seen">Show last seen</Label>
              <p className="text-sm text-muted-foreground">Display when you were last active</p>
            </div>
            <Switch 
              id="last-seen" 
              checked={lastSeenVisibility}
              onCheckedChange={setLastSeenVisibility}
            />
          </div>
        </div>
      </Card>

      {/* Location Privacy */}
      <Card className="p-6">
        <h3 className="text-lg font-semibold mb-4 flex items-center gap-2">
          <MapPin className="w-5 h-5" />
          Location Privacy
        </h3>
        
        <div className="space-y-4">
          <div className="space-y-2">
            <Label>Location visibility</Label>
            <Select value={locationVisibility} onValueChange={setLocationVisibility}>
              <SelectTrigger>
                <SelectValue placeholder="Select location visibility" />
              </SelectTrigger>
              <SelectContent>
                <SelectItem value="exact">Exact location</SelectItem>
                <SelectItem value="city">City only</SelectItem>
                <SelectItem value="region">Region/State only</SelectItem>
                <SelectItem value="country">Country only</SelectItem>
                <SelectItem value="hidden">Hidden</SelectItem>
              </SelectContent>
            </Select>
            <p className="text-sm text-muted-foreground">
              Choose how specific your location appears to other members
            </p>
          </div>
        </div>
      </Card>

      {/* Communication Settings */}
      <Card className="p-6">
        <h3 className="text-lg font-semibold mb-4 flex items-center gap-2">
          <MessageCircle className="w-5 h-5" />
          Communication Settings
        </h3>
        
        <div className="space-y-4">
          <div className="space-y-2">
            <Label>Who can message you</Label>
            <Select value={messagePermissions} onValueChange={setMessagePermissions}>
              <SelectTrigger>
                <SelectValue placeholder="Select messaging permissions" />
              </SelectTrigger>
              <SelectContent>
                <SelectItem value="everyone">Everyone</SelectItem>
                <SelectItem value="verified">Verified members only</SelectItem>
                <SelectItem value="premium">Premium members only</SelectItem>
                <SelectItem value="matches">Matches only</SelectItem>
                <SelectItem value="none">No one</SelectItem>
              </SelectContent>
            </Select>
          </div>

          <div className="flex items-center justify-between">
            <div>
              <Label htmlFor="phone-visibility">Show phone number to matches</Label>
              <p className="text-sm text-muted-foreground">Allow matches to see your phone number</p>
            </div>
            <Switch 
              id="phone-visibility" 
              checked={phoneVisibility}
              onCheckedChange={setPhoneVisibility}
            />
          </div>
        </div>
      </Card>

      {/* Notification Preferences */}
      <Card className="p-6">
        <h3 className="text-lg font-semibold mb-4 flex items-center gap-2">
          <Bell className="w-5 h-5" />
          Notification Preferences
        </h3>
        
        <div className="space-y-4">
          <div className="flex items-center justify-between">
            <div>
              <Label htmlFor="like-notifications">Like notifications</Label>
              <p className="text-sm text-muted-foreground">Get notified when someone likes your profile</p>
            </div>
            <Switch 
              id="like-notifications" 
              checked={likeNotifications}
              onCheckedChange={setLikeNotifications}
            />
          </div>

          <div className="flex items-center justify-between">
            <div>
              <Label htmlFor="view-notifications">Profile view notifications</Label>
              <p className="text-sm text-muted-foreground">Get notified when someone views your profile</p>
            </div>
            <Switch 
              id="view-notifications" 
              checked={viewNotifications}
              onCheckedChange={setViewNotifications}
            />
          </div>

          <div className="flex items-center justify-between">
            <div>
              <Label htmlFor="match-notifications">Match notifications</Label>
              <p className="text-sm text-muted-foreground">Get notified about new matches</p>
            </div>
            <Switch 
              id="match-notifications" 
              checked={matchNotifications}
              onCheckedChange={setMatchNotifications}
            />
          </div>
        </div>
      </Card>

      {/* Account Security */}
      <Card className="p-6">
        <h3 className="text-lg font-semibold mb-4 flex items-center gap-2">
          <Lock className="w-5 h-5" />
          Account Security
        </h3>
        
        <div className="space-y-4">
          <div className="flex items-center justify-between">
            <div>
              <Label htmlFor="two-factor">Two-factor authentication</Label>
              <p className="text-sm text-muted-foreground">Add an extra layer of security to your account</p>
            </div>
            <div className="flex items-center gap-2">
              <Switch 
                id="two-factor" 
                checked={twoFactorAuth}
                onCheckedChange={setTwoFactorAuth}
              />
              {!twoFactorAuth && (
                <Button variant="outline" size="sm">
                  Set Up
                </Button>
              )}
            </div>
          </div>

          <div className="flex items-center justify-between">
            <div>
              <Label htmlFor="recommend-friends">Recommend to friends</Label>
              <p className="text-sm text-muted-foreground">Allow the app to suggest your profile to people in your contacts</p>
            </div>
            <Switch 
              id="recommend-friends" 
              checked={recommendToFriends}
              onCheckedChange={setRecommendToFriends}
            />
          </div>
        </div>
      </Card>

      {/* Privacy Tips */}
      <Card className="p-6 bg-gradient-to-br from-blue-50 to-indigo-50 dark:from-blue-950/20 dark:to-indigo-950/20">
        <h4 className="font-semibold mb-3 text-blue-900 dark:text-blue-100 flex items-center gap-2">
          <Shield className="w-5 h-5" />
          🔒 Privacy & Safety Tips
        </h4>
        <div className="space-y-2 text-sm text-blue-800 dark:text-blue-200">
          <div className="flex items-start gap-2">
            <Info className="w-4 h-4 mt-0.5 text-blue-600" />
            <span><strong>Be selective:</strong> Only share personal information with people you trust</span>
          </div>
          <div className="flex items-start gap-2">
            <Info className="w-4 h-4 mt-0.5 text-blue-600" />
            <span><strong>Report suspicious behavior:</strong> Use our reporting tools if someone makes you uncomfortable</span>
          </div>
          <div className="flex items-start gap-2">
            <Info className="w-4 h-4 mt-0.5 text-blue-600" />
            <span><strong>Control your visibility:</strong> Adjust these settings anytime based on your comfort level</span>
          </div>
          <div className="flex items-start gap-2">
            <AlertTriangle className="w-4 h-4 mt-0.5 text-amber-600" />
            <span><strong>Never share:</strong> Passwords, financial information, or overly personal details</span>
          </div>
        </div>
      </Card>
    </div>
  );
}