import { useState } from "react";
import { Dialog, DialogContent, DialogDescription, DialogHeader, DialogTitle } from "./ui/dialog";
import { Button } from "./ui/button";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "./ui/card";
import { Badge } from "./ui/badge";
import { Alert, AlertDescription } from "./ui/alert";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "./ui/tabs";
import { Progress } from "./ui/progress";
import { 
  Coins, 
  Crown, 
  TrendingUp, 
  Award, 
  Gift, 
  Clock, 
  Zap,
  Heart,
  MessageCircle,
  Eye,
  ShoppingCart,
  CheckCircle,
  AlertTriangle
} from "lucide-react";
import { motion } from "motion/react";
import { usePoints } from "./PointsProvider";
import { useUser } from "./UserProvider";
import { toast } from "sonner";

interface PointsStoreProps {
  isOpen: boolean;
  onClose: () => void;
}

interface StoreProduct {
  id: string;
  name: string;
  description: string;
  pointsCost: number;
  type: "boost" | "premium" | "feature" | "membership";
  duration?: string;
  isActive: boolean;
  icon: any;
  popular?: boolean;
  discount?: number;
  originalCost?: number;
}

export function PointsStore({ isOpen, onClose }: PointsStoreProps) {
  const { points, spendPoints, pointsHistory } = usePoints();
  const { user, updateUser } = useUser();
  const [activeTab, setActiveTab] = useState("products");
  const [purchasingProduct, setPurchasingProduct] = useState<string | null>(null);

  const storeProducts: StoreProduct[] = [
    {
      id: "profile-boost",
      name: "Profile Boost",
      description: "Boost your profile visibility for 24 hours. Get 3x more profile views!",
      pointsCost: 500,
      type: "boost",
      duration: "24 hours",
      isActive: true,
      icon: TrendingUp,
      popular: true
    },
    {
      id: "super-like-pack",
      name: "Super Likes Pack",
      description: "10 Super Likes to stand out and catch someone's attention",
      pointsCost: 300,
      type: "feature",
      isActive: true,
      icon: Heart
    },
    {
      id: "priority-messaging",
      name: "Priority Messaging",
      description: "Your messages appear first in inboxes for 7 days",
      pointsCost: 800,
      type: "feature",
      duration: "7 days",
      isActive: true,
      icon: MessageCircle
    },
    {
      id: "premium-week",
      name: "Premium Membership - 1 Week",
      description: "Unlock all premium features: unlimited likes, advanced filters, and more",
      pointsCost: 1800,
      originalCost: 2000,
      discount: 10,
      type: "premium",
      duration: "7 days",
      isActive: true,
      icon: Crown
    },
    {
      id: "premium-month",
      name: "Premium Membership - 1 Month",
      description: "Full premium access with the best value for serious daters",
      pointsCost: 6000,
      type: "premium",
      duration: "30 days",
      isActive: true,
      icon: Crown,
      popular: true
    },
    {
      id: "incognito-mode",
      name: "Incognito Mode",
      description: "Browse profiles privately for 3 days without being seen",
      pointsCost: 400,
      type: "feature",
      duration: "3 days",
      isActive: true,
      icon: Eye
    },
    {
      id: "boost-bundle",
      name: "Boost Bundle",
      description: "3 Profile Boosts + 5 Super Likes - Save 200 points!",
      pointsCost: 1200,
      originalCost: 1400,
      discount: 14,
      type: "boost",
      isActive: true,
      icon: Zap
    }
  ];

  const handlePurchase = async (product: StoreProduct) => {
    setPurchasingProduct(product.id);

    try {
      // Check if user has enough points
      if (points < product.pointsCost) {
        toast.error("Not enough points!", {
          description: `You need ${product.pointsCost - points} more points to purchase this item.`
        });
        return;
      }

      // Simulate purchase delay
      await new Promise(resolve => setTimeout(resolve, 1000));

      // Spend points
      const success = spendPoints(product.pointsCost, `Purchased: ${product.name}`);
      
      if (success) {
        // Apply the purchased benefit
        applyProductBenefit(product);
        
        toast.success("Purchase successful!", {
          description: `${product.name} has been added to your account.`
        });
      } else {
        toast.error("Purchase failed!", {
          description: "There was an error processing your purchase."
        });
      }
    } catch (error) {
      toast.error("Purchase failed!", {
        description: "There was an error processing your purchase."
      });
    } finally {
      setPurchasingProduct(null);
    }
  };

  const applyProductBenefit = (product: StoreProduct) => {
    // In real app, this would update user's benefits in the backend
    switch (product.type) {
      case "premium":
        if (user) {
          const expiryDate = new Date();
          const days = product.duration?.includes("Week") ? 7 : 30;
          expiryDate.setDate(expiryDate.getDate() + days);
          
          updateUser({
            ...user,
            membershipTier: "premium",
            membershipExpiry: expiryDate.toISOString()
          });
        }
        break;
      case "boost":
        // Apply boost logic
        console.log(`Applied ${product.name} to user profile`);
        break;
      case "feature":
        // Apply feature logic
        console.log(`Applied ${product.name} feature`);
        break;
    }
  };

  const getProgressToNext = () => {
    const nextMilestone = 1000;
    const currentProgress = points % nextMilestone;
    return (currentProgress / nextMilestone) * 100;
  };

  const recentTransactions = pointsHistory.slice(0, 5);

  return (
    <Dialog open={isOpen} onOpenChange={onClose}>
      <DialogContent className="max-w-6xl max-h-[90vh] overflow-y-auto">
        <DialogHeader>
          <DialogTitle className="flex items-center gap-2">
            <ShoppingCart className="w-5 h-5 text-orange-600" />
            Points Store
          </DialogTitle>
          <DialogDescription>
            Use your earned points to unlock premium features and boost your dating success!
          </DialogDescription>
        </DialogHeader>

        {/* Points Balance */}
        <div className="bg-gradient-to-r from-orange-50 to-orange-100 dark:from-orange-900/20 dark:to-orange-800/20 rounded-lg p-6 mb-6">
          <div className="flex items-center justify-between">
            <div>
              <div className="flex items-center gap-2 mb-2">
                <Coins className="w-6 h-6 text-orange-600" />
                <span className="text-2xl font-bold">{points.toLocaleString()}</span>
                <span className="text-muted-foreground">points available</span>
              </div>
              <div className="space-y-2">
                <div className="flex items-center justify-between text-sm">
                  <span>Progress to next 1,000 points</span>
                  <span>{points % 1000}/1,000</span>
                </div>
                <Progress value={getProgressToNext()} className="h-2" />
              </div>
            </div>
            <Button variant="outline" onClick={() => setActiveTab("history")}>
              View History
            </Button>
          </div>
        </div>

        <Tabs value={activeTab} onValueChange={setActiveTab} className="space-y-6">
          <TabsList className="grid w-full grid-cols-3">
            <TabsTrigger value="products">Store</TabsTrigger>
            <TabsTrigger value="membership">Membership</TabsTrigger>
            <TabsTrigger value="history">History</TabsTrigger>
          </TabsList>

          {/* Store Tab */}
          <TabsContent value="products" className="space-y-6">
            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
              {storeProducts
                .filter(p => p.type !== "premium")
                .map((product) => {
                  const Icon = product.icon;
                  const canAfford = points >= product.pointsCost;
                  const isPurchasing = purchasingProduct === product.id;

                  return (
                    <motion.div
                      key={product.id}
                      whileHover={{ scale: 1.02 }}
                      whileTap={{ scale: 0.98 }}
                    >
                      <Card className={`relative h-full ${!canAfford ? 'opacity-70' : ''} ${product.popular ? 'ring-2 ring-orange-600' : ''}`}>
                        {product.popular && (
                          <Badge className="absolute -top-2 left-4 bg-orange-600 text-white">
                            Popular
                          </Badge>
                        )}
                        {product.discount && (
                          <Badge className="absolute -top-2 right-4 bg-green-600 text-white">
                            {product.discount}% OFF
                          </Badge>
                        )}
                        
                        <CardHeader className="pb-3">
                          <div className="flex items-center gap-2 mb-2">
                            <div className="p-2 bg-orange-100 dark:bg-orange-900/20 rounded-lg">
                              <Icon className="w-5 h-5 text-orange-600" />
                            </div>
                            <CardTitle className="text-lg">{product.name}</CardTitle>
                          </div>
                          <CardDescription className="text-sm">
                            {product.description}
                          </CardDescription>
                        </CardHeader>
                        
                        <CardContent className="space-y-4">
                          {product.duration && (
                            <div className="flex items-center gap-1 text-sm text-muted-foreground">
                              <Clock className="w-4 h-4" />
                              Duration: {product.duration}
                            </div>
                          )}
                          
                          <div className="flex items-center justify-between">
                            <div className="flex items-center gap-2">
                              {product.originalCost && (
                                <span className="text-sm text-muted-foreground line-through">
                                  {product.originalCost}
                                </span>
                              )}
                              <div className="flex items-center gap-1">
                                <Coins className="w-4 h-4 text-orange-600" />
                                <span className="font-bold text-lg">{product.pointsCost}</span>
                              </div>
                            </div>
                          </div>
                          
                          <Button
                            onClick={() => handlePurchase(product)}
                            disabled={!canAfford || isPurchasing}
                            className="w-full bg-orange-600 hover:bg-orange-700"
                          >
                            {isPurchasing ? (
                              "Processing..."
                            ) : !canAfford ? (
                              `Need ${product.pointsCost - points} more points`
                            ) : (
                              "Purchase"
                            )}
                          </Button>
                        </CardContent>
                      </Card>
                    </motion.div>
                  );
                })}
            </div>
          </TabsContent>

          {/* Membership Tab */}
          <TabsContent value="membership" className="space-y-6">
            <Alert>
              <Crown className="w-4 h-4" />
              <AlertDescription>
                <strong>Premium Membership Benefits:</strong> Unlimited likes, advanced search filters, 
                see who liked you, priority customer support, ad-free experience, and premium badges.
              </AlertDescription>
            </Alert>

            <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
              {storeProducts
                .filter(p => p.type === "premium")
                .map((product) => {
                  const Icon = product.icon;
                  const canAfford = points >= product.pointsCost;
                  const isPurchasing = purchasingProduct === product.id;

                  return (
                    <motion.div
                      key={product.id}
                      whileHover={{ scale: 1.02 }}
                      whileTap={{ scale: 0.98 }}
                    >
                      <Card className={`relative ${!canAfford ? 'opacity-70' : ''} ${product.popular ? 'ring-2 ring-orange-600' : ''}`}>
                        {product.popular && (
                          <Badge className="absolute -top-2 left-4 bg-orange-600 text-white">
                            Best Value
                          </Badge>
                        )}
                        {product.discount && (
                          <Badge className="absolute -top-2 right-4 bg-green-600 text-white">
                            {product.discount}% OFF
                          </Badge>
                        )}
                        
                        <CardHeader>
                          <div className="flex items-center gap-2 mb-2">
                            <div className="p-3 bg-gradient-to-r from-orange-100 to-yellow-100 dark:from-orange-900/20 dark:to-yellow-900/20 rounded-lg">
                              <Icon className="w-6 h-6 text-orange-600" />
                            </div>
                            <div>
                              <CardTitle className="text-xl">{product.name}</CardTitle>
                              <div className="text-sm text-muted-foreground">{product.duration}</div>
                            </div>
                          </div>
                          <CardDescription>{product.description}</CardDescription>
                        </CardHeader>
                        
                        <CardContent className="space-y-4">
                          <div className="space-y-2">
                            <div className="flex items-center gap-2 text-sm">
                              <CheckCircle className="w-4 h-4 text-green-600" />
                              <span>Unlimited likes and super likes</span>
                            </div>
                            <div className="flex items-center gap-2 text-sm">
                              <CheckCircle className="w-4 h-4 text-green-600" />
                              <span>Advanced search filters</span>
                            </div>
                            <div className="flex items-center gap-2 text-sm">
                              <CheckCircle className="w-4 h-4 text-green-600" />
                              <span>See who liked your profile</span>
                            </div>
                            <div className="flex items-center gap-2 text-sm">
                              <CheckCircle className="w-4 h-4 text-green-600" />
                              <span>Priority customer support</span>
                            </div>
                            <div className="flex items-center gap-2 text-sm">
                              <CheckCircle className="w-4 h-4 text-green-600" />
                              <span>Ad-free experience</span>
                            </div>
                          </div>
                          
                          <div className="flex items-center justify-between pt-4 border-t">
                            <div className="flex items-center gap-2">
                              {product.originalCost && (
                                <span className="text-muted-foreground line-through">
                                  {product.originalCost}
                                </span>
                              )}
                              <div className="flex items-center gap-1">
                                <Coins className="w-5 h-5 text-orange-600" />
                                <span className="font-bold text-2xl">{product.pointsCost}</span>
                              </div>
                            </div>
                          </div>
                          
                          <Button
                            onClick={() => handlePurchase(product)}
                            disabled={!canAfford || isPurchasing}
                            className="w-full bg-gradient-to-r from-orange-600 to-yellow-600 hover:from-orange-700 hover:to-yellow-700"
                            size="lg"
                          >
                            {isPurchasing ? (
                              "Processing..."
                            ) : !canAfford ? (
                              `Need ${product.pointsCost - points} more points`
                            ) : (
                              "Upgrade to Premium"
                            )}
                          </Button>
                        </CardContent>
                      </Card>
                    </motion.div>
                  );
                })}
            </div>
          </TabsContent>

          {/* History Tab */}
          <TabsContent value="history" className="space-y-6">
            <Card>
              <CardHeader>
                <CardTitle>Recent Transactions</CardTitle>
                <CardDescription>
                  Your latest point earnings and purchases
                </CardDescription>
              </CardHeader>
              <CardContent>
                {recentTransactions.length === 0 ? (
                  <div className="text-center py-8 text-muted-foreground">
                    No transactions yet. Start viewing ads to earn points!
                  </div>
                ) : (
                  <div className="space-y-3">
                    {recentTransactions.map((transaction) => (
                      <div key={transaction.id} className="flex items-center justify-between p-3 border rounded-lg">
                        <div className="flex items-center gap-3">
                          <div className={`p-2 rounded-lg ${
                            transaction.type === "earn" 
                              ? "bg-green-100 dark:bg-green-900/20" 
                              : "bg-red-100 dark:bg-red-900/20"
                          }`}>
                            {transaction.type === "earn" ? (
                              <Gift className="w-4 h-4 text-green-600" />
                            ) : (
                              <ShoppingCart className="w-4 h-4 text-red-600" />
                            )}
                          </div>
                          <div>
                            <div className="font-medium">{transaction.reason}</div>
                            <div className="text-sm text-muted-foreground">
                              {new Date(transaction.timestamp).toLocaleDateString()}
                            </div>
                          </div>
                        </div>
                        <div className="text-right">
                          <div className={`font-semibold ${
                            transaction.type === "earn" ? "text-green-600" : "text-red-600"
                          }`}>
                            {transaction.type === "earn" ? "+" : "-"}{transaction.amount}
                          </div>
                          <div className="text-sm text-muted-foreground">
                            Balance: {transaction.balance}
                          </div>
                        </div>
                      </div>
                    ))}
                  </div>
                )}
              </CardContent>
            </Card>
          </TabsContent>
        </Tabs>
      </DialogContent>
    </Dialog>
  );
}