import { createContext, useContext, useState, useEffect, ReactNode } from "react";
import { useUser } from "./UserProvider";

interface PointsContextType {
  points: number;
  earnPoints: (amount: number, reason: string) => void;
  spendPoints: (amount: number, reason: string) => boolean;
  pointsHistory: PointsTransaction[];
  refreshPoints: () => void;
}

interface PointsTransaction {
  id: number;
  amount: number;
  type: "earn" | "spend";
  reason: string;
  timestamp: string;
  balance: number;
}

const PointsContext = createContext<PointsContextType | undefined>(undefined);

interface PointsProviderProps {
  children: ReactNode;
}

export function PointsProvider({ children }: PointsProviderProps) {
  const { user } = useUser();
  const [points, setPoints] = useState(0);
  const [pointsHistory, setPointsHistory] = useState<PointsTransaction[]>([]);

  // Load points from localStorage on mount
  useEffect(() => {
    if (user) {
      const savedPoints = localStorage.getItem(`points_${user.id}`);
      const savedHistory = localStorage.getItem(`pointsHistory_${user.id}`);
      
      if (savedPoints) {
        setPoints(parseInt(savedPoints));
      }
      
      if (savedHistory) {
        setPointsHistory(JSON.parse(savedHistory));
      }
    }
  }, [user]);

  // Save points to localStorage whenever they change
  useEffect(() => {
    if (user) {
      localStorage.setItem(`points_${user.id}`, points.toString());
    }
  }, [points, user]);

  // Save history to localStorage whenever it changes
  useEffect(() => {
    if (user) {
      localStorage.setItem(`pointsHistory_${user.id}`, JSON.stringify(pointsHistory));
    }
  }, [pointsHistory, user]);

  const earnPoints = (amount: number, reason: string) => {
    const newBalance = points + amount;
    const transaction: PointsTransaction = {
      id: Date.now(),
      amount,
      type: "earn",
      reason,
      timestamp: new Date().toISOString(),
      balance: newBalance
    };

    setPoints(newBalance);
    setPointsHistory(prev => [transaction, ...prev].slice(0, 100)); // Keep last 100 transactions
  };

  const spendPoints = (amount: number, reason: string): boolean => {
    if (points < amount) {
      return false; // Not enough points
    }

    const newBalance = points - amount;
    const transaction: PointsTransaction = {
      id: Date.now(),
      amount,
      type: "spend",
      reason,
      timestamp: new Date().toISOString(),
      balance: newBalance
    };

    setPoints(newBalance);
    setPointsHistory(prev => [transaction, ...prev].slice(0, 100));
    return true;
  };

  const refreshPoints = () => {
    // In real app, this would fetch from server
    console.log("Refreshing points from server...");
  };

  return (
    <PointsContext.Provider value={{
      points,
      earnPoints,
      spendPoints,
      pointsHistory,
      refreshPoints
    }}>
      {children}
    </PointsContext.Provider>
  );
}

export function usePoints() {
  const context = useContext(PointsContext);
  if (context === undefined) {
    throw new Error("usePoints must be used within a PointsProvider");
  }
  return context;
}