import React, { useState, useEffect } from "react";
import { ImageWithFallback } from "./figma/ImageWithFallback";
import { Button } from "./ui/button";
import { Heart, Play, Expand, ZoomIn, RotateCw } from "lucide-react";
import { motion, AnimatePresence } from "motion/react";

interface PhotoCollageProps {
  photos: string[];
  onPhotoSelect?: (photo: string) => void;
  selectedPhoto?: string;
  photoLikes?: Record<string, { likes: number; isLiked: boolean }>;
  onPhotoLike?: (photo: string, e: React.MouseEvent) => void;
  className?: string;
  priority?: boolean;
}

export const PhotoCollage: React.FC<PhotoCollageProps> = ({
  photos,
  onPhotoSelect = () => {},
  selectedPhoto = "",
  photoLikes = {},
  onPhotoLike = () => {},
  className = "",
  priority = false,
}) => {
  const [hoveredPhoto, setHoveredPhoto] = useState<string | null>(null);
  const [lightboxOpen, setLightboxOpen] = useState(false);
  const [lightboxPhoto, setLightboxPhoto] = useState<string>("");
  const [displayedPhotos, setDisplayedPhotos] = useState<string[]>([]);
  const [isAutoRotating, setIsAutoRotating] = useState(true);
  const [fadingPositions, setFadingPositions] = useState<number[]>([]);

  // Filter out empty photos and get actual photo count
  const actualPhotos = photos.filter(p => p !== "");
  
  // Initialize displayed photos - all 9 photos in their positions
  useEffect(() => {
    if (actualPhotos.length > 0) {
      // Use all 9 photos directly in their positions
      const initialDisplay = [];
      for (let i = 0; i < 9; i++) {
        initialDisplay.push(actualPhotos[i] || "");
      }
      setDisplayedPhotos(initialDisplay);
    }
  }, [photos.length]);

  // Advanced dual rotation system - rotates 2 positions at once
  useEffect(() => {
    if (!isAutoRotating || actualPhotos.length < 9) return;
    
    const interval = setInterval(() => {
      // Pick 2 random positions to rotate (0-8)
      const positions = [];
      const firstPosition = Math.floor(Math.random() * 9);
      let secondPosition;
      do {
        secondPosition = Math.floor(Math.random() * 9);
      } while (secondPosition === firstPosition);
      
      positions.push(firstPosition, secondPosition);
      setFadingPositions(positions);
      
      // After fade out delay, change the photos
      setTimeout(() => {
        setDisplayedPhotos(prev => {
          const newDisplay = [...prev];
          
          // Swap the two positions with each other for dynamic effect
          const temp = newDisplay[positions[0]];
          newDisplay[positions[0]] = newDisplay[positions[1]];
          newDisplay[positions[1]] = temp;
          
          return newDisplay;
        });
        
        // Clear fading positions after photo change
        setTimeout(() => setFadingPositions([]), 100);
      }, 800); // 800ms fade out duration
      
    }, 6000); // Rotate every 6 seconds for smoother experience

    return () => clearInterval(interval);
  }, [isAutoRotating, actualPhotos.length]);

  const handleManualRotation = () => {
    if (actualPhotos.length < 9) return;
    
    setIsAutoRotating(false);
    
    // Manually rotate 2 random positions
    const positions = [];
    const firstPosition = Math.floor(Math.random() * 9);
    let secondPosition;
    do {
      secondPosition = Math.floor(Math.random() * 9);
    } while (secondPosition === firstPosition);
    
    positions.push(firstPosition, secondPosition);
    setFadingPositions(positions);
    
    setTimeout(() => {
      setDisplayedPhotos(prev => {
        const newDisplay = [...prev];
        
        // Swap the two positions
        const temp = newDisplay[positions[0]];
        newDisplay[positions[0]] = newDisplay[positions[1]];
        newDisplay[positions[1]] = temp;
        
        return newDisplay;
      });
      
      setTimeout(() => setFadingPositions([]), 100);
    }, 800);
    
    // Re-enable auto rotation after 20 seconds of inactivity
    setTimeout(() => setIsAutoRotating(true), 20000);
  };

  // Get current photo for a position
  const getPhotoForPosition = (position: number) => {
    return displayedPhotos[position] || "";
  };

  const openLightbox = (photo: string, e: React.MouseEvent) => {
    e.stopPropagation();
    setLightboxPhoto(photo);
    setLightboxOpen(true);
  };

  // Render photo block with dynamic styling
  const renderPhotoBlock = (
    position: number, 
    size: 'large' | 'medium' | 'small',
    className: string = "",
    aspectRatio: string = "",
    showMainBadge: boolean = false
  ) => {
    const photo = getPhotoForPosition(position);
    const isEmpty = !photo;
    const isRotating = fadingPositions.includes(position);
    const displayNumber = position + 1;

    // Define button sizes based on photo size
    const buttonSizes = {
      large: { container: "h-10 w-10", icon: "w-5 h-5" },
      medium: { container: "h-8 w-8", icon: "w-4 h-4" },
      small: { container: "h-6 w-6", icon: "w-3 h-3" }
    };

    const sizes = buttonSizes[size];

    return (
      <motion.div
        className={`relative overflow-hidden rounded-sm group cursor-pointer ${aspectRatio} ${className} ${
          isEmpty ? "bg-muted border-2 border-dashed border-border flex items-center justify-center" : ""
        }`}
        whileHover={{ scale: 1.02 }}
        transition={{ duration: 0.2 }}
        onClick={() => photo && onPhotoSelect(photo)}
        onMouseEnter={() => setHoveredPhoto(photo)}
        onMouseLeave={() => setHoveredPhoto(null)}
      >
        {isEmpty ? (
          <div className="flex flex-col items-center justify-center text-muted-foreground">
            <div className={`${size === 'large' ? 'w-12 h-12' : size === 'medium' ? 'w-8 h-8' : 'w-6 h-6'} rounded-full bg-muted-foreground/20 flex items-center justify-center ${size === 'large' ? 'mb-3' : 'mb-1'}`}>
              <span className={size === 'large' ? 'text-2xl' : size === 'medium' ? 'text-lg' : 'text-sm'}>+</span>
            </div>
            <span className={`text-center ${size === 'large' ? 'text-sm' : 'text-xs'}`}>
              {showMainBadge ? 'Add Main Photo' : 'Add Photo'}
            </span>
          </div>
        ) : (
          <>
            <motion.div
              animate={{ 
                opacity: isRotating ? 0 : 1,
                scale: isRotating ? 0.95 : 1
              }}
              transition={{ duration: 0.8, ease: "easeInOut" }}
            >
              <ImageWithFallback
                src={photo}
                alt={`Gallery photo ${displayNumber}`}
                className="w-full h-full object-cover object-center transition-transform duration-300 group-hover:scale-110"
              />
            </motion.div>
            
            {/* Selected indicator */}
            {selectedPhoto === photo && (
              <div className={`absolute inset-0 ring-2 ring-primary ring-offset-1 rounded-sm`}></div>
            )}
            
            {/* Hover overlay */}
            <AnimatePresence>
              {hoveredPhoto === photo && (
                <motion.div
                  initial={{ opacity: 0 }}
                  animate={{ opacity: 1 }}
                  exit={{ opacity: 0 }}
                  className="absolute inset-0 bg-black/40 flex items-center justify-center"
                >
                  <div className={`flex ${size === 'large' ? 'gap-3' : 'gap-1'}`}>
                    <Button
                      size="sm"
                      variant="secondary"
                      className={`${sizes.container} p-0`}
                      onClick={(e) => openLightbox(photo, e)}
                    >
                      <ZoomIn className={sizes.icon} />
                    </Button>
                    <Button
                      size="sm"
                      variant="secondary"
                      className={`${sizes.container} p-0`}
                      onClick={(e) => onPhotoLike(photo, e)}
                    >
                      <Heart 
                        className={`${sizes.icon} ${
                          photo && photoLikes && photoLikes[photo]?.isLiked ? "fill-red-500 text-red-500" : ""
                        }`} 
                      />
                    </Button>
                  </div>
                </motion.div>
              )}
            </AnimatePresence>
            

            
            {/* Like count */}
            {photo && photoLikes && (photoLikes[photo]?.likes || 0) > 0 && (
              <div className={`absolute bottom-1 left-1 bg-black/70 text-white text-xs ${size === 'large' ? 'px-2 py-1' : 'px-1 py-0.5'} rounded-full flex items-center gap-1`}>
                <Heart className={size === 'large' ? 'w-3 h-3' : 'w-2 h-2'} />
                {photoLikes[photo]?.likes || 0}
              </div>
            )}
          </>
        )}
      </motion.div>
    );
  };

  return (
    <>
      <div className="w-full px-0">
        <div className="flex gap-3 h-[450px]">
          {/* Hero Photo Section - Position 1 */}
          <div className="flex-1 max-w-sm">
            {renderPhotoBlock(0, 'large', 'h-full', '', true)}
          </div>

          {/* Dynamic Grid Section - 8 photos in varied sizes */}
          <div className="flex-1 flex flex-col gap-2 h-full">
            {/* Top Section - Mixed Heights */}
            <div className="flex gap-2 h-[65%]">
              {/* Left Side - 2 medium photos stacked */}
              <div className="flex flex-col gap-2" style={{ flex: '1' }}>
                {renderPhotoBlock(1, 'medium', 'h-[58%]', 'aspect-4/3')}
                {renderPhotoBlock(2, 'medium', 'h-[42%]', 'aspect-4/3')}
              </div>
              
              {/* Center - Photo 4 aligned with photo 8 below */}
              <div className="flex flex-col gap-2" style={{ flex: '1' }}>
                {renderPhotoBlock(3, 'medium', 'h-full', 'aspect-3/4')}
              </div>
              
              {/* Right Side - 2 photos stacked */}
              <div className="flex flex-col gap-2" style={{ flex: '1' }}>
                {renderPhotoBlock(4, 'medium', 'h-[42%]', 'aspect-4/3')}
                {renderPhotoBlock(5, 'medium', 'h-[58%]', 'aspect-4/3')}
              </div>
            </div>

            {/* Bottom Section - 3 small photos in a row, with middle one aligned to photo 4 */}
            <div className="flex gap-2 h-[35%]">
              <div style={{ flex: '1' }}>
                {renderPhotoBlock(6, 'small', 'h-full', 'aspect-1')}
              </div>
              <div style={{ flex: '1' }}>
                {renderPhotoBlock(7, 'small', 'h-full', 'aspect-1')}
              </div>
              <div style={{ flex: '1' }}>
                {renderPhotoBlock(8, 'small', 'h-full', 'aspect-1')}
              </div>
            </div>
          </div>
        </div>
      </div>

      {/* Lightbox Modal */}
      <AnimatePresence>
        {lightboxOpen && (
          <motion.div
            initial={{ opacity: 0 }}
            animate={{ opacity: 1 }}
            exit={{ opacity: 0 }}
            className="fixed inset-0 bg-black/90 z-50 flex items-center justify-center p-4"
            onClick={() => setLightboxOpen(false)}
          >
            <motion.div
              initial={{ scale: 0.8, opacity: 0 }}
              animate={{ scale: 1, opacity: 1 }}
              exit={{ scale: 0.8, opacity: 0 }}
              className="relative max-w-4xl max-h-full"
              onClick={(e) => e.stopPropagation()}
            >
              <ImageWithFallback
                src={lightboxPhoto}
                alt="Lightbox photo"
                className="w-full h-full object-contain rounded-sm"
              />
              <Button
                size="sm"
                variant="secondary"
                className="absolute top-4 right-4"
                onClick={() => setLightboxOpen(false)}
              >
                ×
              </Button>
            </motion.div>
          </motion.div>
        )}
      </AnimatePresence>
    </>
  );
};