import { useState } from 'react';
import { Card } from './ui/card';
import { Button } from './ui/button';
import { Label } from './ui/label';
import { Switch } from './ui/switch';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from './ui/select';
import { Slider } from './ui/slider';
import { Alert, AlertDescription } from './ui/alert';
import { Separator } from './ui/separator';
import { 
  Bell, 
  BellOff, 
  Smartphone, 
  Mail, 
  MessageCircle, 
  Heart,
  Eye,
  Star,
  Users,
  Calendar,
  Volume2,
  VolumeX,
  Settings,
  Info,
  Clock,
  Zap
} from 'lucide-react';

export function NotificationSettingsTab() {
  // Email Notifications
  const [emailEnabled, setEmailEnabled] = useState(true);
  const [emailMatches, setEmailMatches] = useState(true);
  const [emailMessages, setEmailMessages] = useState(true);
  const [emailLikes, setEmailLikes] = useState(false);
  const [emailViews, setEmailViews] = useState(false);
  const [emailPromotions, setEmailPromotions] = useState(false);
  const [emailWeeklyDigest, setEmailWeeklyDigest] = useState(true);

  // Push Notifications
  const [pushEnabled, setPushEnabled] = useState(true);
  const [pushMatches, setPushMatches] = useState(true);
  const [pushMessages, setPushMessages] = useState(true);
  const [pushLikes, setPushLikes] = useState(true);
  const [pushViews, setPushViews] = useState(false);
  const [pushEvents, setPushEvents] = useState(true);

  // In-App Notifications
  const [inAppSounds, setInAppSounds] = useState(true);
  const [inAppPopups, setInAppPopups] = useState(true);
  const [soundVolume, setSoundVolume] = useState([75]);

  // Timing Settings
  const [quietHours, setQuietHours] = useState(true);
  const [quietStart, setQuietStart] = useState('22:00');
  const [quietEnd, setQuietEnd] = useState('08:00');
  const [frequency, setFrequency] = useState('instant');

  const handleSaveNotificationSettings = () => {
    // In a real app, this would save to the backend
    console.log('Saving notification settings:', {
      email: {
        enabled: emailEnabled,
        matches: emailMatches,
        messages: emailMessages,
        likes: emailLikes,
        views: emailViews,
        promotions: emailPromotions,
        weeklyDigest: emailWeeklyDigest
      },
      push: {
        enabled: pushEnabled,
        matches: pushMatches,
        messages: pushMessages,
        likes: pushLikes,
        views: pushViews,
        events: pushEvents
      },
      inApp: {
        sounds: inAppSounds,
        popups: inAppPopups,
        volume: soundVolume[0]
      },
      timing: {
        quietHours,
        quietStart,
        quietEnd,
        frequency
      }
    });
    // Show success toast
  };

  const testNotification = () => {
    // Simulate sending a test notification
    if ('Notification' in window && Notification.permission === 'granted') {
      new Notification('SinglesZonne Test', {
        body: 'This is how your notifications will appear!',
        icon: '/favicon.ico'
      });
    } else {
      alert('Test notification sent! Check your notification settings.');
    }
  };

  const requestNotificationPermission = async () => {
    if ('Notification' in window) {
      const permission = await Notification.requestPermission();
      if (permission === 'granted') {
        setPushEnabled(true);
      }
    }
  };

  return (
    <div className="space-y-6">
      {/* Header */}
      <div className="flex items-center justify-between">
        <div className="flex items-center gap-3">
          <div className="p-2 rounded-full bg-gradient-to-r from-blue-500 to-purple-600">
            <Bell className="w-6 h-6 text-white" />
          </div>
          <div>
            <h2 className="text-2xl font-bold">Notification Settings</h2>
            <p className="text-muted-foreground">Customize how and when you receive notifications</p>
          </div>
        </div>
        
        <div className="flex gap-2">
          <Button variant="outline" onClick={testNotification} className="flex items-center gap-2">
            <Zap className="w-4 h-4" />
            Test Notifications
          </Button>
          <Button onClick={handleSaveNotificationSettings} className="flex items-center gap-2">
            <Bell className="w-4 h-4" />
            Save Settings
          </Button>
        </div>
      </div>

      {/* Notification Permission Alert */}
      {!pushEnabled && (
        <Alert>
          <Bell className="h-4 w-4" />
          <AlertDescription className="flex items-center justify-between">
            <span>
              <strong>Enable Push Notifications:</strong> Get instant alerts for matches, messages, and more.
            </span>
            <Button 
              variant="outline" 
              size="sm" 
              onClick={requestNotificationPermission}
              className="ml-4"
            >
              Enable Now
            </Button>
          </AlertDescription>
        </Alert>
      )}

      {/* Quick Settings */}
      <Card className="p-6">
        <h3 className="text-lg font-semibold mb-4 flex items-center gap-2">
          <Settings className="w-5 h-5" />
          Quick Settings
        </h3>
        
        <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
          <Button 
            variant={emailEnabled && pushEnabled ? "default" : "outline"}
            onClick={() => {
              setEmailEnabled(true);
              setPushEnabled(true);
              setEmailMatches(true);
              setPushMatches(true);
              setEmailMessages(true);
              setPushMessages(true);
            }}
            className="flex items-center gap-2 h-auto py-4"
          >
            <Bell className="w-5 h-5" />
            <div className="text-left">
              <div className="font-medium">All Notifications</div>
              <div className="text-xs opacity-75">Get notified about everything</div>
            </div>
          </Button>
          
          <Button 
            variant="outline"
            onClick={() => {
              setEmailEnabled(true);
              setPushEnabled(true);
              setEmailMatches(true);
              setPushMatches(true);
              setEmailMessages(true);
              setPushMessages(true);
              setEmailLikes(false);
              setPushLikes(false);
              setEmailViews(false);
              setPushViews(false);
            }}
            className="flex items-center gap-2 h-auto py-4"
          >
            <MessageCircle className="w-5 h-5" />
            <div className="text-left">
              <div className="font-medium">Essential Only</div>
              <div className="text-xs opacity-75">Matches and messages only</div>
            </div>
          </Button>
          
          <Button 
            variant={!emailEnabled && !pushEnabled ? "destructive" : "outline"}
            onClick={() => {
              setEmailEnabled(false);
              setPushEnabled(false);
              setEmailMatches(false);
              setPushMatches(false);
              setEmailMessages(false);
              setPushMessages(false);
              setEmailLikes(false);
              setPushLikes(false);
            }}
            className="flex items-center gap-2 h-auto py-4"
          >
            <BellOff className="w-5 h-5" />
            <div className="text-left">
              <div className="font-medium">Turn Off All</div>
              <div className="text-xs opacity-75">Disable all notifications</div>
            </div>
          </Button>
        </div>
      </Card>

      {/* Push Notifications */}
      <Card className="p-6">
        <div className="flex items-center justify-between mb-4">
          <h3 className="text-lg font-semibold flex items-center gap-2">
            <Smartphone className="w-5 h-5" />
            Push Notifications
          </h3>
          <Switch 
            checked={pushEnabled}
            onCheckedChange={setPushEnabled}
          />
        </div>
        
        {pushEnabled && (
          <div className="space-y-4 pl-7">
            <div className="flex items-center justify-between">
              <div className="flex items-center gap-2">
                <Heart className="w-4 h-4 text-pink-500" />
                <div>
                  <Label>New matches</Label>
                  <p className="text-sm text-muted-foreground">When someone matches with you</p>
                </div>
              </div>
              <Switch 
                checked={pushMatches}
                onCheckedChange={setPushMatches}
              />
            </div>

            <div className="flex items-center justify-between">
              <div className="flex items-center gap-2">
                <MessageCircle className="w-4 h-4 text-blue-500" />
                <div>
                  <Label>New messages</Label>
                  <p className="text-sm text-muted-foreground">When you receive a message</p>
                </div>
              </div>
              <Switch 
                checked={pushMessages}
                onCheckedChange={setPushMessages}
              />
            </div>

            <div className="flex items-center justify-between">
              <div className="flex items-center gap-2">
                <Star className="w-4 h-4 text-yellow-500" />
                <div>
                  <Label>Profile likes</Label>
                  <p className="text-sm text-muted-foreground">When someone likes your profile</p>
                </div>
              </div>
              <Switch 
                checked={pushLikes}
                onCheckedChange={setPushLikes}
              />
            </div>

            <div className="flex items-center justify-between">
              <div className="flex items-center gap-2">
                <Eye className="w-4 h-4 text-green-500" />
                <div>
                  <Label>Profile views</Label>
                  <p className="text-sm text-muted-foreground">When someone views your profile</p>
                </div>
              </div>
              <Switch 
                checked={pushViews}
                onCheckedChange={setPushViews}
              />
            </div>

            <div className="flex items-center justify-between">
              <div className="flex items-center gap-2">
                <Calendar className="w-4 h-4 text-purple-500" />
                <div>
                  <Label>Events & groups</Label>
                  <p className="text-sm text-muted-foreground">Updates about events and groups you joined</p>
                </div>
              </div>
              <Switch 
                checked={pushEvents}
                onCheckedChange={setPushEvents}
              />
            </div>
          </div>
        )}
      </Card>

      {/* Email Notifications */}
      <Card className="p-6">
        <div className="flex items-center justify-between mb-4">
          <h3 className="text-lg font-semibold flex items-center gap-2">
            <Mail className="w-5 h-5" />
            Email Notifications
          </h3>
          <Switch 
            checked={emailEnabled}
            onCheckedChange={setEmailEnabled}
          />
        </div>
        
        {emailEnabled && (
          <div className="space-y-4 pl-7">
            <div className="flex items-center justify-between">
              <div className="flex items-center gap-2">
                <Heart className="w-4 h-4 text-pink-500" />
                <div>
                  <Label>Match notifications</Label>
                  <p className="text-sm text-muted-foreground">Daily summary of new matches</p>
                </div>
              </div>
              <Switch 
                checked={emailMatches}
                onCheckedChange={setEmailMatches}
              />
            </div>

            <div className="flex items-center justify-between">
              <div className="flex items-center gap-2">
                <MessageCircle className="w-4 h-4 text-blue-500" />
                <div>
                  <Label>Message notifications</Label>
                  <p className="text-sm text-muted-foreground">When you have unread messages</p>
                </div>
              </div>
              <Switch 
                checked={emailMessages}
                onCheckedChange={setEmailMessages}
              />
            </div>

            <div className="flex items-center justify-between">
              <div className="flex items-center gap-2">
                <Star className="w-4 h-4 text-yellow-500" />
                <div>
                  <Label>Like notifications</Label>
                  <p className="text-sm text-muted-foreground">Weekly summary of profile likes</p>
                </div>
              </div>
              <Switch 
                checked={emailLikes}
                onCheckedChange={setEmailLikes}
              />
            </div>

            <div className="flex items-center justify-between">
              <div className="flex items-center gap-2">
                <Eye className="w-4 h-4 text-green-500" />
                <div>
                  <Label>View notifications</Label>
                  <p className="text-sm text-muted-foreground">Weekly summary of profile views</p>
                </div>
              </div>
              <Switch 
                checked={emailViews}
                onCheckedChange={setEmailViews}
              />
            </div>

            <Separator />

            <div className="flex items-center justify-between">
              <div className="flex items-center gap-2">
                <Users className="w-4 h-4 text-indigo-500" />
                <div>
                  <Label>Weekly digest</Label>
                  <p className="text-sm text-muted-foreground">Summary of your activity and suggestions</p>
                </div>
              </div>
              <Switch 
                checked={emailWeeklyDigest}
                onCheckedChange={setEmailWeeklyDigest}
              />
            </div>

            <div className="flex items-center justify-between">
              <div className="flex items-center gap-2">
                <Zap className="w-4 h-4 text-orange-500" />
                <div>
                  <Label>Promotions & updates</Label>
                  <p className="text-sm text-muted-foreground">Special offers and app updates</p>
                </div>
              </div>
              <Switch 
                checked={emailPromotions}
                onCheckedChange={setEmailPromotions}
              />
            </div>
          </div>
        )}
      </Card>

      {/* In-App Settings */}
      <Card className="p-6">
        <h3 className="text-lg font-semibold mb-4 flex items-center gap-2">
          <Volume2 className="w-5 h-5" />
          In-App Settings
        </h3>
        
        <div className="space-y-4">
          <div className="flex items-center justify-between">
            <div>
              <Label htmlFor="sounds">Notification sounds</Label>
              <p className="text-sm text-muted-foreground">Play sounds for new notifications</p>
            </div>
            <Switch 
              id="sounds"
              checked={inAppSounds}
              onCheckedChange={setInAppSounds}
            />
          </div>

          {inAppSounds && (
            <div className="space-y-2 ml-6">
              <Label className="flex items-center gap-2">
                <Volume2 className="w-4 h-4" />
                Sound Volume: {soundVolume[0]}%
              </Label>
              <Slider
                value={soundVolume}
                onValueChange={setSoundVolume}
                max={100}
                step={1}
                className="w-full"
              />
            </div>
          )}

          <div className="flex items-center justify-between">
            <div>
              <Label htmlFor="popups">In-app popups</Label>
              <p className="text-sm text-muted-foreground">Show notification popups while using the app</p>
            </div>
            <Switch 
              id="popups"
              checked={inAppPopups}
              onCheckedChange={setInAppPopups}
            />
          </div>
        </div>
      </Card>

      {/* Timing & Frequency */}
      <Card className="p-6">
        <h3 className="text-lg font-semibold mb-4 flex items-center gap-2">
          <Clock className="w-5 h-5" />
          Timing & Frequency
        </h3>
        
        <div className="space-y-4">
          <div className="space-y-2">
            <Label>Notification frequency</Label>
            <Select value={frequency} onValueChange={setFrequency}>
              <SelectTrigger>
                <SelectValue placeholder="Select frequency" />
              </SelectTrigger>
              <SelectContent>
                <SelectItem value="instant">Instant</SelectItem>
                <SelectItem value="hourly">Hourly digest</SelectItem>
                <SelectItem value="daily">Daily digest</SelectItem>
                <SelectItem value="weekly">Weekly digest</SelectItem>
              </SelectContent>
            </Select>
          </div>

          <Separator />

          <div className="flex items-center justify-between">
            <div>
              <Label htmlFor="quiet-hours">Enable quiet hours</Label>
              <p className="text-sm text-muted-foreground">Pause notifications during specified hours</p>
            </div>
            <Switch 
              id="quiet-hours"
              checked={quietHours}
              onCheckedChange={setQuietHours}
            />
          </div>

          {quietHours && (
            <div className="grid grid-cols-2 gap-4 ml-6">
              <div className="space-y-2">
                <Label htmlFor="quiet-start">Start time</Label>
                <input
                  id="quiet-start"
                  type="time"
                  value={quietStart}
                  onChange={(e) => setQuietStart(e.target.value)}
                  className="w-full px-3 py-2 border border-input rounded-md bg-background"
                />
              </div>
              <div className="space-y-2">
                <Label htmlFor="quiet-end">End time</Label>
                <input
                  id="quiet-end"
                  type="time"
                  value={quietEnd}
                  onChange={(e) => setQuietEnd(e.target.value)}
                  className="w-full px-3 py-2 border border-input rounded-md bg-background"
                />
              </div>
            </div>
          )}
        </div>
      </Card>

      {/* Notification Tips */}
      <Card className="p-6 bg-gradient-to-br from-blue-50 to-indigo-50 dark:from-blue-950/20 dark:to-indigo-950/20">
        <h4 className="font-semibold mb-3 text-blue-900 dark:text-blue-100 flex items-center gap-2">
          <Bell className="w-5 h-5" />
          📱 Notification Tips
        </h4>
        <div className="space-y-2 text-sm text-blue-800 dark:text-blue-200">
          <div className="flex items-start gap-2">
            <Info className="w-4 h-4 mt-0.5 text-blue-600" />
            <span><strong>Stay connected:</strong> Enable match and message notifications to never miss opportunities</span>
          </div>
          <div className="flex items-start gap-2">
            <Info className="w-4 h-4 mt-0.5 text-blue-600" />
            <span><strong>Reduce overwhelm:</strong> Use quiet hours and digest options to control notification timing</span>
          </div>
          <div className="flex items-start gap-2">
            <Info className="w-4 h-4 mt-0.5 text-blue-600" />
            <span><strong>Test settings:</strong> Use the test notification button to see how your settings feel</span>
          </div>
        </div>
      </Card>
    </div>
  );
}