import { useState } from "react";
import { Header } from "./Header";
import { Footer } from "./Footer";
import { Button } from "./ui/button";
import { Card } from "./ui/card";
import { Badge } from "./ui/badge";
import { Input } from "./ui/input";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "./ui/tabs";
import { Textarea } from "./ui/textarea";
import { Avatar, AvatarFallback, AvatarImage } from "./ui/avatar";
import { 
  MessageCircle, 
  Search, 
  Send, 
  Heart, 
  Star, 
  MoreVertical,
  Phone,
  Video,
  Smile,
  Paperclip,
  Circle,
  Clock
} from "lucide-react";
import { motion } from "motion/react";
import { ImageWithFallback } from "./figma/ImageWithFallback";

interface MessagesPageProps {
  onNavigate: (view: string) => void;
  onOpenCookieConsent?: () => void;
}

interface Message {
  id: number;
  senderId: number;
  content: string;
  timestamp: string;
  isRead: boolean;
}

interface Conversation {
  id: number;
  name: string;
  avatar: string;
  lastMessage: string;
  timestamp: string;
  unreadCount: number;
  isOnline: boolean;
  lastSeen?: string;
  matchPercentage?: number;
  type: "match" | "like" | "conversation";
}

export function MessagesPage({ onNavigate, onOpenCookieConsent }: MessagesPageProps) {
  const [activeTab, setActiveTab] = useState("all");
  const [selectedConversation, setSelectedConversation] = useState<number | null>(1);
  const [searchQuery, setSearchQuery] = useState("");
  const [newMessage, setNewMessage] = useState("");
  
  const [conversations] = useState<Conversation[]>([
    {
      id: 1,
      name: "Emma",
      avatar: "https://images.unsplash.com/photo-1494790108755-2616b612b8c0?w=100&h=100&fit=crop&crop=face",
      lastMessage: "That sounds like a great plan! I'd love to join you for coffee this weekend 😊",
      timestamp: "2 min ago",
      unreadCount: 2,
      isOnline: true,
      matchPercentage: 95,
      type: "match"
    },
    {
      id: 2,
      name: "Sophia",
      avatar: "https://images.unsplash.com/photo-1438761681033-6461ffad8d80?w=100&h=100&fit=crop&crop=face",
      lastMessage: "Thanks for the like! Your profile looks amazing 💕",
      timestamp: "1 hour ago",
      unreadCount: 1,
      isOnline: false,
      lastSeen: "30 min ago",
      matchPercentage: 89,
      type: "like"
    },
    {
      id: 3,
      name: "Isabella",
      avatar: "https://images.unsplash.com/photo-1544005313-94ddf0286df2?w=100&h=100&fit=crop&crop=face",
      lastMessage: "Hey! How was your day?",
      timestamp: "3 hours ago",
      unreadCount: 0,
      isOnline: true,
      matchPercentage: 92,
      type: "conversation"
    },
    {
      id: 4,
      name: "Olivia",
      avatar: "https://images.unsplash.com/photo-1534528741775-53994a69daeb?w=100&h=100&fit=crop&crop=face",
      lastMessage: "I saw your hiking photos, they're incredible! Where was that taken?",
      timestamp: "1 day ago",
      unreadCount: 0,
      isOnline: false,
      lastSeen: "2 hours ago",
      matchPercentage: 87,
      type: "conversation"
    },
    {
      id: 5,
      name: "Ava",
      avatar: "https://images.unsplash.com/photo-1517841905240-472988babdf9?w=100&h=100&fit=crop&crop=face",
      lastMessage: "You liked my photo!",
      timestamp: "2 days ago",
      unreadCount: 0,
      isOnline: false,
      lastSeen: "1 day ago",
      type: "like"
    }
  ]);

  const [messages] = useState<Message[]>([
    {
      id: 1,
      senderId: 1,
      content: "Hi! I noticed we both love photography. Your travel photos are stunning! 📸",
      timestamp: "10:30 AM",
      isRead: true
    },
    {
      id: 2,
      senderId: 0, // Current user
      content: "Thank you so much! I saw you're into hiking too. Have you been to any good spots lately?",
      timestamp: "10:45 AM",
      isRead: true
    },
    {
      id: 3,
      senderId: 1,
      content: "Yes! I just got back from a amazing trail in the mountains. The sunrise was incredible. I actually got some great shots there.",
      timestamp: "11:00 AM",
      isRead: true
    },
    {
      id: 4,
      senderId: 0,
      content: "That sounds amazing! I'd love to hear more about it. Maybe we could grab coffee and you could show me some of those photos?",
      timestamp: "11:15 AM",
      isRead: true
    },
    {
      id: 5,
      senderId: 1,
      content: "That sounds like a great plan! I'd love to join you for coffee this weekend 😊",
      timestamp: "Just now",
      isRead: false
    }
  ]);

  const filteredConversations = conversations.filter(conv => {
    const matchesSearch = conv.name.toLowerCase().includes(searchQuery.toLowerCase());
    const matchesTab = activeTab === "all" || 
      (activeTab === "matches" && conv.type === "match") ||
      (activeTab === "likes" && conv.type === "like") ||
      (activeTab === "unread" && conv.unreadCount > 0);
    return matchesSearch && matchesTab;
  });

  const handleSendMessage = () => {
    if (newMessage.trim()) {
      console.log("Sending message:", newMessage);
      setNewMessage("");
      // In a real app, this would send the message to the backend
    }
  };

  const handleKeyPress = (e: React.KeyboardEvent) => {
    if (e.key === "Enter" && !e.shiftKey) {
      e.preventDefault();
      handleSendMessage();
    }
  };

  const selectedConv = conversations.find(c => c.id === selectedConversation);

  return (
    <div className="min-h-screen bg-background">
      {/* Header */}
      <Header currentView="messages" onNavigate={onNavigate} />
      
      {/* Messages Banner */}
      <div className="bg-gradient-to-r from-blue-600 via-blue-700 to-blue-800 dark:from-blue-700 dark:via-blue-800 dark:to-blue-900 text-white py-11 relative overflow-hidden">
        {/* Moving gradient overlay */}
        <motion.div 
          className="absolute inset-0 bg-gradient-to-r from-transparent via-blue-300/30 to-transparent opacity-40"
          animate={{
            x: ['-100%', '100%']
          }}
          transition={{
            duration: 8,
            repeat: Infinity,
            ease: "linear",
            repeatDelay: 2
          }}
          style={{
            width: '200%',
            background: 'linear-gradient(90deg, transparent 0%, rgba(255,255,255,0.15) 25%, rgba(255,255,255,0.3) 50%, rgba(255,255,255,0.15) 75%, transparent 100%)'
          }}
        />
        
        {/* Additional gradient overlay */}
        <div className="absolute inset-0 bg-gradient-to-r from-transparent via-blue-600/20 to-blue-800/40 dark:from-transparent dark:via-blue-700/20 dark:to-blue-900/40"></div>
        
        <div className="max-w-7xl mx-auto px-6 text-center relative z-10">
          <motion.h1 
            className="text-4xl font-bold mb-4"
            initial={{ opacity: 0, y: 20 }}
            animate={{ opacity: 1, y: 0 }}
            transition={{ 
              duration: 0.8,
              ease: "easeOut"
            }}
          >
            <motion.span
              animate={{ 
                scale: [1, 1.05, 1],
              }}
              transition={{
                duration: 3,
                repeat: Infinity,
                ease: "easeInOut",
                repeatDelay: 0.5
              }}
            >
              💬 Messages
            </motion.span>
          </motion.h1>
          <motion.p 
            className="text-lg text-blue-100 dark:text-blue-200"
            initial={{ opacity: 0, y: 20 }}
            animate={{ opacity: 1, y: 0 }}
            transition={{ 
              duration: 0.8,
              delay: 0.2,
              ease: "easeOut"
            }}
          >
            Start meaningful conversations with your matches
          </motion.p>
        </div>
      </div>

      {/* Main Content */}
      <div className="max-w-7xl mx-auto px-6 py-8">
        <div className="grid grid-cols-1 lg:grid-cols-12 gap-6 h-[600px]">
          {/* Conversations Sidebar */}
          <div className="lg:col-span-4 flex flex-col">
            <Card className="flex-1 p-0">
              {/* Search and Tabs */}
              <div className="p-4 border-b">
                <div className="relative mb-4">
                  <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-muted-foreground w-4 h-4" />
                  <Input
                    placeholder="Search conversations..."
                    value={searchQuery}
                    onChange={(e) => setSearchQuery(e.target.value)}
                    className="pl-10"
                  />
                </div>
                
                <Tabs value={activeTab} onValueChange={setActiveTab} className="w-full">
                  <TabsList className="grid grid-cols-4 w-full h-auto p-1">
                    <TabsTrigger value="all" className="text-xs px-2 py-1">All</TabsTrigger>
                    <TabsTrigger value="matches" className="text-xs px-2 py-1">Matches</TabsTrigger>
                    <TabsTrigger value="likes" className="text-xs px-2 py-1">Likes</TabsTrigger>
                    <TabsTrigger value="unread" className="text-xs px-2 py-1">Unread</TabsTrigger>
                  </TabsList>
                </Tabs>
              </div>

              {/* Conversations List */}
              <div className="flex-1 overflow-y-auto">
                {filteredConversations.map((conversation) => (
                  <motion.div
                    key={conversation.id}
                    initial={{ opacity: 0, x: -20 }}
                    animate={{ opacity: 1, x: 0 }}
                    transition={{ duration: 0.3 }}
                    className={`p-4 border-b cursor-pointer transition-colors hover:bg-accent ${
                      selectedConversation === conversation.id ? 'bg-accent' : ''
                    }`}
                    onClick={() => setSelectedConversation(conversation.id)}
                  >
                    <div className="flex items-start gap-3">
                      <div className="relative">
                        <Avatar className="w-12 h-12">
                          <AvatarImage src={conversation.avatar} />
                          <AvatarFallback>{conversation.name[0]}</AvatarFallback>
                        </Avatar>
                        {conversation.isOnline && (
                          <Circle className="absolute -bottom-1 -right-1 w-4 h-4 text-green-500 fill-green-500" />
                        )}
                      </div>
                      
                      <div className="flex-1 min-w-0">
                        <div className="flex items-center justify-between mb-1">
                          <h4 className="font-medium truncate">{conversation.name}</h4>
                          <div className="flex items-center gap-1">
                            {conversation.matchPercentage && (
                              <Badge variant="secondary" className="text-xs">
                                {conversation.matchPercentage}%
                              </Badge>
                            )}
                            {conversation.type === "match" && <Heart className="w-3 h-3 text-red-500" />}
                            {conversation.type === "like" && <Star className="w-3 h-3 text-yellow-500" />}
                          </div>
                        </div>
                        
                        <p className="text-sm text-muted-foreground truncate mb-1">
                          {conversation.lastMessage}
                        </p>
                        
                        <div className="flex items-center justify-between">
                          <span className="text-xs text-muted-foreground">
                            {conversation.isOnline ? (
                              <span className="text-green-600">Online</span>
                            ) : (
                              conversation.lastSeen && `Active ${conversation.lastSeen}`
                            )}
                          </span>
                          
                          <div className="flex items-center gap-2">
                            <span className="text-xs text-muted-foreground">{conversation.timestamp}</span>
                            {conversation.unreadCount > 0 && (
                              <Badge className="bg-blue-600 text-white text-xs min-w-[20px] h-5 flex items-center justify-center rounded-full">
                                {conversation.unreadCount}
                              </Badge>
                            )}
                          </div>
                        </div>
                      </div>
                    </div>
                  </motion.div>
                ))}
              </div>
            </Card>
          </div>

          {/* Chat Area */}
          <div className="lg:col-span-8 flex flex-col">
            {selectedConv ? (
              <Card className="flex-1 flex flex-col p-0">
                {/* Chat Header */}
                <div className="p-4 border-b flex items-center justify-between">
                  <div className="flex items-center gap-3">
                    <div className="relative">
                      <Avatar className="w-10 h-10">
                        <AvatarImage src={selectedConv.avatar} />
                        <AvatarFallback>{selectedConv.name[0]}</AvatarFallback>
                      </Avatar>
                      {selectedConv.isOnline && (
                        <Circle className="absolute -bottom-1 -right-1 w-3 h-3 text-green-500 fill-green-500" />
                      )}
                    </div>
                    <div>
                      <h3 className="font-medium">{selectedConv.name}</h3>
                      <p className="text-sm text-muted-foreground">
                        {selectedConv.isOnline ? "Online now" : `Active ${selectedConv.lastSeen}`}
                      </p>
                    </div>
                  </div>
                  
                  <div className="flex items-center gap-2">
                    <Button variant="outline" size="sm">
                      <Phone className="w-4 h-4" />
                    </Button>
                    <Button variant="outline" size="sm">
                      <Video className="w-4 h-4" />
                    </Button>
                    <Button variant="outline" size="sm">
                      <MoreVertical className="w-4 h-4" />
                    </Button>
                  </div>
                </div>

                {/* Messages */}
                <div className="flex-1 p-4 overflow-y-auto space-y-4">
                  {messages.map((message) => (
                    <motion.div
                      key={message.id}
                      initial={{ opacity: 0, y: 10 }}
                      animate={{ opacity: 1, y: 0 }}
                      transition={{ duration: 0.3 }}
                      className={`flex ${message.senderId === 0 ? 'justify-end' : 'justify-start'}`}
                    >
                      <div className={`max-w-[70%] p-3 rounded-lg ${
                        message.senderId === 0 
                          ? 'bg-blue-600 text-white' 
                          : 'bg-muted'
                      }`}>
                        <p className="text-sm">{message.content}</p>
                        <div className={`flex items-center gap-1 mt-1 ${
                          message.senderId === 0 ? 'justify-end' : 'justify-start'
                        }`}>
                          <span className={`text-xs ${
                            message.senderId === 0 ? 'text-blue-100' : 'text-muted-foreground'
                          }`}>
                            {message.timestamp}
                          </span>
                          {message.senderId === 0 && (
                            <div className="w-2 h-2 rounded-full bg-blue-200" />
                          )}
                        </div>
                      </div>
                    </motion.div>
                  ))}
                </div>

                {/* Message Input */}
                <div className="p-4 border-t">
                  <div className="flex items-end gap-2">
                    <Button variant="outline" size="sm">
                      <Paperclip className="w-4 h-4" />
                    </Button>
                    <div className="flex-1">
                      <Textarea
                        placeholder="Type your message..."
                        value={newMessage}
                        onChange={(e) => setNewMessage(e.target.value)}
                        onKeyPress={handleKeyPress}
                        className="min-h-[40px] max-h-[120px] resize-none"
                        rows={1}
                      />
                    </div>
                    <Button variant="outline" size="sm">
                      <Smile className="w-4 h-4" />
                    </Button>
                    <Button 
                      onClick={handleSendMessage}
                      size="sm"
                      className="bg-blue-600 hover:bg-blue-700"
                      disabled={!newMessage.trim()}
                    >
                      <Send className="w-4 h-4" />
                    </Button>
                  </div>
                </div>
              </Card>
            ) : (
              <Card className="flex-1 flex items-center justify-center">
                <div className="text-center">
                  <MessageCircle className="w-16 h-16 text-muted-foreground mx-auto mb-4" />
                  <h3 className="text-xl font-semibold mb-2">Select a conversation</h3>
                  <p className="text-muted-foreground">Choose a conversation from the left to start messaging</p>
                </div>
              </Card>
            )}
          </div>
        </div>
      </div>
      
      {/* Footer */}
      <Footer onNavigate={onNavigate} onOpenCookieConsent={onOpenCookieConsent} />
    </div>
  );
}