import { useState } from "react";
import { Header } from "./Header";
import { Footer } from "./Footer";
import { Button } from "./ui/button";
import { Card } from "./ui/card";
import { Badge } from "./ui/badge";
import { Alert, AlertDescription } from "./ui/alert";
import { Progress } from "./ui/progress";
import { Separator } from "./ui/separator";
import { 
  Shield, 
  Upload, 
  FileImage, 
  CheckCircle, 
  Clock, 
  X,
  AlertTriangle,
  FileText,
  Camera,
  Verified,
  Lock,
  Eye,
  Trash2
} from "lucide-react";
import { useTranslation } from "./TranslationProvider";
import { ImageWithFallback } from "./figma/ImageWithFallback";

interface MemberVerificationPageProps {
  onNavigate: (view: string) => void;
  onOpenCookieConsent?: () => void;
}

interface VerificationDocument {
  id: string;
  type: 'drivers_license' | 'passport' | 'government_id';
  frontImage: string;
  backImage?: string;
  submittedAt: string;
  status: 'pending' | 'approved' | 'rejected';
  rejectionReason?: string;
}

export function MemberVerificationPage({ onNavigate, onOpenCookieConsent }: MemberVerificationPageProps) {
  const { t } = useTranslation();
  const [uploadStep, setUploadStep] = useState<'select' | 'upload' | 'review' | 'submitted'>('select');
  const [selectedDocType, setSelectedDocType] = useState<'drivers_license' | 'passport' | 'government_id' | null>(null);
  const [frontImage, setFrontImage] = useState<string | null>(null);
  const [backImage, setBackImage] = useState<string | null>(null);
  const [isUploading, setIsUploading] = useState(false);
  const [uploadProgress, setUploadProgress] = useState(0);

  // Mock user verification data
  const [userVerification, setUserVerification] = useState<{
    isVerified: boolean;
    currentRequest?: VerificationDocument;
    verificationHistory: VerificationDocument[];
  }>({
    isVerified: false,
    currentRequest: undefined,
    verificationHistory: [
      {
        id: '1',
        type: 'drivers_license',
        frontImage: 'https://images.unsplash.com/photo-1554224155-6726b3ff858f?crop=entropy&cs=tinysrgb&fit=max&fm=jpg&q=80&w=400',
        backImage: 'https://images.unsplash.com/photo-1554224155-8d04cb21cd6c?crop=entropy&cs=tinysrgb&fit=max&fm=jpg&q=80&w=400',
        submittedAt: '2024-01-15T10:30:00Z',
        status: 'rejected',
        rejectionReason: 'Document image was too blurry. Please upload a clearer image.'
      }
    ]
  });

  const documentTypes = [
    {
      type: 'drivers_license' as const,
      title: 'Driver\'s License',
      description: 'Valid government-issued driver\'s license',
      icon: <FileText className="w-8 h-8" />,
      requiresBack: true
    },
    {
      type: 'passport' as const,
      title: 'Passport',
      description: 'Valid passport from any country',
      icon: <FileImage className="w-8 h-8" />,
      requiresBack: false
    },
    {
      type: 'government_id' as const,
      title: 'Government ID',
      description: 'Valid government-issued identification card',
      icon: <Shield className="w-8 h-8" />,
      requiresBack: true
    }
  ];

  const handleFileUpload = (file: File, isBack: boolean = false) => {
    if (file && file.type.startsWith('image/')) {
      const reader = new FileReader();
      reader.onload = (e) => {
        const result = e.target?.result as string;
        if (isBack) {
          setBackImage(result);
        } else {
          setFrontImage(result);
        }
      };
      reader.readAsDataURL(file);
    }
  };

  const handleSubmitVerification = async () => {
    if (!selectedDocType || !frontImage) return;

    setIsUploading(true);
    setUploadProgress(0);

    // Simulate upload progress
    const progressInterval = setInterval(() => {
      setUploadProgress((prev) => {
        if (prev >= 100) {
          clearInterval(progressInterval);
          return 100;
        }
        return prev + 10;
      });
    }, 200);

    // Simulate API call
    setTimeout(() => {
      const newRequest: VerificationDocument = {
        id: Date.now().toString(),
        type: selectedDocType,
        frontImage: frontImage,
        backImage: backImage || undefined,
        submittedAt: new Date().toISOString(),
        status: 'pending'
      };

      setUserVerification(prev => ({
        ...prev,
        currentRequest: newRequest,
        verificationHistory: [newRequest, ...prev.verificationHistory]
      }));

      setIsUploading(false);
      setUploadStep('submitted');
    }, 2000);
  };

  const resetUpload = () => {
    setUploadStep('select');
    setSelectedDocType(null);
    setFrontImage(null);
    setBackImage(null);
    setUploadProgress(0);
  };

  const getStatusColor = (status: string) => {
    switch (status) {
      case 'approved': return 'bg-green-500';
      case 'rejected': return 'bg-red-500';
      default: return 'bg-yellow-500';
    }
  };

  const getStatusIcon = (status: string) => {
    switch (status) {
      case 'approved': return <CheckCircle className="w-4 h-4" />;
      case 'rejected': return <X className="w-4 h-4" />;
      default: return <Clock className="w-4 h-4" />;
    }
  };

  return (
    <div className="min-h-screen bg-background">
      <Header currentView="verification" onNavigate={onNavigate} />
      
      <div className="w-full px-6 py-8">
        <div className="max-w-4xl mx-auto">
          {/* Header */}
          <div className="mb-8">
            <div className="flex items-center justify-between mb-4">
              <Button
                variant="ghost"
                onClick={() => onNavigate("profile-settings")}
                className="flex items-center gap-2"
              >
                <span>←</span>
                Back to Profile Settings
              </Button>
            </div>
            
            <div className="text-center mb-6">
              <div className="w-16 h-16 bg-blue-100 dark:bg-blue-900 rounded-full flex items-center justify-center mx-auto mb-4">
                <Shield className="w-8 h-8 text-blue-600 dark:text-blue-400" />
              </div>
              <h1 className="text-3xl font-bold mb-2">Account Verification</h1>
              <p className="text-muted-foreground max-w-2xl mx-auto">
                Verify your identity to gain trust and credibility on SinglesZonne. Verified members 
                get a special badge and increased visibility in search results.
              </p>
            </div>
          </div>

          {/* Current Verification Status */}
          {userVerification.isVerified ? (
            <Card className="p-6 mb-8 border-green-200 bg-green-50 dark:bg-green-900/20">
              <div className="flex items-center gap-4">
                <div className="w-12 h-12 bg-green-100 dark:bg-green-800 rounded-full flex items-center justify-center">
                  <Verified className="w-6 h-6 text-green-600 dark:text-green-400" />
                </div>
                <div className="flex-1">
                  <h3 className="font-semibold text-green-800 dark:text-green-200">Account Verified</h3>
                  <p className="text-sm text-green-600 dark:text-green-300">
                    Your account has been successfully verified. You now have a verified badge on your profile.
                  </p>
                </div>
                <Badge className="bg-green-500 text-white">
                  <Verified className="w-3 h-3 mr-1" />
                  Verified
                </Badge>
              </div>
            </Card>
          ) : userVerification.currentRequest ? (
            <Card className="p-6 mb-8 border-yellow-200 bg-yellow-50 dark:bg-yellow-900/20">
              <div className="flex items-center gap-4">
                <div className="w-12 h-12 bg-yellow-100 dark:bg-yellow-800 rounded-full flex items-center justify-center">
                  <Clock className="w-6 h-6 text-yellow-600 dark:text-yellow-400" />
                </div>
                <div className="flex-1">
                  <h3 className="font-semibold text-yellow-800 dark:text-yellow-200">Verification Pending</h3>
                  <p className="text-sm text-yellow-600 dark:text-yellow-300">
                    Your verification request is being reviewed. We'll notify you once it's processed.
                  </p>
                  <p className="text-xs text-yellow-500 dark:text-yellow-400 mt-1">
                    Submitted: {new Date(userVerification.currentRequest.submittedAt).toLocaleDateString()}
                  </p>
                </div>
                <Badge className="bg-yellow-500 text-white">
                  <Clock className="w-3 h-3 mr-1" />
                  Pending Review
                </Badge>
              </div>
            </Card>
          ) : null}

          {/* Verification Process */}
          {!userVerification.isVerified && !userVerification.currentRequest && (
            <div className="space-y-8">
              {/* Benefits of Verification */}
              <Card className="p-6">
                <h3 className="font-semibold mb-4 flex items-center gap-2">
                  <CheckCircle className="w-5 h-5 text-green-500" />
                  Benefits of Verification
                </h3>
                <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                  <div className="flex items-center gap-3">
                    <Verified className="w-5 h-5 text-blue-500" />
                    <span className="text-sm">Verified badge on your profile</span>
                  </div>
                  <div className="flex items-center gap-3">
                    <Eye className="w-5 h-5 text-purple-500" />
                    <span className="text-sm">Increased profile visibility</span>
                  </div>
                  <div className="flex items-center gap-3">
                    <Shield className="w-5 h-5 text-green-500" />
                    <span className="text-sm">Enhanced trust and credibility</span>
                  </div>
                  <div className="flex items-center gap-3">
                    <Lock className="w-5 h-5 text-blue-500" />
                    <span className="text-sm">Access to premium features</span>
                  </div>
                </div>
              </Card>

              {/* Document Type Selection */}
              {uploadStep === 'select' && (
                <Card className="p-6">
                  <h3 className="font-semibold mb-4">Select Document Type</h3>
                  <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
                    {documentTypes.map((doc) => (
                      <button
                        key={doc.type}
                        onClick={() => {
                          setSelectedDocType(doc.type);
                          setUploadStep('upload');
                        }}
                        className="p-4 border-2 border-dashed border-muted-foreground/20 rounded-lg hover:border-primary/50 transition-colors text-center"
                      >
                        <div className="flex flex-col items-center gap-3">
                          <div className="text-muted-foreground">{doc.icon}</div>
                          <div>
                            <h4 className="font-medium">{doc.title}</h4>
                            <p className="text-sm text-muted-foreground mt-1">{doc.description}</p>
                          </div>
                        </div>
                      </button>
                    ))}
                  </div>
                </Card>
              )}

              {/* Document Upload */}
              {uploadStep === 'upload' && selectedDocType && (
                <Card className="p-6">
                  <div className="flex items-center justify-between mb-6">
                    <h3 className="font-semibold">Upload {documentTypes.find(d => d.type === selectedDocType)?.title}</h3>
                    <Button variant="ghost" onClick={resetUpload}>
                      <X className="w-4 h-4 mr-2" />
                      Cancel
                    </Button>
                  </div>

                  <div className="space-y-6">
                    {/* Front Image Upload */}
                    <div>
                      <label className="block font-medium mb-2">
                        Front of Document *
                      </label>
                      {frontImage ? (
                        <div className="relative">
                          <ImageWithFallback
                            src={frontImage}
                            alt="Front of document"
                            className="w-full max-w-md h-48 object-cover rounded-lg border"
                          />
                          <Button
                            variant="destructive"
                            size="sm"
                            className="absolute top-2 right-2"
                            onClick={() => setFrontImage(null)}
                          >
                            <Trash2 className="w-4 h-4" />
                          </Button>
                        </div>
                      ) : (
                        <div className="border-2 border-dashed border-muted-foreground/20 rounded-lg p-8 text-center">
                          <input
                            type="file"
                            accept="image/*"
                            onChange={(e) => e.target.files?.[0] && handleFileUpload(e.target.files[0])}
                            className="hidden"
                            id="front-upload"
                          />
                          <label htmlFor="front-upload" className="cursor-pointer">
                            <Upload className="w-8 h-8 text-muted-foreground mx-auto mb-2" />
                            <p className="text-muted-foreground">Click to upload front image</p>
                            <p className="text-xs text-muted-foreground mt-1">PNG, JPG up to 10MB</p>
                          </label>
                        </div>
                      )}
                    </div>

                    {/* Back Image Upload (if required) */}
                    {documentTypes.find(d => d.type === selectedDocType)?.requiresBack && (
                      <div>
                        <label className="block font-medium mb-2">
                          Back of Document *
                        </label>
                        {backImage ? (
                          <div className="relative">
                            <ImageWithFallback
                              src={backImage}
                              alt="Back of document"
                              className="w-full max-w-md h-48 object-cover rounded-lg border"
                            />
                            <Button
                              variant="destructive"
                              size="sm"
                              className="absolute top-2 right-2"
                              onClick={() => setBackImage(null)}
                            >
                              <Trash2 className="w-4 h-4" />
                            </Button>
                          </div>
                        ) : (
                          <div className="border-2 border-dashed border-muted-foreground/20 rounded-lg p-8 text-center">
                            <input
                              type="file"
                              accept="image/*"
                              onChange={(e) => e.target.files?.[0] && handleFileUpload(e.target.files[0], true)}
                              className="hidden"
                              id="back-upload"
                            />
                            <label htmlFor="back-upload" className="cursor-pointer">
                              <Upload className="w-8 h-8 text-muted-foreground mx-auto mb-2" />
                              <p className="text-muted-foreground">Click to upload back image</p>
                              <p className="text-xs text-muted-foreground mt-1">PNG, JPG up to 10MB</p>
                            </label>
                          </div>
                        )}
                      </div>
                    )}

                    {/* Upload Guidelines */}
                    <Alert>
                      <Camera className="h-4 w-4" />
                      <AlertDescription>
                        <strong>Photo Guidelines:</strong> Ensure your document is clearly visible, well-lit, and all text is readable. 
                        Avoid glare, shadows, or blurry images. All personal information must be visible and match your profile details.
                      </AlertDescription>
                    </Alert>

                    {/* Action Buttons */}
                    <div className="flex gap-3">
                      <Button 
                        onClick={() => setUploadStep('review')}
                        disabled={!frontImage || (documentTypes.find(d => d.type === selectedDocType)?.requiresBack && !backImage)}
                        className="flex-1"
                      >
                        Review & Submit
                      </Button>
                      <Button variant="outline" onClick={resetUpload}>
                        Start Over
                      </Button>
                    </div>
                  </div>
                </Card>
              )}

              {/* Review Before Submit */}
              {uploadStep === 'review' && (
                <Card className="p-6">
                  <h3 className="font-semibold mb-4">Review Your Submission</h3>
                  
                  <div className="space-y-4 mb-6">
                    <div>
                      <p className="font-medium mb-2">Document Type:</p>
                      <p className="text-muted-foreground">
                        {documentTypes.find(d => d.type === selectedDocType)?.title}
                      </p>
                    </div>

                    <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                      {frontImage && (
                        <div>
                          <p className="font-medium mb-2">Front Image:</p>
                          <ImageWithFallback
                            src={frontImage}
                            alt="Front of document"
                            className="w-full h-32 object-cover rounded-lg border"
                          />
                        </div>
                      )}
                      
                      {backImage && (
                        <div>
                          <p className="font-medium mb-2">Back Image:</p>
                          <ImageWithFallback
                            src={backImage}
                            alt="Back of document"
                            className="w-full h-32 object-cover rounded-lg border"
                          />
                        </div>
                      )}
                    </div>
                  </div>

                  <Alert className="mb-6">
                    <AlertTriangle className="h-4 w-4" />
                    <AlertDescription>
                      By submitting this verification request, you confirm that the information provided is accurate and belongs to you. 
                      False information may result in account suspension.
                    </AlertDescription>
                  </Alert>

                  {isUploading && (
                    <div className="mb-6">
                      <div className="flex items-center justify-between mb-2">
                        <span className="text-sm font-medium">Uploading...</span>
                        <span className="text-sm text-muted-foreground">{uploadProgress}%</span>
                      </div>
                      <Progress value={uploadProgress} />
                    </div>
                  )}

                  <div className="flex gap-3">
                    <Button 
                      onClick={handleSubmitVerification}
                      disabled={isUploading}
                      className="flex-1"
                    >
                      {isUploading ? 'Submitting...' : 'Submit for Review'}
                    </Button>
                    <Button 
                      variant="outline" 
                      onClick={() => setUploadStep('upload')}
                      disabled={isUploading}
                    >
                      Back to Edit
                    </Button>
                  </div>
                </Card>
              )}

              {/* Submission Success */}
              {uploadStep === 'submitted' && (
                <Card className="p-6 text-center">
                  <div className="w-16 h-16 bg-green-100 dark:bg-green-900 rounded-full flex items-center justify-center mx-auto mb-4">
                    <CheckCircle className="w-8 h-8 text-green-600 dark:text-green-400" />
                  </div>
                  <h3 className="font-semibold mb-2">Verification Submitted!</h3>
                  <p className="text-muted-foreground mb-6">
                    Your verification request has been submitted successfully. Our team will review your documents 
                    within 24-48 hours and notify you of the result.
                  </p>
                  <Button onClick={() => onNavigate("profile-settings")}>
                    Return to Profile Settings
                  </Button>
                </Card>
              )}
            </div>
          )}

          {/* Verification History */}
          {userVerification.verificationHistory.length > 0 && (
            <Card className="p-6 mt-8">
              <h3 className="font-semibold mb-4">Verification History</h3>
              <div className="space-y-4">
                {userVerification.verificationHistory.map((request) => (
                  <div key={request.id} className="flex items-center justify-between p-4 border rounded-lg">
                    <div className="flex items-center gap-4">
                      <div className={`w-3 h-3 rounded-full ${getStatusColor(request.status)}`}></div>
                      <div>
                        <p className="font-medium">
                          {documentTypes.find(d => d.type === request.type)?.title}
                        </p>
                        <p className="text-sm text-muted-foreground">
                          Submitted: {new Date(request.submittedAt).toLocaleDateString()}
                        </p>
                        {request.rejectionReason && (
                          <p className="text-sm text-red-600 mt-1">
                            <strong>Rejection Reason:</strong> {request.rejectionReason}
                          </p>
                        )}
                      </div>
                    </div>
                    <div className="flex items-center gap-2">
                      {getStatusIcon(request.status)}
                      <span className="text-sm font-medium capitalize">{request.status}</span>
                    </div>
                  </div>
                ))}
              </div>
            </Card>
          )}
        </div>
      </div>

      <Footer onNavigate={onNavigate} onOpenCookieConsent={onOpenCookieConsent} />
    </div>
  );
}