import { createContext, useContext, useState, useRef, useEffect, ReactNode } from 'react';

interface MediaContextType {
  // Video state
  introVideo: string | null;
  isVideoPlaying: boolean;
  videoElement: HTMLVideoElement | null;
  
  // Music state
  profileMusic: string | null;
  isMusicPlaying: boolean;
  isMusicPaused: boolean;
  musicElement: HTMLAudioElement | null;
  
  // Background state
  customBackground: string | null;
  
  // Control methods
  setIntroVideo: (src: string | null) => void;
  setProfileMusic: (src: string | null) => void;
  setCustomBackground: (src: string | null) => void;
  
  // Media control methods
  registerVideoElement: (element: HTMLVideoElement | null) => void;
  registerMusicElement: (element: HTMLAudioElement | null) => void;
  
  playMusic: () => void;
  pauseMusic: () => void;
  stopMusic: () => void;
  
  // Volume control
  musicVolume: number;
  setMusicVolume: (volume: number) => void;
}

const MediaContext = createContext<MediaContextType | undefined>(undefined);

interface MediaProviderProps {
  children: ReactNode;
}

export function MediaProvider({ children }: MediaProviderProps) {
  // Media sources
  const [introVideo, setIntroVideo] = useState<string | null>(null);
  const [profileMusic, setProfileMusic] = useState<string | null>(null);
  const [customBackground, setCustomBackground] = useState<string | null>(null);
  
  // Media elements
  const [videoElement, setVideoElement] = useState<HTMLVideoElement | null>(null);
  const [musicElement, setMusicElement] = useState<HTMLAudioElement | null>(null);
  
  // Playback states
  const [isVideoPlaying, setIsVideoPlaying] = useState(false);
  const [isMusicPlaying, setIsMusicPlaying] = useState(false);
  const [isMusicPaused, setIsMusicPaused] = useState(false);
  const [musicVolume, setMusicVolume] = useState(0.7);
  
  // Refs for cleanup and user interaction tracking
  const musicTimeoutRef = useRef<NodeJS.Timeout | null>(null);
  const userInteracted = useRef(false);

  // Register media elements
  const registerVideoElement = (element: HTMLVideoElement | null) => {
    if (videoElement) {
      // Clean up previous event listeners
      videoElement.removeEventListener('play', handleVideoPlay);
      videoElement.removeEventListener('pause', handleVideoPause);
      videoElement.removeEventListener('ended', handleVideoEnded);
    }
    
    setVideoElement(element);
    
    if (element) {
      // Add event listeners for video state management
      element.addEventListener('play', handleVideoPlay);
      element.addEventListener('pause', handleVideoPause);
      element.addEventListener('ended', handleVideoEnded);
    }
  };

  const registerMusicElement = (element: HTMLAudioElement | null) => {
    if (musicElement) {
      // Clean up previous event listeners
      musicElement.removeEventListener('play', handleMusicPlay);
      musicElement.removeEventListener('pause', handleMusicPauseEvent);
      musicElement.removeEventListener('ended', handleMusicEnded);
      musicElement.removeEventListener('loadeddata', handleMusicLoaded);
      musicElement.removeEventListener('canplaythrough', handleMusicCanPlay);
    }
    
    setMusicElement(element);
    
    if (element) {
      // Set initial volume and loop
      element.volume = musicVolume;
      element.loop = true;
      element.preload = 'auto';
      
      // Add event listeners for music state management
      element.addEventListener('play', handleMusicPlay);
      element.addEventListener('pause', handleMusicPauseEvent);
      element.addEventListener('ended', handleMusicEnded);
      element.addEventListener('loadeddata', handleMusicLoaded);
      element.addEventListener('canplaythrough', handleMusicCanPlay);
    }
  };

  // Video event handlers
  const handleVideoPlay = () => {
    setIsVideoPlaying(true);
    // Pause music when video starts playing
    if (musicElement && isMusicPlaying) {
      musicElement.pause();
      setIsMusicPaused(true);
    }
  };

  const handleVideoPause = () => {
    setIsVideoPlaying(false);
    // Resume music if it was paused by video
    if (musicElement && isMusicPaused && profileMusic) {
      // Small delay to prevent audio overlap
      musicTimeoutRef.current = setTimeout(() => {
        if (musicElement) {
          musicElement.play();
          setIsMusicPaused(false);
        }
      }, 300);
    }
  };

  const handleVideoEnded = () => {
    setIsVideoPlaying(false);
    // Resume music when video ends
    if (musicElement && profileMusic) {
      // Small delay for smooth transition
      musicTimeoutRef.current = setTimeout(() => {
        if (musicElement) {
          musicElement.play();
          setIsMusicPaused(false);
        }
      }, 500);
    }
  };

  // Music event handlers
  const handleMusicPlay = () => {
    setIsMusicPlaying(true);
    setIsMusicPaused(false);
  };

  const handleMusicPauseEvent = () => {
    setIsMusicPlaying(false);
  };

  const handleMusicEnded = () => {
    setIsMusicPlaying(false);
    // Auto-restart music since it should loop
    if (musicElement && profileMusic) {
      musicElement.currentTime = 0;
      musicElement.play().catch(() => {
        // If auto-play fails, wait for user interaction
        console.log('Music loop prevented by browser, waiting for user interaction');
      });
    }
  };

  // New handlers for music loading
  const handleMusicLoaded = () => {
    // Music is loaded and ready
    if (musicElement && profileMusic && !isVideoPlaying && userInteracted.current) {
      attemptAutoPlay();
    }
  };

  const handleMusicCanPlay = () => {
    // Music can start playing
    if (musicElement && profileMusic && !isVideoPlaying) {
      attemptAutoPlay();
    }
  };

  // Attempt to auto-play with better error handling
  const attemptAutoPlay = () => {
    if (musicElement && profileMusic && !isVideoPlaying && !isMusicPlaying) {
      musicElement.play().then(() => {
        setIsMusicPlaying(true);
        setIsMusicPaused(false);
      }).catch((error) => {
        console.log('Auto-play prevented:', error.message);
        // Auto-play failed, we'll try again when user interacts
      });
    }
  };

  // Music control methods
  const playMusic = () => {
    if (musicElement && profileMusic && !isVideoPlaying) {
      musicElement.play().catch(console.error);
    }
  };

  const pauseMusic = () => {
    if (musicElement) {
      musicElement.pause();
      setIsMusicPaused(true);
    }
  };

  const stopMusic = () => {
    if (musicElement) {
      musicElement.pause();
      musicElement.currentTime = 0;
      setIsMusicPaused(false);
    }
  };

  // Update music volume when it changes
  useEffect(() => {
    if (musicElement) {
      musicElement.volume = musicVolume;
    }
  }, [musicVolume, musicElement]);

  // Auto-start music when profile music is set (if no video is playing)
  useEffect(() => {
    if (profileMusic && musicElement && !isVideoPlaying) {
      // Multiple attempts with increasing delays to handle different browser behaviors
      const attempts = [500, 1000, 2000];
      
      attempts.forEach((delay, index) => {
        setTimeout(() => {
          if (musicElement && profileMusic && !isVideoPlaying && !isMusicPlaying) {
            attemptAutoPlay();
          }
        }, delay);
      });
    }
  }, [profileMusic, musicElement, isVideoPlaying]);

  // Add user interaction listeners to enable auto-play after first interaction
  useEffect(() => {
    const handleUserInteraction = () => {
      userInteracted.current = true;
      // Try to start music if conditions are right
      if (profileMusic && musicElement && !isVideoPlaying && !isMusicPlaying) {
        attemptAutoPlay();
      }
      
      // Remove listeners after first interaction
      document.removeEventListener('click', handleUserInteraction);
      document.removeEventListener('touchstart', handleUserInteraction);
      document.removeEventListener('keydown', handleUserInteraction);
    };

    document.addEventListener('click', handleUserInteraction);
    document.addEventListener('touchstart', handleUserInteraction);
    document.addEventListener('keydown', handleUserInteraction);

    return () => {
      document.removeEventListener('click', handleUserInteraction);
      document.removeEventListener('touchstart', handleUserInteraction);
      document.removeEventListener('keydown', handleUserInteraction);
    };
  }, [profileMusic, musicElement, isVideoPlaying, isMusicPlaying]);

  // Cleanup on unmount
  useEffect(() => {
    return () => {
      if (musicTimeoutRef.current) {
        clearTimeout(musicTimeoutRef.current);
      }
    };
  }, []);

  const value: MediaContextType = {
    // Video state
    introVideo,
    isVideoPlaying,
    videoElement,
    
    // Music state
    profileMusic,
    isMusicPlaying,
    isMusicPaused,
    musicElement,
    
    // Background state
    customBackground,
    
    // Control methods
    setIntroVideo,
    setProfileMusic,
    setCustomBackground,
    
    // Media control methods
    registerVideoElement,
    registerMusicElement,
    
    playMusic,
    pauseMusic,
    stopMusic,
    
    // Volume control
    musicVolume,
    setMusicVolume,
  };

  return (
    <MediaContext.Provider value={value}>
      {children}
    </MediaContext.Provider>
  );
}

export function useMedia() {
  const context = useContext(MediaContext);
  if (context === undefined) {
    throw new Error('useMedia must be used within a MediaProvider');
  }
  return context;
}

// Hook for easy video management
export function useVideoManager(videoRef: React.RefObject<HTMLVideoElement>) {
  const { registerVideoElement } = useMedia();
  
  useEffect(() => {
    if (videoRef.current) {
      registerVideoElement(videoRef.current);
    }
    
    return () => {
      registerVideoElement(null);
    };
  }, [videoRef, registerVideoElement]);
}

// Hook for easy music management
export function useMusicManager(audioRef: React.RefObject<HTMLAudioElement>) {
  const { registerMusicElement } = useMedia();
  
  useEffect(() => {
    if (audioRef.current) {
      registerMusicElement(audioRef.current);
    }
    
    return () => {
      registerMusicElement(null);
    };
  }, [audioRef, registerMusicElement]);
}