import { useState } from "react";
import { Header } from "./Header";
import { Footer } from "./Footer";
import { Button } from "./ui/button";
import { Card } from "./ui/card";
import { Badge } from "./ui/badge";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "./ui/tabs";
import { Heart, MessageCircle, Star, Clock, Eye, Crown, Target, Shield } from "lucide-react";
import { motion } from "motion/react";
import { ImageWithFallback } from "./figma/ImageWithFallback";
import { getQuickMatchPercentage } from "./MatchingAlgorithm";
import { useColorTheme } from "./ColorThemeProvider";

interface MatchesPageProps {
  onNavigate: (view: string) => void;
  onOpenCookieConsent?: () => void;
}

interface Match {
  id: string;
  name: string;
  age: number;
  location: string;
  image: string;
  matchPercentage: number;
  lastActive: string;
  isNew: boolean;
  commonInterests: string[];
  colorTheme?: {
    primary: string;
    secondary: string;
    accent: string;
    gradient1: string;
    gradient2: string;
    gradient3: string;
  };
}

interface ProfileViewer {
  id: string;
  name: string;
  age: number;
  location: string;
  country: string;
  image: string;
  viewedAt: string;
  viewCount: number;
  isOnline: boolean;
  isVerified: boolean;
  isPremium: boolean;
  interests: string[];
  bio: string;
  occupation: string;
  education: string;
  height: string;
  distance: number;
  joinDate: string;
  lookingFor?: {
    relationshipType: string;
    lookingForText: string;
    ageRange: { min: number; max: number };
    maxDistance: number;
    dealBreakers: string[];
    importantQualities: string[];
    countriesPreference: "all" | "specific" | "multiple";
    selectedCountries: string[];
  };
}

interface ProfileLiker {
  id: string;
  name: string;
  age: number;
  location: string;
  country: string;
  image: string;
  likedAt: string;
  isOnline: boolean;
  isVerified: boolean;
  isPremium: boolean;
  interests: string[];
  bio: string;
  occupation: string;
  education: string;
  height: string;
  distance: number;
  joinDate: string;
  lookingFor?: {
    relationshipType: string;
    lookingForText: string;
    ageRange: { min: number; max: number };
    maxDistance: number;
    dealBreakers: string[];
    importantQualities: string[];
    countriesPreference: "all" | "specific" | "multiple";
    selectedCountries: string[];
  };
}

export function MatchesPage({ onNavigate, onOpenCookieConsent }: MatchesPageProps) {
  const { theme } = useColorTheme();
  const [activeMainTab, setActiveMainTab] = useState<"matches" | "views" | "likes">("matches");
  const [activeTab, setActiveTab] = useState<"all" | "new" | "favorites">("all");

  // Sample color themes for matches
  const matchColorThemes = [
    { primary: '#ec4899', secondary: '#a855f7', accent: '#6366f1', gradient1: '#ec4899', gradient2: '#a855f7', gradient3: '#6366f1' },
    { primary: '#f59e0b', secondary: '#ef4444', accent: '#ec4899', gradient1: '#f59e0b', gradient2: '#ef4444', gradient3: '#ec4899' },
    { primary: '#0ea5e9', secondary: '#06b6d4', accent: '#10b981', gradient1: '#0ea5e9', gradient2: '#06b6d4', gradient3: '#10b981' },
    { primary: '#10b981', secondary: '#22c55e', accent: '#84cc16', gradient1: '#10b981', gradient2: '#22c55e', gradient3: '#84cc16' }
  ];

  // Mock current user profile for match calculations
  const currentUser: ProfileViewer = {
    id: "current-user-0",
    name: "You",
    age: 32,
    location: "New York",
    country: "United States",
    image: "https://images.unsplash.com/photo-1507003211169-0a1dd7228f2d?w=400&h=400&fit=crop&crop=face",
    viewedAt: "N/A",
    viewCount: 0,
    isOnline: true,
    isVerified: true,
    isPremium: true,
    interests: ["Technology", "Business", "Travel", "Innovation", "Networking", "Leadership", "Photography", "Coffee"],
    bio: "Tech entrepreneur and dating app creator. Passionate about bringing people together and creating meaningful connections.",
    occupation: "CEO & Founder",
    education: "MBA in Business Administration",
    height: "5'6\"",
    distance: 0,
    joinDate: "January 2020",
    lookingFor: {
      relationshipType: "Serious relationship",
      lookingForText: "Seeking someone who values growth, authenticity, and building something meaningful together.",
      ageRange: { min: 28, max: 40 },
      maxDistance: 25,
      dealBreakers: ["Dishonesty", "No ambition", "Disrespectful"],
      importantQualities: ["Honesty", "Ambition", "Humor", "Kindness", "Intelligence", "Authenticity"],
      countriesPreference: "multiple",
      selectedCountries: ["United States", "Canada", "United Kingdom"]
    }
  };
  
  // Mock data for profile viewers
  const [profileViewers] = useState<ProfileViewer[]>([
    {
      id: "viewer-101",
      name: "Alexandra",
      age: 29,
      location: "Manhattan",
      country: "United States",
      image: "https://images.unsplash.com/photo-1594736797933-d0d9e5a0b51c?w=400&h=400&fit=crop&crop=face",
      viewedAt: "2 hours ago",
      viewCount: 3,
      isOnline: true,
      isVerified: true,
      isPremium: false,
      interests: ["Photography", "Travel", "Art", "Coffee", "Design"],
      bio: "Creative professional with a passion for capturing life's beautiful moments.",
      occupation: "Photographer",
      education: "Bachelor's Degree",
      height: "5'6\"",
      distance: 5,
      joinDate: "March 2023",
      lookingFor: {
        relationshipType: "Serious relationship",
        lookingForText: "Looking for someone creative and ambitious to share adventures with.",
        ageRange: { min: 27, max: 38 },
        maxDistance: 20,
        dealBreakers: ["Dishonesty", "Smoking"],
        importantQualities: ["Creativity", "Authenticity", "Ambition"],
        countriesPreference: "specific",
        selectedCountries: ["United States"]
      }
    },
    {
      id: "viewer-102",
      name: "Jessica",
      age: 31,
      location: "Brooklyn",
      country: "United States",
      image: "https://images.unsplash.com/photo-1487412720507-e7ab37603c6f?w=400&h=400&fit=crop&crop=face",
      viewedAt: "5 hours ago",
      viewCount: 1,
      isOnline: false,
      isVerified: false,
      isPremium: true,
      interests: ["Business", "Fitness", "Networking", "Technology"],
      bio: "Entrepreneur building the next big thing in tech.",
      occupation: "Product Manager",
      education: "Master's Degree",
      height: "5'7\"",
      distance: 12,
      joinDate: "January 2023",
      lookingFor: {
        relationshipType: "Serious relationship",
        lookingForText: "Seeking a partner who understands the startup life and shares my ambition.",
        ageRange: { min: 29, max: 42 },
        maxDistance: 30,
        dealBreakers: ["No ambition", "Disrespectful"],
        importantQualities: ["Ambition", "Intelligence", "Understanding"],
        countriesPreference: "multiple",
        selectedCountries: ["United States", "Canada"]
      }
    },
    {
      id: "viewer-103",
      name: "Maria",
      age: 27,
      location: "Queens",
      country: "United States",
      image: "https://images.unsplash.com/photo-1524250502761-1ac6f2e30d43?w=400&h=400&fit=crop&crop=face",
      viewedAt: "1 day ago",
      viewCount: 2,
      isOnline: true,
      isVerified: true,
      isPremium: false,
      interests: ["Yoga", "Cooking", "Music", "Travel"],
      bio: "Wellness coach who believes in balance and mindful living.",
      occupation: "Wellness Coach",
      education: "Bachelor's Degree",
      height: "5'4\"",
      distance: 18,
      joinDate: "May 2023",
      lookingFor: {
        relationshipType: "Serious relationship",
        lookingForText: "Looking for someone who values health, mindfulness, and authentic connections.",
        ageRange: { min: 25, max: 35 },
        maxDistance: 25,
        dealBreakers: ["Smoking", "Negativity"],
        importantQualities: ["Kindness", "Health-conscious", "Authenticity"],
        countriesPreference: "all",
        selectedCountries: []
      }
    },
    {
      id: "viewer-104",
      name: "Sarah",
      age: 33,
      location: "Hoboken",
      country: "United States",
      image: "https://images.unsplash.com/photo-1489424731084-a5d8b219a5bb?w=400&h=400&fit=crop&crop=face",
      viewedAt: "2 days ago",
      viewCount: 1,
      isOnline: false,
      isVerified: true,
      isPremium: true,
      interests: ["Law", "Reading", "Wine", "Theater"],
      bio: "Corporate lawyer with a passion for justice and the arts.",
      occupation: "Corporate Lawyer",
      education: "PhD",
      height: "5'8\"",
      distance: 8,
      joinDate: "February 2023",
      lookingFor: {
        relationshipType: "Marriage",
        lookingForText: "Ready to find my life partner and build a future together.",
        ageRange: { min: 30, max: 45 },
        maxDistance: 20,
        dealBreakers: ["Dishonesty", "No ambition"],
        importantQualities: ["Intelligence", "Integrity", "Ambition"],
        countriesPreference: "specific",
        selectedCountries: ["United States"]
      }
    },
    {
      id: "viewer-105",
      name: "Emily",
      age: 26,
      location: "Jersey City",
      country: "United States",
      image: "https://images.unsplash.com/photo-1531123897727-8f129e1688ce?w=400&h=400&fit=crop&crop=face",
      viewedAt: "3 days ago",
      viewCount: 4,
      isOnline: true,
      isVerified: false,
      isPremium: false,
      interests: ["Teaching", "Children", "Books", "Volunteering"],
      bio: "Elementary school teacher who loves making a difference in young lives.",
      occupation: "Teacher",
      education: "Bachelor's Degree",
      height: "5'5\"",
      distance: 15,
      joinDate: "April 2023",
      lookingFor: {
        relationshipType: "Serious relationship",
        lookingForText: "Looking for someone family-oriented who shares my love for helping others.",
        ageRange: { min: 24, max: 32 },
        maxDistance: 30,
        dealBreakers: ["Disrespectful", "No family values"],
        importantQualities: ["Kindness", "Family-oriented", "Caring"],
        countriesPreference: "specific",
        selectedCountries: ["United States"]
      }
    }
  ]);

  // Mock data for profile likers
  const [profileLikers] = useState<ProfileLiker[]>([
    {
      id: "liker-201",
      name: "Isabella",
      age: 30,
      location: "Upper East Side",
      country: "United States",
      image: "https://images.unsplash.com/photo-1517841905240-472988babdf9?w=400&h=400&fit=crop&crop=face",
      likedAt: "1 hour ago",
      isOnline: true,
      isVerified: true,
      isPremium: true,
      interests: ["Business", "Fashion", "Travel", "Wine"],
      bio: "Fashion executive with a love for beautiful things and meaningful connections.",
      occupation: "Fashion Executive",
      education: "Master's Degree",
      height: "5'7\"",
      distance: 3,
      joinDate: "January 2023",
      lookingFor: {
        relationshipType: "Serious relationship",
        lookingForText: "Seeking an ambitious partner who appreciates both style and substance.",
        ageRange: { min: 28, max: 40 },
        maxDistance: 15,
        dealBreakers: ["No style", "Lack of ambition"],
        importantQualities: ["Ambition", "Style", "Sophistication"],
        countriesPreference: "multiple",
        selectedCountries: ["United States", "France", "Italy"]
      }
    },
    {
      id: "liker-202",
      name: "Sophia",
      age: 28,
      location: "Williamsburg",
      country: "United States",
      image: "https://images.unsplash.com/photo-1508214751196-bcfd4ca60f91?w=400&h=400&fit=crop&crop=face",
      likedAt: "3 hours ago",
      isOnline: false,
      isVerified: true,
      isPremium: false,
      interests: ["Technology", "Innovation", "Coding", "Coffee"],
      bio: "Software engineer passionate about building the future through code.",
      occupation: "Software Engineer",
      education: "Bachelor's Degree",
      height: "5'6\"",
      distance: 10,
      joinDate: "February 2023",
      lookingFor: {
        relationshipType: "Serious relationship",
        lookingForText: "Looking for someone who gets the tech world and shares my passion for innovation.",
        ageRange: { min: 26, max: 38 },
        maxDistance: 25,
        dealBreakers: ["Technology averse", "No intellectual curiosity"],
        importantQualities: ["Intelligence", "Innovation", "Understanding"],
        countriesPreference: "all",
        selectedCountries: []
      }
    },
    {
      id: "liker-203",
      name: "Grace",
      age: 34,
      location: "Tribeca",
      country: "United States",
      image: "https://images.unsplash.com/photo-1506277886164-e25aa3f4ef7f?w=400&h=400&fit=crop&crop=face",
      likedAt: "6 hours ago",
      isOnline: true,
      isVerified: true,
      isPremium: true,
      interests: ["Finance", "Investing", "Leadership", "Travel"],
      bio: "Investment banker with a passion for financial markets and global adventures.",
      occupation: "Investment Banker",
      education: "MBA",
      height: "5'9\"",
      distance: 2,
      joinDate: "December 2022",
      lookingFor: {
        relationshipType: "Marriage",
        lookingForText: "Ready for a power couple partnership with someone equally driven.",
        ageRange: { min: 30, max: 45 },
        maxDistance: 20,
        dealBreakers: ["Financial irresponsibility", "Lack of drive"],
        importantQualities: ["Ambition", "Financial savvy", "Leadership"],
        countriesPreference: "multiple",
        selectedCountries: ["United States", "United Kingdom", "Switzerland"]
      }
    },
    {
      id: "liker-204",
      name: "Olivia",
      age: 29,
      location: "SoHo",
      country: "United States",
      image: "https://images.unsplash.com/photo-1507003211169-0a1dd7228f2d?w=400&h=400&fit=crop&crop=face",
      likedAt: "1 day ago",
      isOnline: false,
      isVerified: false,
      isPremium: false,
      interests: ["Art", "Museums", "Culture", "Design"],
      bio: "Museum curator with a deep appreciation for art and cultural experiences.",
      occupation: "Museum Curator",
      education: "Master's Degree",
      height: "5'5\"",
      distance: 7,
      joinDate: "March 2023",
      lookingFor: {
        relationshipType: "Serious relationship",
        lookingForText: "Seeking someone who appreciates culture and wants to explore the world's beauty together.",
        ageRange: { min: 27, max: 37 },
        maxDistance: 30,
        dealBreakers: ["Lack of culture", "Closed-mindedness"],
        importantQualities: ["Cultural awareness", "Open-mindedness", "Curiosity"],
        countriesPreference: "multiple",
        selectedCountries: ["United States", "France", "Italy", "Spain"]
      }
    }
  ]);

  const [matches] = useState<Match[]>([
    {
      id: "match-1",
      name: "Emma",
      age: 26,
      location: "Los Angeles, CA",
      image: "https://images.unsplash.com/photo-1494790108755-2616b612b8c0?w=400&h=400&fit=crop&crop=face",
      matchPercentage: 95,
      lastActive: "2 hours ago",
      isNew: true,
      commonInterests: ["Travel", "Photography", "Fitness"],
      colorTheme: matchColorThemes[0]
    },
    {
      id: "match-2",
      name: "Sophia",
      age: 29,
      location: "New York, NY",
      image: "https://images.unsplash.com/photo-1438761681033-6461ffad8d80?w=400&h=400&fit=crop&crop=face",
      matchPercentage: 89,
      lastActive: "1 day ago",
      isNew: false,
      commonInterests: ["Music", "Art", "Coffee"],
      colorTheme: matchColorThemes[1]
    },
    {
      id: "match-3",
      name: "Isabella",
      age: 24,
      location: "Miami, FL",
      image: "https://images.unsplash.com/photo-1544005313-94ddf0286df2?w=400&h=400&fit=crop&crop=face",
      matchPercentage: 92,
      lastActive: "30 minutes ago",
      isNew: true,
      commonInterests: ["Dancing", "Beach", "Yoga"],
      colorTheme: matchColorThemes[2]
    },
    {
      id: "match-4",
      name: "Olivia",
      age: 27,
      location: "Chicago, IL",
      image: "https://images.unsplash.com/photo-1534528741775-53994a69daeb?w=400&h=400&fit=crop&crop=face",
      matchPercentage: 87,
      lastActive: "3 days ago",
      isNew: false,
      commonInterests: ["Books", "Wine", "Hiking"],
      colorTheme: matchColorThemes[3]
    }
  ]);

  const filteredMatches = matches.filter(match => {
    if (activeTab === "new") return match.isNew;
    if (activeTab === "favorites") return match.matchPercentage >= 90;
    return true;
  });

  const handleMessage = (matchId: number | string) => {
    console.log(`Starting conversation with match ${matchId}`);
    // In a real app, this would navigate to the messages page with the specific conversation
    onNavigate("messages");
  };

  const handleViewProfile = (matchId: number | string) => {
    console.log(`Viewing profile for match ${matchId}`);
    // In a real app, this would show the full profile
    onNavigate("user-profile");
  };

  const handleLikeBack = (userId: string) => {
    console.log(`Liking back user ${userId}`);
    // In a real app, this would create a match and potentially send them to matches or messages
    // For now, we'll show a success message
    alert("It's a match! 💕");
  };

  return (
    <div className="min-h-screen bg-background">
      {/* Header */}
      <Header currentView="matches" onNavigate={onNavigate} />
      
      {/* Matches Banner */}
      <div className="bg-gradient-to-r from-pink-600 via-pink-700 to-pink-800 dark:from-pink-700 dark:via-pink-800 dark:to-pink-900 text-white py-11 relative overflow-hidden">
        {/* Moving gradient overlay */}
        <motion.div 
          className="absolute inset-0 bg-gradient-to-r from-transparent via-pink-300/30 to-transparent opacity-40"
          animate={{
            x: ['-100%', '100%']
          }}
          transition={{
            duration: 8,
            repeat: Infinity,
            ease: "linear",
            repeatDelay: 2
          }}
          style={{
            width: '200%',
            background: 'linear-gradient(90deg, transparent 0%, rgba(255,255,255,0.15) 25%, rgba(255,255,255,0.3) 50%, rgba(255,255,255,0.15) 75%, transparent 100%)'
          }}
        />
        
        {/* Additional gradient overlay */}
        <div className="absolute inset-0 bg-gradient-to-r from-transparent via-pink-600/20 to-pink-800/40 dark:from-transparent dark:via-pink-700/20 dark:to-pink-900/40"></div>
        
        <div className="max-w-7xl mx-auto px-6 text-center relative z-10">
          <motion.h1 
            className="text-4xl font-bold mb-4"
            initial={{ opacity: 0, y: 20 }}
            animate={{ opacity: 1, y: 0 }}
            transition={{ 
              duration: 0.8,
              ease: "easeOut"
            }}
          >
            <motion.span
              animate={{ 
                scale: [1, 1.05, 1],
              }}
              transition={{
                duration: 3,
                repeat: Infinity,
                ease: "easeInOut",
                repeatDelay: 0.5
              }}
            >
              💕 Your Activity
            </motion.span>
          </motion.h1>
          <motion.p 
            className="text-lg text-pink-100 dark:text-pink-200"
            initial={{ opacity: 0, y: 20 }}
            animate={{ opacity: 1, y: 0 }}
            transition={{ 
              duration: 0.8,
              delay: 0.2,
              ease: "easeOut"
            }}
          >
            See your matches, who viewed you, and who liked your profile!
          </motion.p>
        </div>
      </div>

      {/* Main Content */}
      <div className="max-w-6xl mx-auto px-6 py-12">
        <Tabs value={activeMainTab} onValueChange={(value) => setActiveMainTab(value as any)} className="w-full">
          {/* Main Navigation Tabs */}
          <TabsList className="grid w-full grid-cols-3 mb-8">
            <TabsTrigger value="matches" className="flex items-center gap-2">
              <Heart className="w-4 h-4" />
              Matches ({matches.length})
            </TabsTrigger>
            <TabsTrigger value="views" className="flex items-center gap-2">
              <Eye className="w-4 h-4" />
              Who Viewed You ({profileViewers.length})
            </TabsTrigger>
            <TabsTrigger value="likes" className="flex items-center gap-2">
              <Star className="w-4 h-4" />
              Who Liked You ({profileLikers.length})
            </TabsTrigger>
          </TabsList>

          {/* Matches Tab Content */}
          <TabsContent value="matches">
            {/* Sub-tabs for matches */}
            <div className="flex justify-center mb-8">
              <div className="flex bg-muted rounded-lg p-1">
                {[
                  { id: "all", label: "All Matches", count: matches.length },
                  { id: "new", label: "New", count: matches.filter(m => m.isNew).length },
                  { id: "favorites", label: "Top Matches", count: matches.filter(m => m.matchPercentage >= 90).length }
                ].map((tab) => (
                  <button
                    key={tab.id}
                    onClick={() => setActiveTab(tab.id as any)}
                    className={`px-4 py-2 rounded-md transition-colors ${
                      activeTab === tab.id
                        ? "bg-background text-foreground shadow-sm"
                        : "text-muted-foreground hover:text-foreground"
                    }`}
                  >
                    {tab.label} ({tab.count})
                  </button>
                ))}
              </div>
            </div>

            {/* Stats */}
            <div className="text-center mb-8">
              <div className="flex justify-center gap-8">
                <div className="text-center">
                  <div className="text-2xl font-bold text-pink-600">{matches.length}</div>
                  <div className="text-muted-foreground">Total Matches</div>
                </div>
                <div className="text-center">
                  <div className="text-2xl font-bold text-green-600">{matches.filter(m => m.isNew).length}</div>
                  <div className="text-muted-foreground">New Today</div>
                </div>
                <div className="text-center">
                  <div className="text-2xl font-bold text-yellow-600">{matches.filter(m => m.matchPercentage >= 90).length}</div>
                  <div className="text-muted-foreground">Top Matches</div>
                </div>
              </div>
            </div>

            {/* Matches Grid */}
            {filteredMatches.length === 0 ? (
              <div className="text-center py-12">
                <Heart className="w-16 h-16 text-muted-foreground mx-auto mb-4" />
                <h3 className="text-xl font-semibold mb-2">No matches yet</h3>
                <p className="text-muted-foreground">Keep browsing to find your perfect match!</p>
              </div>
            ) : (
              <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
                {filteredMatches.map((match, index) => (
                  <motion.div
                    key={match.id}
                    initial={{ opacity: 0, y: 20 }}
                    animate={{ opacity: 1, y: 0 }}
                    transition={{ duration: 0.3, delay: index * 0.1 }}
                  >
                    <Card className="p-6 hover:shadow-lg transition-shadow">
                      <div className="relative mb-4">
                        <ImageWithFallback
                          src={match.image}
                          alt={match.name}
                          className="w-full h-48 object-cover rounded-lg"
                        />
                        {match.isNew && (
                          <Badge className="absolute top-2 left-2 bg-green-600 text-white">
                            New
                          </Badge>
                        )}
                        <div className="absolute top-2 right-2 bg-white/90 dark:bg-black/90 px-2 py-1 rounded-full">
                          <span className="font-semibold text-sm text-pink-600">
                            {match.matchPercentage}%
                          </span>
                        </div>
                      </div>
                      
                      <div className="space-y-3">
                        <div>
                          <h3 className="font-semibold">{match.name}, {match.age}</h3>
                          <p className="text-muted-foreground text-sm">{match.location}</p>
                        </div>
                        
                        <div className="flex items-center gap-1 text-sm text-muted-foreground">
                          <Clock className="w-4 h-4" />
                          <span>Active {match.lastActive}</span>
                        </div>
                        
                        <div className="flex flex-wrap gap-1">
                          {match.commonInterests.slice(0, 2).map((interest) => (
                            <Badge key={interest} variant="secondary" className="text-xs">
                              {interest}
                            </Badge>
                          ))}
                          {match.commonInterests.length > 2 && (
                            <Badge variant="secondary" className="text-xs">
                              +{match.commonInterests.length - 2}
                            </Badge>
                          )}
                        </div>
                        
                        <div className="flex gap-2 pt-2">
                          <Button
                            onClick={() => handleViewProfile(match.id)}
                            variant="outline"
                            size="sm"
                            className="flex-1"
                          >
                            <Star className="w-4 h-4 mr-1" />
                            View
                          </Button>
                          <Button
                            onClick={() => handleMessage(match.id)}
                            size="sm"
                            className="flex-1 bg-pink-600 hover:bg-pink-700"
                          >
                            <MessageCircle className="w-4 h-4 mr-1" />
                            Message
                          </Button>
                        </div>
                      </div>
                    </Card>
                  </motion.div>
                ))}
              </div>
            )}
          </TabsContent>

          {/* Who Viewed You Tab Content */}
          <TabsContent value="views">
            <div className="space-y-6">
              {/* Header */}
              <div className="text-center">
                <h2 className="text-2xl font-bold mb-2">Who Viewed Your Profile</h2>
                <p className="text-muted-foreground">See who's been checking out your profile and their compatibility with you</p>
              </div>

              {/* Stats */}
              <div className="text-center mb-8">
                <div className="flex justify-center gap-8">
                  <div className="text-center">
                    <div className="text-2xl font-bold text-blue-600">{profileViewers.length}</div>
                    <div className="text-muted-foreground">Total Views</div>
                  </div>
                  <div className="text-center">
                    <div className="text-2xl font-bold text-green-600">
                      {profileViewers.filter(v => v.viewedAt.includes('hour')).length}
                    </div>
                    <div className="text-muted-foreground">Today</div>
                  </div>
                  <div className="text-center">
                    <div className="text-2xl font-bold text-purple-600">
                      {profileViewers.filter(v => v.isPremium).length}
                    </div>
                    <div className="text-muted-foreground">Premium Members</div>
                  </div>
                </div>
              </div>

              {/* Viewers Grid */}
              {profileViewers.length === 0 ? (
                <div className="text-center py-12">
                  <Eye className="w-16 h-16 text-muted-foreground mx-auto mb-4" />
                  <h3 className="text-xl font-semibold mb-2">No profile views yet</h3>
                  <p className="text-muted-foreground">Optimize your profile to get more views!</p>
                </div>
              ) : (
                <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
                  {profileViewers.map((viewer, index) => {
                    const matchPercentage = getQuickMatchPercentage(currentUser, viewer);
                    const getMatchColor = (percentage: number) => {
                      if (percentage >= 80) return "bg-green-500";
                      if (percentage >= 60) return "bg-blue-500";
                      if (percentage >= 40) return "bg-yellow-500";
                      return "bg-red-500";
                    };

                    return (
                      <motion.div
                        key={viewer.id}
                        initial={{ opacity: 0, y: 20 }}
                        animate={{ opacity: 1, y: 0 }}
                        transition={{ duration: 0.3, delay: index * 0.1 }}
                      >
                        <Card className="p-6 hover:shadow-lg transition-shadow">
                          <div className="relative mb-4">
                            <ImageWithFallback
                              src={viewer.image}
                              alt={viewer.name}
                              className="w-full h-48 object-cover rounded-lg"
                            />
                            {/* Match Percentage Badge */}
                            <div className="absolute top-2 left-2">
                              <Badge className={`${getMatchColor(matchPercentage)} text-white flex items-center gap-1 px-2 py-1`}>
                                <Target className="w-3 h-3" />
                                {matchPercentage}%
                              </Badge>
                            </div>
                            {/* Premium Badge */}
                            {viewer.isPremium && (
                              <Badge className="absolute top-2 right-2 bg-yellow-500 text-white">
                                <Crown className="w-3 h-3 mr-1" />
                                Premium
                              </Badge>
                            )}
                            {/* Verified Badge */}
                            {viewer.isVerified && (
                              <Badge className="absolute bottom-2 right-2 bg-blue-500 text-white">
                                <Shield className="w-3 h-3 mr-1" />
                                Verified
                              </Badge>
                            )}
                            {/* Online Status */}
                            {viewer.isOnline && (
                              <div className="absolute bottom-2 left-2 w-3 h-3 bg-green-500 rounded-full border-2 border-white"></div>
                            )}
                          </div>
                          
                          <div className="space-y-3">
                            <div>
                              <h3 className="font-semibold">{viewer.name}, {viewer.age}</h3>
                              <p className="text-muted-foreground text-sm">{viewer.location}</p>
                            </div>
                            
                            <div className="flex items-center gap-1 text-sm text-muted-foreground">
                              <Eye className="w-4 h-4" />
                              <span>Viewed {viewer.viewedAt}</span>
                              {viewer.viewCount > 1 && (
                                <Badge variant="outline" className="ml-2 text-xs">
                                  {viewer.viewCount} times
                                </Badge>
                              )}
                            </div>
                            
                            <div className="flex flex-wrap gap-1">
                              {viewer.interests.slice(0, 2).map((interest) => (
                                <Badge key={interest} variant="secondary" className="text-xs">
                                  {interest}
                                </Badge>
                              ))}
                              {viewer.interests.length > 2 && (
                                <Badge variant="secondary" className="text-xs">
                                  +{viewer.interests.length - 2}
                                </Badge>
                              )}
                            </div>
                            
                            <div className="flex gap-2 pt-2">
                              <Button
                                onClick={() => handleViewProfile(viewer.id)}
                                variant="outline"
                                size="sm"
                                className="flex-1"
                              >
                                <Star className="w-4 h-4 mr-1" />
                                View Profile
                              </Button>
                              <Button
                                onClick={() => handleLikeBack(viewer.id)}
                                size="sm"
                                className="flex-1 bg-blue-600 hover:bg-blue-700"
                              >
                                <Heart className="w-4 h-4 mr-1" />
                                Like Back
                              </Button>
                            </div>
                          </div>
                        </Card>
                      </motion.div>
                    );
                  })}
                </div>
              )}
            </div>
          </TabsContent>

          {/* Who Liked You Tab Content */}
          <TabsContent value="likes">
            <div className="space-y-6">
              {/* Header */}
              <div className="text-center">
                <h2 className="text-2xl font-bold mb-2">Who Liked Your Profile</h2>
                <p className="text-muted-foreground">Members who showed interest in your profile - like them back to create a match!</p>
              </div>

              {/* Stats */}
              <div className="text-center mb-8">
                <div className="flex justify-center gap-8">
                  <div className="text-center">
                    <div className="text-2xl font-bold text-pink-600">{profileLikers.length}</div>
                    <div className="text-muted-foreground">Total Likes</div>
                  </div>
                  <div className="text-center">
                    <div className="text-2xl font-bold text-green-600">
                      {profileLikers.filter(l => l.likedAt.includes('hour')).length}
                    </div>
                    <div className="text-muted-foreground">Today</div>
                  </div>
                  <div className="text-center">
                    <div className="text-2xl font-bold text-purple-600">
                      {profileLikers.filter(l => l.isPremium).length}
                    </div>
                    <div className="text-muted-foreground">Premium Members</div>
                  </div>
                </div>
              </div>

              {/* Likers Grid */}
              {profileLikers.length === 0 ? (
                <div className="text-center py-12">
                  <Star className="w-16 h-16 text-muted-foreground mx-auto mb-4" />
                  <h3 className="text-xl font-semibold mb-2">No likes yet</h3>
                  <p className="text-muted-foreground">Be active and engaging to attract more likes!</p>
                </div>
              ) : (
                <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
                  {profileLikers.map((liker, index) => {
                    const matchPercentage = getQuickMatchPercentage(currentUser, liker);
                    const getMatchColor = (percentage: number) => {
                      if (percentage >= 80) return "bg-green-500";
                      if (percentage >= 60) return "bg-blue-500";
                      if (percentage >= 40) return "bg-yellow-500";
                      return "bg-red-500";
                    };

                    return (
                      <motion.div
                        key={liker.id}
                        initial={{ opacity: 0, y: 20 }}
                        animate={{ opacity: 1, y: 0 }}
                        transition={{ duration: 0.3, delay: index * 0.1 }}
                      >
                        <Card className="p-6 hover:shadow-lg transition-shadow border-pink-200 dark:border-pink-800">
                          <div className="relative mb-4">
                            <ImageWithFallback
                              src={liker.image}
                              alt={liker.name}
                              className="w-full h-48 object-cover rounded-lg"
                            />
                            {/* Match Percentage Badge */}
                            <div className="absolute top-2 left-2">
                              <Badge className={`${getMatchColor(matchPercentage)} text-white flex items-center gap-1 px-2 py-1`}>
                                <Target className="w-3 h-3" />
                                {matchPercentage}%
                              </Badge>
                            </div>
                            {/* Premium Badge */}
                            {liker.isPremium && (
                              <Badge className="absolute top-2 right-2 bg-yellow-500 text-white">
                                <Crown className="w-3 h-3 mr-1" />
                                Premium
                              </Badge>
                            )}
                            {/* Verified Badge */}
                            {liker.isVerified && (
                              <Badge className="absolute bottom-2 right-2 bg-blue-500 text-white">
                                <Shield className="w-3 h-3 mr-1" />
                                Verified
                              </Badge>
                            )}
                            {/* Online Status */}
                            {liker.isOnline && (
                              <div className="absolute bottom-2 left-2 w-3 h-3 bg-green-500 rounded-full border-2 border-white"></div>
                            )}
                            {/* Heart Overlay */}
                            <div className="absolute inset-0 flex items-center justify-center bg-pink-500/20 rounded-lg">
                              <Heart className="w-8 h-8 text-pink-500 fill-pink-500" />
                            </div>
                          </div>
                          
                          <div className="space-y-3">
                            <div>
                              <h3 className="font-semibold">{liker.name}, {liker.age}</h3>
                              <p className="text-muted-foreground text-sm">{liker.location}</p>
                            </div>
                            
                            <div className="flex items-center gap-1 text-sm text-muted-foreground">
                              <Heart className="w-4 h-4 text-pink-500" />
                              <span>Liked you {liker.likedAt}</span>
                            </div>
                            
                            <div className="flex flex-wrap gap-1">
                              {liker.interests.slice(0, 2).map((interest) => (
                                <Badge key={interest} variant="secondary" className="text-xs">
                                  {interest}
                                </Badge>
                              ))}
                              {liker.interests.length > 2 && (
                                <Badge variant="secondary" className="text-xs">
                                  +{liker.interests.length - 2}
                                </Badge>
                              )}
                            </div>
                            
                            <div className="flex gap-2 pt-2">
                              <Button
                                onClick={() => handleViewProfile(liker.id)}
                                variant="outline"
                                size="sm"
                                className="flex-1"
                              >
                                <Star className="w-4 h-4 mr-1" />
                                View Profile
                              </Button>
                              <Button
                                onClick={() => handleLikeBack(liker.id)}
                                size="sm"
                                className="flex-1 bg-pink-600 hover:bg-pink-700"
                              >
                                <Heart className="w-4 h-4 mr-1" />
                                Like Back
                              </Button>
                            </div>
                          </div>
                        </Card>
                      </motion.div>
                    );
                  })}
                </div>
              )}
            </div>
          </TabsContent>
        </Tabs>
      </div>
      
      {/* Footer */}
      <Footer onNavigate={onNavigate} onOpenCookieConsent={onOpenCookieConsent} />
    </div>
  );
}