import { useState } from "react";
import { Button } from "./ui/button";
import { Input } from "./ui/input";
import { Label } from "./ui/label";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "./ui/card";
import { Checkbox } from "./ui/checkbox";
import { Alert, AlertDescription } from "./ui/alert";
import { Heart, Eye, EyeOff, AlertTriangle, CheckCircle } from "lucide-react";
import { ReCaptcha } from "./ReCaptcha";
import { useTranslation } from "./TranslationProvider";

interface LoginFormProps {
  onSwitchToRegister: () => void;
  onLoginSuccess?: () => void;
}

interface LoginData {
  email: string;
  password: string;
  rememberMe: boolean;
  captchaVerified: boolean;
}

export function LoginForm({ onSwitchToRegister, onLoginSuccess }: LoginFormProps) {
  const { t } = useTranslation();
  const [loginData, setLoginData] = useState<LoginData>({
    email: "",
    password: "",
    rememberMe: false,
    captchaVerified: false,
  });
  
  const [showPassword, setShowPassword] = useState(false);
  const [loginAttempts, setLoginAttempts] = useState(0);
  const [isLoading, setIsLoading] = useState(false);
  const [loginError, setLoginError] = useState("");
  const [showForgotPassword, setShowForgotPassword] = useState(false);
  const [resetEmailSent, setResetEmailSent] = useState(false);
  const [resetEmail, setResetEmail] = useState("");

  const updateLoginData = (field: keyof LoginData, value: any) => {
    setLoginData(prev => ({ ...prev, [field]: value }));
    setLoginError(""); // Clear error when user starts typing
  };

  const handleLogin = async () => {
    setIsLoading(true);
    setLoginError("");

    // Simulate login API call
    await new Promise(resolve => setTimeout(resolve, 2000));

    // Demo login logic - in real app, this would be API validation
    if (loginData.email === "demo@singleszonne.com" && loginData.password === "password123") {
      // Successful login
      console.log("Login successful:", loginData);
      if (onLoginSuccess) {
        onLoginSuccess();
      } else {
        alert("Welcome back to SinglesZonne! Login successful.");
      }
    } else {
      // Failed login
      setLoginAttempts(prev => prev + 1);
      setLoginError("Invalid email or password. Please try again.");
    }

    setIsLoading(false);
  };

  const handleForgotPassword = async () => {
    if (!resetEmail) {
      setLoginError("Please enter your email address.");
      return;
    }

    setIsLoading(true);
    // Simulate password reset email
    await new Promise(resolve => setTimeout(resolve, 1500));
    setResetEmailSent(true);
    setIsLoading(false);
  };

  const isLoginDisabled = () => {
    return !loginData.email || !loginData.password || !loginData.captchaVerified || isLoading;
  };

  if (showForgotPassword) {
    return (
      <div className="w-full max-w-md mx-auto">
        <div className="mb-8 text-center">
          <div className="flex items-center justify-center gap-2 mb-4">
            <Heart className="w-8 h-8 text-red-500" />
            <h1 className="text-3xl">SinglesZonne</h1>
          </div>
          <p className="text-muted-foreground">Reset your password</p>
        </div>

        <Card>
          <CardHeader>
            <CardTitle>Forgot Password</CardTitle>
            <CardDescription>
              {resetEmailSent 
                ? "Check your email for reset instructions"
                : "Enter your email to receive password reset instructions"
              }
            </CardDescription>
          </CardHeader>
          <CardContent className="space-y-4">
            {resetEmailSent ? (
              <div className="space-y-4">
                <Alert className="border-green-200 bg-green-50">
                  <CheckCircle className="h-4 w-4 text-green-600" />
                  <AlertDescription className="text-green-700">
                    Password reset instructions have been sent to {resetEmail}
                  </AlertDescription>
                </Alert>
                
                <div className="space-y-2">
                  <Button onClick={() => setShowForgotPassword(false)} className="w-full">
                    Back to Login
                  </Button>
                  <Button 
                    variant="outline" 
                    onClick={() => {
                      setResetEmailSent(false);
                      setResetEmail("");
                    }}
                    className="w-full"
                  >
                    Try Different Email
                  </Button>
                </div>
              </div>
            ) : (
              <div className="space-y-4">
                <div className="space-y-2">
                  <Label htmlFor="resetEmail">Email Address</Label>
                  <Input
                    id="resetEmail"
                    type="email"
                    value={resetEmail}
                    onChange={(e) => setResetEmail(e.target.value)}
                    placeholder="Enter your email address"
                  />
                </div>

                {loginError && (
                  <Alert variant="destructive">
                    <AlertTriangle className="h-4 w-4" />
                    <AlertDescription>{loginError}</AlertDescription>
                  </Alert>
                )}

                <div className="space-y-2">
                  <Button 
                    onClick={handleForgotPassword}
                    disabled={isLoading}
                    className="w-full"
                  >
                    {isLoading ? "Sending..." : "Send Reset Instructions"}
                  </Button>
                  <Button 
                    variant="outline" 
                    onClick={() => setShowForgotPassword(false)}
                    className="w-full"
                  >
                    Back to Login
                  </Button>
                </div>
              </div>
            )}
          </CardContent>
        </Card>
      </div>
    );
  }

  return (
    <div className="w-full max-w-md mx-auto">
      <div className="mb-8 text-center">
        <div className="flex items-center justify-center gap-2 mb-4">
          <Heart className="w-8 h-8 text-red-500" />
          <h1 className="text-3xl">SinglesZonne</h1>
        </div>
        <h2 className="text-2xl font-semibold mb-2">{t('auth.login.title')}</h2>
        <p className="text-muted-foreground">{t('auth.login.subtitle')}</p>
      </div>

      <Card>
        <CardHeader>
          <CardTitle>{t('nav.login')}</CardTitle>
          <CardDescription>Enter your credentials to access your account</CardDescription>
        </CardHeader>
        <CardContent className="space-y-4">
          <div className="space-y-2">
            <Label htmlFor="email">{t('auth.email')}</Label>
            <Input
              id="email"
              type="email"
              value={loginData.email}
              onChange={(e) => updateLoginData("email", e.target.value)}
              placeholder={t('auth.email')}
              autoComplete="email"
            />
          </div>

          <div className="space-y-2">
            <Label htmlFor="password">{t('auth.password')}</Label>
            <div className="relative">
              <Input
                id="password"
                type={showPassword ? "text" : "password"}
                value={loginData.password}
                onChange={(e) => updateLoginData("password", e.target.value)}
                placeholder={t('auth.password')}
                autoComplete="current-password"
                className="pr-10"
              />
              <Button
                type="button"
                variant="ghost"
                size="sm"
                className="absolute right-0 top-0 h-full px-3 py-2 hover:bg-transparent"
                onClick={() => setShowPassword(!showPassword)}
              >
                {showPassword ? (
                  <EyeOff className="h-4 w-4 text-muted-foreground" />
                ) : (
                  <Eye className="h-4 w-4 text-muted-foreground" />
                )}
              </Button>
            </div>
          </div>

          <div className="flex items-center justify-between">
            <div className="flex items-center space-x-2">
              <Checkbox
                id="rememberMe"
                checked={loginData.rememberMe}
                onCheckedChange={(checked) => updateLoginData("rememberMe", checked)}
              />
              <Label htmlFor="rememberMe" className="text-sm">
                Remember me
              </Label>
            </div>
            <Button
              variant="link"
              size="sm"
              onClick={() => setShowForgotPassword(true)}
              className="px-0 text-sm"
            >
              {t('auth.forgotPassword')}
            </Button>
          </div>

          <ReCaptcha 
            verified={loginData.captchaVerified}
            onVerify={(verified) => updateLoginData("captchaVerified", verified)}
          />

          {loginError && (
            <Alert variant="destructive">
              <AlertTriangle className="h-4 w-4" />
              <AlertDescription>{loginError}</AlertDescription>
            </Alert>
          )}

          {loginAttempts >= 2 && (
            <Alert>
              <AlertTriangle className="h-4 w-4" />
              <AlertDescription>
                Multiple failed attempts detected. Additional security verification required.
              </AlertDescription>
            </Alert>
          )}

          <Button 
            onClick={handleLogin}
            disabled={isLoginDisabled()}
            className="w-full"
          >
            {isLoading ? "Signing In..." : t('auth.signIn')}
          </Button>

          <div className="text-center">
            <p className="text-sm text-muted-foreground">
              {t('auth.noAccount')}{" "}
              <Button
                variant="link"
                size="sm"
                onClick={onSwitchToRegister}
                className="px-0"
              >
                {t('auth.signUp')}
              </Button>
            </p>
          </div>

          <div className="text-center pt-4 border-t">
            <p className="text-xs text-muted-foreground">
              Demo credentials: demo@singleszonne.com / password123
            </p>
          </div>
        </CardContent>
      </Card>
    </div>
  );
}