import { useState, useEffect, useRef } from "react";
import { Header } from "./Header";
import { Footer } from "./Footer";
import { Button } from "./ui/button";
import { Card, CardContent } from "./ui/card";
import { Badge } from "./ui/badge";
import { Heart, Star, MapPin, Verified, Crown, ChevronLeft, ChevronRight, Pause, Play } from "lucide-react";
import { motion } from "motion/react";
import { ImageWithFallback } from "./figma/ImageWithFallback";

import { useTranslation } from "./TranslationProvider";

interface LandingPageProps {
  onNavigate: (view: string) => void;
  onOpenCookieConsent?: () => void;
}

interface Profile {
  id: number;
  name: string;
  age: number;
  location: string;
  bio: string;
  images: string[];
  interests: string[];
  verified: boolean;
  premium: boolean;
  rating: number;
  height: number; // For varied card heights
}

interface ProfileCardProps {
  profile: Profile;
  index: number;
  onViewProfile: (id: number) => void;
  onLike: (id: number) => void;
}

// Profile Card Component
function ProfileCard({ profile, index, onViewProfile, onLike }: ProfileCardProps) {
  return (
    <motion.div
      initial={{ opacity: 0, scale: 0.9 }}
      animate={{ opacity: 1, scale: 1 }}
      transition={{ duration: 0.4, delay: (index % 12) * 0.02 }}
      className="flex-shrink-0 w-[300px]"
    >
      <Card className="group hover:shadow-xl transition-all duration-300 hover:-translate-y-1 overflow-hidden h-full">
        <div className="relative">
          <ImageWithFallback
            src={profile.images[0]}
            alt={profile.name}
            className="w-full h-[400px] object-cover transition-transform duration-300 group-hover:scale-105"
          />
          
          {/* Overlay on hover */}
          <div className="absolute inset-0 bg-black/0 group-hover:bg-black/20 transition-all duration-300 flex items-end opacity-0 group-hover:opacity-100">
            <div className="p-4 w-full">
              <div className="flex gap-2">
                <Button
                  size="sm"
                  className="bg-white/90 text-gray-800 hover:bg-white flex-1"
                  onClick={() => onViewProfile(profile.id)}
                >
                  View Profile
                </Button>
                <Button
                  size="sm"
                  className="bg-red-500 hover:bg-red-600 text-white px-3"
                  onClick={() => onLike(profile.id)}
                >
                  <Heart className="w-4 h-4" />
                </Button>
              </div>
            </div>
          </div>

          {/* Badges */}
          <div className="absolute top-3 left-3 flex flex-col gap-2">
            {profile.verified && (
              <Badge className="bg-blue-500 text-white">
                <Verified className="w-3 h-3 mr-1" />
                Verified
              </Badge>
            )}
            {profile.premium && (
              <Badge className="bg-yellow-500 text-white">
                <Crown className="w-3 h-3 mr-1" />
                Premium
              </Badge>
            )}
          </div>

          {/* Rating */}
          <div className="absolute top-3 right-3">
            <Badge className="bg-black/50 text-white">
              <Star className="w-3 h-3 mr-1 fill-current" />
              {profile.rating}
            </Badge>
          </div>
        </div>

        <CardContent className="p-4">
          <div className="flex items-center justify-between mb-2">
            <h3 className="font-semibold text-lg">{profile.name}, {profile.age}</h3>
          </div>
          
          <div className="flex items-center text-muted-foreground mb-3">
            <MapPin className="w-4 h-4 mr-1" />
            <span className="text-sm">{profile.location}</span>
          </div>
          
          <p className="text-sm text-muted-foreground mb-3 line-clamp-2">
            {profile.bio}
          </p>
          
          <div className="flex flex-wrap gap-1">
            {profile.interests.slice(0, 3).map((interest) => (
              <Badge key={interest} variant="secondary" className="text-xs">
                {interest}
              </Badge>
            ))}
          </div>
        </CardContent>
      </Card>
    </motion.div>
  );
}

export function LandingPage({ onNavigate, onOpenCookieConsent }: LandingPageProps) {
  const { t } = useTranslation();
  const [profiles, setProfiles] = useState<Profile[]>([]);
  const [loading, setLoading] = useState(true);
  const [isScrolling, setIsScrolling] = useState(true);
  const scrollContainerRef1 = useRef<HTMLDivElement>(null);
  const scrollContainerRef2 = useRef<HTMLDivElement>(null);
  const scrollContainerRef3 = useRef<HTMLDivElement>(null);
  const scrollContainerRef4 = useRef<HTMLDivElement>(null);

  // Generate sample profiles with random heights for masonry effect
  useEffect(() => {
    const generateProfiles = () => {
      const sampleProfiles: Profile[] = [
        {
          id: 1,
          name: "Emma",
          age: 25,
          location: "🇺🇸 New York, USA",
          bio: "Love traveling, photography, and trying new restaurants. Looking for someone to share adventures with!",
          images: ["https://images.unsplash.com/photo-1544005313-94ddf0286df2?w=400&h=600&fit=crop&crop=face"],
          interests: ["Photography", "Travel", "Food"],
          verified: true,
          premium: true,
          rating: 4.8,
          height: Math.floor(Math.random() * 200) + 300
        },
        {
          id: 2,
          name: "Sakura",
          age: 26,
          location: "🇯🇵 Tokyo, Japan",
          bio: "Traditional tea ceremony instructor and modern art enthusiast. Love exploring both old and new culture.",
          images: ["https://images.unsplash.com/photo-1619235327941-c0e47a3ac74a?crop=entropy&cs=tinysrgb&fit=max&fm=jpg&ixid=M3w3Nzg4Nzd8MHwxfHNlYXJjaHwxfHxiZWF1dGlmdWwlMjBhc2lhbiUyMHdvbWFuJTIwcG9ydHJhaXR8ZW58MXx8fHwxNzU4MjM2MzczfDA&ixlib=rb-4.1.0&q=80&w=1080"],
          interests: ["Tea Ceremony", "Art", "Culture"],
          verified: true,
          premium: false,
          rating: 4.9,
          height: Math.floor(Math.random() * 200) + 300
        },
        {
          id: 3,
          name: "Isabella",
          age: 24,
          location: "🇪🇸 Barcelona, Spain",
          bio: "Flamenco dancer and architect. Passionate about preserving cultural heritage through beautiful design.",
          images: ["https://images.unsplash.com/photo-1646767749079-043faae3deb7?crop=entropy&cs=tinysrgb&fit=max&fm=jpg&ixid=M3w3Nzg4Nzd8MHwxfHNlYXJjaHwxfHxiZWF1dGlmdWwlMjBsYXRpbmElMjB3b21hbiUyMHBvcnRyYWl0fGVufDF8fHx8MTc1ODIzNjM4M3ww&ixlib=rb-4.1.0&q=80&w=1080"],
          interests: ["Dance", "Architecture", "Heritage"],
          verified: false,
          premium: true,
          rating: 4.7,
          height: Math.floor(Math.random() * 200) + 300
        },
        {
          id: 4,
          name: "Amara",
          age: 27,
          location: "🇰🇪 Nairobi, Kenya",
          bio: "Wildlife photographer and conservation activist. Dedicated to protecting Africa's magnificent creatures.",
          images: ["https://images.unsplash.com/photo-1669040084821-097235fe53ff?crop=entropy&cs=tinysrgb&fit=max&fm=jpg&ixid=M3w3Nzg4Nzd8MHwxfHNlYXJjaHwxfHxiZWF1dGlmdWwlMjBhZnJpY2FuJTIwd29tYW4lMjBwb3J0cmFpdHxlbnwxfHx8fDE3NTgyMzYzODl8MA&ixlib=rb-4.1.0&q=80&w=1080"],
          interests: ["Wildlife", "Photography", "Conservation"],
          verified: true,
          premium: false,
          rating: 4.6,
          height: Math.floor(Math.random() * 200) + 300
        },
        {
          id: 5,
          name: "Layla",
          age: 25,
          location: "🇦🇪 Dubai, UAE",
          bio: "International business consultant and luxury travel blogger. Love discovering hidden gems around the world.",
          images: ["https://images.unsplash.com/photo-1547944379-e0b132d2ddfd?crop=entropy&cs=tinysrgb&fit=max&fm=jpg&ixid=M3w3Nzg4Nzd8MHwxfHNlYXJjaHwxfHxiZWF1dGlmdWwlMjBtaWRkbGUlMjBlYXN0ZXJuJTIwd29tYW4lMjBwb3J0cmFpdHxlbnwxfHx8fDE3NTgyMzYzOTV8MA&ixlib=rb-4.1.0&q=80&w=1080"],
          interests: ["Business", "Travel", "Luxury"],
          verified: true,
          premium: true,
          rating: 4.8,
          height: Math.floor(Math.random() * 200) + 300
        },
        {
          id: 6,
          name: "Sophia",
          age: 23,
          location: "🇮🇹 Rome, Italy",
          bio: "Fashion design student and vintage collector. Passionate about sustainable fashion and timeless elegance.",
          images: ["https://images.unsplash.com/photo-1577091937200-29915517c0bd?crop=entropy&cs=tinysrgb&fit=max&fm=jpg&ixid=M3w3Nzg4Nzd8MHwxfHNlYXJjaHwxfHxiZWF1dGlmdWwlMjBldXJvcGVhbiUyMHdvbWFuJTIwcG9ydHJhaXR8ZW58MXx8fHwxNzU4MjM2Mzk4fDA&ixlib=rb-4.1.0&q=80&w=1080"],
          interests: ["Fashion", "Sustainability", "Vintage"],
          verified: false,
          premium: false,
          rating: 4.5,
          height: Math.floor(Math.random() * 200) + 300
        },
        {
          id: 7,
          name: "Priya",
          age: 28,
          location: "🇮🇳 Mumbai, India",
          bio: "Bollywood choreographer and yoga instructor. Believe in expressing joy through movement and mindfulness.",
          images: ["https://images.unsplash.com/photo-1734764627104-6ad22c48af6a?crop=entropy&cs=tinysrgb&fit=max&fm=jpg&ixid=M3w3Nzg4Nzd8MHwxfHNlYXJjaHwxfHxiZWF1dGlmdWwlMjBpbmRpYW4lMjB3b21hbiUyMHBvcnRyYWl0fGVufDF8fHx8MTc1ODIzNjQwOHww&ixlib=rb-4.1.0&q=80&w=1080"],
          interests: ["Dance", "Yoga", "Mindfulness"],
          verified: true,
          premium: true,
          rating: 4.9,
          height: Math.floor(Math.random() * 200) + 300
        },
        {
          id: 8,
          name: "Astrid",
          age: 29,
          location: "🇸🇪 Stockholm, Sweden",
          bio: "Sustainable design engineer and Northern Lights chaser. Love creating beautiful solutions for a better world.",
          images: ["https://images.unsplash.com/photo-1625566303386-11dab79684a5?crop=entropy&cs=tinysrgb&fit=max&fm=jpg&ixid=M3w3Nzg4Nzd8MHwxfHNlYXJjaHwxfHxiZWF1dGlmdWwlMjBzY2FuZGluYXZpYW4lMjB3b21hbiUyMHBvcnRyYWl0fGVufDF8fHx8MTc1ODIzNjQxNnww&ixlib=rb-4.1.0&q=80&w=1080"],
          interests: ["Design", "Sustainability", "Nature"],
          verified: true,
          premium: false,
          rating: 4.7,
          height: Math.floor(Math.random() * 200) + 300
        },
        {
          id: 9,
          name: "Camila",
          age: 26,
          location: "🇧🇷 Rio de Janeiro, Brazil",
          bio: "Capoeira instructor and marine biologist. Love the ocean, movement, and protecting our blue planet.",
          images: ["https://images.unsplash.com/photo-1659500599203-fb8535fcdf37?crop=entropy&cs=tinysrgb&fit=max&fm=jpg&ixid=M3w3Nzg4Nzd8MHwxfHNlYXJjaHwxfHxiZWF1dGlmdWwlMjBicmF6aWxpYW4lMjB3b21hbiUyMHBvcnRyYWl0fGVufDF8fHx8MTc1ODIzNjQyNHww&ixlib=rb-4.1.0&q=80&w=1080"],
          interests: ["Capoeira", "Ocean", "Biology"],
          verified: false,
          premium: true,
          rating: 4.8,
          height: Math.floor(Math.random() * 200) + 300
        },
        {
          id: 10,
          name: "Sophie",
          age: 24,
          location: "🇫🇷 Paris, France",
          bio: "Pastry chef and art history student. Love creating sweet masterpieces and exploring museums.",
          images: ["https://images.unsplash.com/photo-1524250502761-1ac6f2e30d43?w=400&h=600&fit=crop&crop=face"],
          interests: ["Baking", "Art History", "Museums"],
          verified: true,
          premium: false,
          rating: 4.6,
          height: Math.floor(Math.random() * 200) + 300
        },
        {
          id: 11,
          name: "Zara",
          age: 27,
          location: "🇿🇦 Cape Town, South Africa",
          bio: "Wine sommelier and adventure hiking guide. Love exploring vineyards and mountain trails.",
          images: ["https://images.unsplash.com/photo-1504199367641-aba8151af406?w=400&h=600&fit=crop&crop=face"],
          interests: ["Wine", "Hiking", "Adventure"],
          verified: true,
          premium: true,
          rating: 4.9,
          height: Math.floor(Math.random() * 200) + 300
        },
        {
          id: 12,
          name: "Yuki",
          age: 25,
          location: "🇰🇷 Seoul, South Korea",
          bio: "K-pop choreographer and skincare enthusiast. Passionate about beauty, music, and self-expression.",
          images: ["https://images.unsplash.com/photo-1531746020798-e6953c6e8e04?w=400&h=600&fit=crop&crop=face"],
          interests: ["K-pop", "Skincare", "Dance"],
          verified: false,
          premium: false,
          rating: 4.7,
          height: Math.floor(Math.random() * 200) + 300
        },
        {
          id: 13,
          name: "Elena",
          age: 28,
          location: "🇷🇺 Moscow, Russia",
          bio: "Classical pianist and literature professor. Love sharing the beauty of music and timeless stories.",
          images: ["https://images.unsplash.com/photo-1488426862026-3ee34a7d66df?w=400&h=600&fit=crop&crop=face"],
          interests: ["Piano", "Literature", "Classical Arts"],
          verified: true,
          premium: true,
          rating: 4.8,
          height: Math.floor(Math.random() * 200) + 300
        },
        {
          id: 14,
          name: "Lily",
          age: 26,
          location: "🇦🇺 Sydney, Australia",
          bio: "Surf instructor and marine conservation advocate. Love the beach lifestyle and protecting ocean wildlife.",
          images: ["https://images.unsplash.com/photo-1517841905240-472988babdf9?w=400&h=600&fit=crop&crop=face"],
          interests: ["Surfing", "Marine Conservation", "Beach Life"],
          verified: false,
          premium: false,
          rating: 4.5,
          height: Math.floor(Math.random() * 200) + 300
        },
        {
          id: 15,
          name: "Aria",
          age: 29,
          location: "🇨🇦 Vancouver, Canada",
          bio: "Documentary filmmaker and environmental activist. Dedicated to telling stories that inspire positive change.",
          images: ["https://images.unsplash.com/photo-1507003211169-0a1dd7228f2d?w=400&h=600&fit=crop&crop=face"],
          interests: ["Filmmaking", "Environment", "Storytelling"],
          verified: true,
          premium: true,
          rating: 4.9,
          height: Math.floor(Math.random() * 200) + 300
        },
        {
          id: 16,
          name: "Mei",
          age: 24,
          location: "🇨🇳 Shanghai, China",
          bio: "Digital artist and tech entrepreneur. Creating the future through innovation and creativity.",
          images: ["https://images.unsplash.com/photo-1529626455594-4ff0802cfb7e?w=400&h=600&fit=crop&crop=face"],
          interests: ["Digital Art", "Technology", "Innovation"],
          verified: true,
          premium: true,
          rating: 4.7,
          height: Math.floor(Math.random() * 200) + 300
        },
        {
          id: 17,
          name: "Chiara",
          age: 27,
          location: "🇨🇭 Zurich, Switzerland",
          bio: "Chocolatier and alpine hiking enthusiast. Love crafting sweet delights and conquering mountain peaks.",
          images: ["https://images.unsplash.com/photo-1539571696357-5a69c17a67c6?w=400&h=600&fit=crop&crop=face"],
          interests: ["Chocolate", "Hiking", "Mountains"],
          verified: false,
          premium: false,
          rating: 4.6,
          height: Math.floor(Math.random() * 200) + 300
        },
        {
          id: 18,
          name: "Nadia",
          age: 26,
          location: "🇪🇬 Cairo, Egypt",
          bio: "Archaeologist and history curator. Passionate about uncovering ancient mysteries and preserving heritage.",
          images: ["https://images.unsplash.com/photo-1580489944761-15a19d654956?w=400&h=600&fit=crop&crop=face"],
          interests: ["Archaeology", "History", "Culture"],
          verified: true,
          premium: true,
          rating: 4.8,
          height: Math.floor(Math.random() * 200) + 300
        },
        {
          id: 19,
          name: "Freya",
          age: 25,
          location: "🇳🇴 Oslo, Norway",
          bio: "Marine biologist and fjord kayaker. Love exploring the ocean depths and Nordic wilderness.",
          images: ["https://images.unsplash.com/photo-1541271696563-3be2f555fc4e?w=400&h=600&fit=crop&crop=face"],
          interests: ["Marine Biology", "Kayaking", "Nature"],
          verified: true,
          premium: false,
          rating: 4.7,
          height: Math.floor(Math.random() * 200) + 300
        },
        {
          id: 20,
          name: "Ana",
          age: 28,
          location: "🇵🇹 Lisbon, Portugal",
          bio: "Fado singer and Portuguese cuisine chef. Love sharing my culture through music and food.",
          images: ["https://images.unsplash.com/photo-1564485377539-4af72d1f6a2f?w=400&h=600&fit=crop&crop=face"],
          interests: ["Music", "Cooking", "Culture"],
          verified: false,
          premium: true,
          rating: 4.9,
          height: Math.floor(Math.random() * 200) + 300
        },
        {
          id: 21,
          name: "Nia",
          age: 24,
          location: "🇬🇭 Accra, Ghana",
          bio: "Fashion designer and cultural ambassador. Creating modern African fashion with traditional roots.",
          images: ["https://images.unsplash.com/photo-1595152772835-219674b2a8a6?w=400&h=600&fit=crop&crop=face"],
          interests: ["Fashion Design", "Culture", "Arts"],
          verified: true,
          premium: false,
          rating: 4.6,
          height: Math.floor(Math.random() * 200) + 300
        },
        {
          id: 22,
          name: "Katarina",
          age: 29,
          location: "🇨🇿 Prague, Czech Republic",
          bio: "Glassblowing artist and classical violinist. Love creating beauty through both craft and music.",
          images: ["https://images.unsplash.com/photo-1542206395-9feb3edaa68d?w=400&h=600&fit=crop&crop=face"],
          interests: ["Glassblowing", "Violin", "Arts"],
          verified: true,
          premium: true,
          rating: 4.8,
          height: Math.floor(Math.random() * 200) + 300
        },
        {
          id: 23,
          name: "Luna",
          age: 23,
          location: "🇲🇽 Mexico City, Mexico",
          bio: "Street artist and salsa dancer. Expressing passion through color, movement, and rhythm.",
          images: ["https://images.unsplash.com/photo-1508214751196-bcfd4ca60f91?w=400&h=600&fit=crop&crop=face"],
          interests: ["Street Art", "Salsa", "Dance"],
          verified: false,
          premium: false,
          rating: 4.7,
          height: Math.floor(Math.random() * 200) + 300
        },
        {
          id: 24,
          name: "Aisha",
          age: 26,
          location: "🇲🇦 Marrakech, Morocco",
          bio: "Spice merchant and culinary tour guide. Love sharing the aromatic flavors of Moroccan cuisine.",
          images: ["https://images.unsplash.com/photo-1616683693068-3dcf6d36a1c9?w=400&h=600&fit=crop&crop=face"],
          interests: ["Cuisine", "Spices", "Culture"],
          verified: true,
          premium: true,
          rating: 4.9,
          height: Math.floor(Math.random() * 200) + 300
        },
        {
          id: 25,
          name: "Ingrid",
          age: 27,
          location: "🇩🇰 Copenhagen, Denmark",
          bio: "Sustainable fashion advocate and bicycle tour guide. Love eco-friendly living and Danish design.",
          images: ["https://images.unsplash.com/photo-1552374196-c4e7ffc6e126?w=400&h=600&fit=crop&crop=face"],
          interests: ["Sustainability", "Cycling", "Design"],
          verified: true,
          premium: false,
          rating: 4.6,
          height: Math.floor(Math.random() * 200) + 300
        },
        {
          id: 26,
          name: "Valentina",
          age: 25,
          location: "🇦🇷 Buenos Aires, Argentina",
          bio: "Tango instructor and wine expert. Passionate about Argentine culture, dance, and Malbec.",
          images: ["https://images.unsplash.com/photo-1529339944280-4c27f7d5e4f5?w=400&h=600&fit=crop&crop=face"],
          interests: ["Tango", "Wine", "Dance"],
          verified: false,
          premium: true,
          rating: 4.8,
          height: Math.floor(Math.random() * 200) + 300
        },
        {
          id: 27,
          name: "Thea",
          age: 28,
          location: "🇬🇷 Athens, Greece",
          bio: "Olive oil sommelier and ancient history enthusiast. Love connecting past wisdom with present flavor.",
          images: ["https://images.unsplash.com/photo-1591084728795-1149f32d9866?w=400&h=600&fit=crop&crop=face"],
          interests: ["Olive Oil", "History", "Cuisine"],
          verified: true,
          premium: false,
          rating: 4.7,
          height: Math.floor(Math.random() * 200) + 300
        },
        {
          id: 28,
          name: "Suki",
          age: 24,
          location: "🇹🇭 Bangkok, Thailand",
          bio: "Thai massage therapist and street food blogger. Love sharing wellness and authentic flavors.",
          images: ["https://images.unsplash.com/photo-1582610285985-a42d9193f2fd?w=400&h=600&fit=crop&crop=face"],
          interests: ["Wellness", "Food", "Travel"],
          verified: true,
          premium: true,
          rating: 4.9,
          height: Math.floor(Math.random() * 200) + 300
        },
        {
          id: 29,
          name: "Amelia",
          age: 26,
          location: "🇳🇿 Auckland, New Zealand",
          bio: "Adventure photographer and bungee jumping instructor. Love capturing thrilling moments.",
          images: ["https://images.unsplash.com/photo-1598439210625-5067c578f3f6?w=400&h=600&fit=crop&crop=face"],
          interests: ["Photography", "Adventure", "Adrenaline"],
          verified: false,
          premium: false,
          rating: 4.6,
          height: Math.floor(Math.random() * 200) + 300
        },
        {
          id: 30,
          name: "Leila",
          age: 27,
          location: "🇱🇧 Beirut, Lebanon",
          bio: "Jazz singer and mezze chef. Blending smooth melodies with Mediterranean flavors.",
          images: ["https://images.unsplash.com/photo-1613005798967-632017e477c8?w=400&h=600&fit=crop&crop=face"],
          interests: ["Jazz", "Cooking", "Music"],
          verified: true,
          premium: true,
          rating: 4.8,
          height: Math.floor(Math.random() * 200) + 300
        },
        {
          id: 31,
          name: "Hana",
          age: 25,
          location: "🇻🇳 Hanoi, Vietnam",
          bio: "Coffee roaster and traditional embroidery artist. Love preserving heritage through crafts.",
          images: ["https://images.unsplash.com/photo-1572804013309-59a88b7e92f1?w=400&h=600&fit=crop&crop=face"],
          interests: ["Coffee", "Embroidery", "Arts"],
          verified: true,
          premium: false,
          rating: 4.7,
          height: Math.floor(Math.random() * 200) + 300
        },
        {
          id: 32,
          name: "Nina",
          age: 29,
          location: "🇵🇱 Warsaw, Poland",
          bio: "Opera singer and pierogi maker. Passionate about Polish culture and classical music.",
          images: ["https://images.unsplash.com/photo-1521146764736-56c929d59c83?w=400&h=600&fit=crop&crop=face"],
          interests: ["Opera", "Cooking", "Music"],
          verified: false,
          premium: true,
          rating: 4.9,
          height: Math.floor(Math.random() * 200) + 300
        },
        {
          id: 33,
          name: "Isla",
          age: 23,
          location: "🇮🇪 Dublin, Ireland",
          bio: "Celtic harp player and Irish whiskey expert. Love sharing traditional Irish culture.",
          images: ["https://images.unsplash.com/photo-1605462863863-10d9e47e15ee?w=400&h=600&fit=crop&crop=face"],
          interests: ["Harp", "Whiskey", "Culture"],
          verified: true,
          premium: false,
          rating: 4.6,
          height: Math.floor(Math.random() * 200) + 300
        },
        {
          id: 34,
          name: "Carmen",
          age: 28,
          location: "🇨🇴 Bogotá, Colombia",
          bio: "Coffee expert and cumbia dancer. Love Colombian rhythms and the perfect espresso shot.",
          images: ["https://images.unsplash.com/photo-1603460000768-4a86bb5a95fb?w=400&h=600&fit=crop&crop=face"],
          interests: ["Coffee", "Dance", "Culture"],
          verified: true,
          premium: true,
          rating: 4.8,
          height: Math.floor(Math.random() * 200) + 300
        },
        {
          id: 35,
          name: "Zoe",
          age: 24,
          location: "🇬🇧 London, UK",
          bio: "Theater director and afternoon tea enthusiast. Love British culture and the performing arts.",
          images: ["https://images.unsplash.com/photo-1619895862022-09114b41f16f?w=400&h=600&fit=crop&crop=face"],
          interests: ["Theater", "Tea", "Arts"],
          verified: false,
          premium: false,
          rating: 4.7,
          height: Math.floor(Math.random() * 200) + 300
        },
        {
          id: 36,
          name: "Mira",
          age: 26,
          location: "🇮🇱 Tel Aviv, Israel",
          bio: "Tech innovator and hummus connoisseur. Blending innovation with Mediterranean traditions.",
          images: ["https://images.unsplash.com/photo-1604004555489-723a93d6ce74?w=400&h=600&fit=crop&crop=face"],
          interests: ["Technology", "Food", "Innovation"],
          verified: true,
          premium: true,
          rating: 4.9,
          height: Math.floor(Math.random() * 200) + 300
        },
        {
          id: 37,
          name: "Sana",
          age: 27,
          location: "🇵🇰 Karachi, Pakistan",
          bio: "Textile designer and classical dancer. Love creating beautiful patterns inspired by tradition.",
          images: ["https://images.unsplash.com/photo-1494790108377-be9c29b29330?w=400&h=600&fit=crop&crop=face"],
          interests: ["Textiles", "Dance", "Design"],
          verified: true,
          premium: false,
          rating: 4.6,
          height: Math.floor(Math.random() * 200) + 300
        },
        {
          id: 38,
          name: "Petra",
          age: 25,
          location: "🇭🇷 Dubrovnik, Croatia",
          bio: "Sailing instructor and seafood chef. Love the Adriatic Sea and Mediterranean cuisine.",
          images: ["https://images.unsplash.com/photo-1499557354967-2b2d8910bcca?w=400&h=600&fit=crop&crop=face"],
          interests: ["Sailing", "Cooking", "Ocean"],
          verified: false,
          premium: true,
          rating: 4.8,
          height: Math.floor(Math.random() * 200) + 300
        },
        {
          id: 39,
          name: "Aaliyah",
          age: 28,
          location: "🇯🇲 Kingston, Jamaica",
          bio: "Reggae musician and jerk chicken master. Love island vibes, music, and Caribbean flavors.",
          images: ["https://images.unsplash.com/photo-1485178575877-1a13bf489dfe?w=400&h=600&fit=crop&crop=face"],
          interests: ["Reggae", "Music", "Cooking"],
          verified: true,
          premium: false,
          rating: 4.7,
          height: Math.floor(Math.random() * 200) + 300
        },
        {
          id: 40,
          name: "Kira",
          age: 24,
          location: "🇦🇹 Vienna, Austria",
          bio: "Classical ballet dancer and Sachertorte baker. Love Viennese elegance and sweet traditions.",
          images: ["https://images.unsplash.com/photo-1502716119720-b23a93e5fe1b?w=400&h=600&fit=crop&crop=face"],
          interests: ["Ballet", "Baking", "Classical Arts"],
          verified: true,
          premium: true,
          rating: 4.9,
          height: Math.floor(Math.random() * 200) + 300
        },
        {
          id: 41,
          name: "Rania",
          age: 26,
          location: "🇸🇦 Riyadh, Saudi Arabia",
          bio: "Calligraphy artist and perfume creator. Love blending ancient arts with modern aesthetics.",
          images: ["https://images.unsplash.com/photo-1601412436009-d964bd02edbc?w=400&h=600&fit=crop&crop=face"],
          interests: ["Calligraphy", "Perfumes", "Arts"],
          verified: false,
          premium: false,
          rating: 4.6,
          height: Math.floor(Math.random() * 200) + 300
        },
        {
          id: 42,
          name: "Olivia",
          age: 27,
          location: "🇧🇪 Brussels, Belgium",
          bio: "Chocolate sommelier and comic book illustrator. Love Belgian culture and creative storytelling.",
          images: ["https://images.unsplash.com/photo-1543610892-0b1f7e6d8ac1?w=400&h=600&fit=crop&crop=face"],
          interests: ["Chocolate", "Comics", "Arts"],
          verified: true,
          premium: true,
          rating: 4.8,
          height: Math.floor(Math.random() * 200) + 300
        },
        {
          id: 43,
          name: "Indira",
          age: 25,
          location: "🇳🇵 Kathmandu, Nepal",
          bio: "Mountain guide and meditation instructor. Love Himalayan adventures and mindful living.",
          images: ["https://images.unsplash.com/photo-1615109398623-88346a601842?w=400&h=600&fit=crop&crop=face"],
          interests: ["Mountaineering", "Meditation", "Nature"],
          verified: true,
          premium: false,
          rating: 4.7,
          height: Math.floor(Math.random() * 200) + 300
        },
        {
          id: 44,
          name: "Eva",
          age: 29,
          location: "🇭🇺 Budapest, Hungary",
          bio: "Thermal spa therapist and goulash chef. Love Hungarian traditions and wellness culture.",
          images: ["https://images.unsplash.com/photo-1573496359142-b8d87734a5a2?w=400&h=600&fit=crop&crop=face"],
          interests: ["Wellness", "Cooking", "Culture"],
          verified: false,
          premium: true,
          rating: 4.9,
          height: Math.floor(Math.random() * 200) + 300
        },
        {
          id: 45,
          name: "Jade",
          age: 23,
          location: "🇸🇬 Singapore, Singapore",
          bio: "Food blogger and urban gardener. Love exploring Singapore's diverse cuisine and green spaces.",
          images: ["https://images.unsplash.com/photo-1607746882042-944635dfe10e?w=400&h=600&fit=crop&crop=face"],
          interests: ["Food", "Gardening", "Travel"],
          verified: true,
          premium: false,
          rating: 4.6,
          height: Math.floor(Math.random() * 200) + 300
        }
      ];
      
      setProfiles(sampleProfiles);
      setLoading(false);
    };

    // Simulate loading
    setTimeout(generateProfiles, 800);
  }, []);

  // Auto-scroll functionality for Row 1 (left to right)
  useEffect(() => {
    if (!isScrolling || !scrollContainerRef1.current) return;

    const scrollContainer = scrollContainerRef1.current;
    let scrollPosition = 0;
    const scrollSpeed = 1;

    const animate = () => {
      if (!scrollContainer) return;
      
      scrollPosition += scrollSpeed;
      scrollContainer.scrollLeft = scrollPosition;

      if (scrollPosition >= scrollContainer.scrollWidth / 2) {
        scrollPosition = 0;
      }

      if (isScrolling) {
        animationId = requestAnimationFrame(animate);
      }
    };

    let animationId = requestAnimationFrame(animate);
    return () => {
      if (animationId) {
        cancelAnimationFrame(animationId);
      }
    };
  }, [isScrolling, profiles]);

  // Auto-scroll functionality for Row 2 (right to left)
  useEffect(() => {
    if (!isScrolling || !scrollContainerRef2.current) return;

    const scrollContainer = scrollContainerRef2.current;
    // Set initial position after a small delay to ensure content is loaded
    scrollContainer.scrollLeft = scrollContainer.scrollWidth / 2;
    let scrollPosition = scrollContainer.scrollWidth / 2;
    const scrollSpeed = 1;

    const animate = () => {
      if (!scrollContainer) return;
      
      scrollPosition -= scrollSpeed;
      scrollContainer.scrollLeft = scrollPosition;

      if (scrollPosition <= 0) {
        scrollPosition = scrollContainer.scrollWidth / 2;
      }

      if (isScrolling) {
        animationId = requestAnimationFrame(animate);
      }
    };

    let animationId = requestAnimationFrame(animate);
    return () => {
      if (animationId) {
        cancelAnimationFrame(animationId);
      }
    };
  }, [isScrolling, profiles]);

  // Auto-scroll functionality for Row 3 (left to right)
  useEffect(() => {
    if (!isScrolling || !scrollContainerRef3.current) return;

    const scrollContainer = scrollContainerRef3.current;
    let scrollPosition = 0;
    const scrollSpeed = 1;

    const animate = () => {
      if (!scrollContainer) return;
      
      scrollPosition += scrollSpeed;
      scrollContainer.scrollLeft = scrollPosition;

      if (scrollPosition >= scrollContainer.scrollWidth / 2) {
        scrollPosition = 0;
      }

      if (isScrolling) {
        animationId = requestAnimationFrame(animate);
      }
    };

    let animationId = requestAnimationFrame(animate);
    return () => {
      if (animationId) {
        cancelAnimationFrame(animationId);
      }
    };
  }, [isScrolling, profiles]);

  // Auto-scroll functionality for Row 4 (right to left)
  useEffect(() => {
    if (!isScrolling || !scrollContainerRef4.current) return;

    const scrollContainer = scrollContainerRef4.current;
    // Set initial position after a small delay to ensure content is loaded
    scrollContainer.scrollLeft = scrollContainer.scrollWidth / 2;
    let scrollPosition = scrollContainer.scrollWidth / 2;
    const scrollSpeed = 1;

    const animate = () => {
      if (!scrollContainer) return;
      
      scrollPosition -= scrollSpeed;
      scrollContainer.scrollLeft = scrollPosition;

      if (scrollPosition <= 0) {
        scrollPosition = scrollContainer.scrollWidth / 2;
      }

      if (isScrolling) {
        animationId = requestAnimationFrame(animate);
      }
    };

    let animationId = requestAnimationFrame(animate);
    return () => {
      if (animationId) {
        cancelAnimationFrame(animationId);
      }
    };
  }, [isScrolling, profiles]);

  const handleLike = (profileId: number) => {
    console.log(`Liked profile ${profileId}`);
    // In a real app, this would send a like to the backend
  };

  const handleViewProfile = (profileId: number) => {
    console.log(`Viewing profile ${profileId}`);
    // In a real app, this would navigate to the full profile view
  };

  const toggleScroll = () => {
    setIsScrolling(!isScrolling);
  };

  return (
    <div className="min-h-screen bg-background">
      {/* Header */}
      <Header currentView="landing" onNavigate={onNavigate} />
      
      {/* Hero Section */}
      <div className="bg-gradient-to-br from-purple-600 via-pink-600 to-red-600 dark:from-purple-700 dark:via-pink-700 dark:to-red-700 text-white py-16 relative overflow-hidden">
        {/* Animated background elements */}
        <motion.div 
          className="absolute inset-0 opacity-20"
          animate={{
            background: [
              'radial-gradient(circle at 20% 30%, rgba(255,255,255,0.3) 0%, transparent 50%)',
              'radial-gradient(circle at 80% 70%, rgba(255,255,255,0.3) 0%, transparent 50%)',
              'radial-gradient(circle at 40% 80%, rgba(255,255,255,0.3) 0%, transparent 50%)',
              'radial-gradient(circle at 20% 30%, rgba(255,255,255,0.3) 0%, transparent 50%)'
            ]
          }}
          transition={{
            duration: 8,
            repeat: Infinity,
            ease: "easeInOut"
          }}
        />
        
        <div className="max-w-7xl mx-auto px-6 text-center relative z-10">
          <motion.div
            initial={{ opacity: 0, y: 30 }}
            animate={{ opacity: 1, y: 0 }}
            transition={{ duration: 0.8 }}
          >
            <h1 className="text-5xl md:text-6xl lg:text-7xl font-bold mb-6">
              <motion.span
                animate={{ 
                  scale: [1, 1.02, 1],
                }}
                transition={{
                  duration: 4,
                  repeat: Infinity,
                  ease: "easeInOut"
                }}
              >
                Find Your ❤️ Perfect Match
              </motion.span>
            </h1>
            <p className="text-xl md:text-2xl text-white/90 mb-8 max-w-3xl mx-auto">
              Join millions of singles worldwide and discover meaningful connections on SinglesZonne
            </p>
            <div className="flex flex-col sm:flex-row gap-4 justify-center">
              <Button 
                size="lg"
                className="bg-white text-purple-600 hover:bg-white/90 text-lg px-8 py-4"
                onClick={() => onNavigate("register")}
              >
                <Heart className="w-5 h-5 mr-2" />
                Join Now - It's Free!
              </Button>
              <Button 
                size="lg"
                variant="outline"
                className="border-white text-white hover:bg-white/10 text-lg px-8 py-4"
                onClick={() => onNavigate("search")}
              >
                Explore Profiles
              </Button>
            </div>
          </motion.div>
        </div>
      </div>

      {/* Stats Section */}
      <div className="py-12 bg-muted/30">
        <div className="max-w-7xl mx-auto px-6">
          <div className="grid grid-cols-2 md:grid-cols-4 gap-8 text-center">
            <motion.div
              initial={{ opacity: 0, y: 20 }}
              animate={{ opacity: 1, y: 0 }}
              transition={{ duration: 0.5, delay: 0.1 }}
            >
              <div className="text-3xl md:text-4xl font-bold text-purple-600 mb-2">2.5M+</div>
              <div className="text-muted-foreground">Active Members</div>
            </motion.div>
            <motion.div
              initial={{ opacity: 0, y: 20 }}
              animate={{ opacity: 1, y: 0 }}
              transition={{ duration: 0.5, delay: 0.2 }}
            >
              <div className="text-3xl md:text-4xl font-bold text-pink-600 mb-2">500K+</div>
              <div className="text-muted-foreground">Successful Matches</div>
            </motion.div>
            <motion.div
              initial={{ opacity: 0, y: 20 }}
              animate={{ opacity: 1, y: 0 }}
              transition={{ duration: 0.5, delay: 0.3 }}
            >
              <div className="text-3xl md:text-4xl font-bold text-red-600 mb-2">45+</div>
              <div className="text-muted-foreground">Countries</div>
            </motion.div>
            <motion.div
              initial={{ opacity: 0, y: 20 }}
              animate={{ opacity: 1, y: 0 }}
              transition={{ duration: 0.5, delay: 0.4 }}
            >
              <div className="text-3xl md:text-4xl font-bold text-purple-600 mb-2">98%</div>
              <div className="text-muted-foreground">Satisfaction Rate</div>
            </motion.div>
          </div>
        </div>
      </div>

      {/* Featured Profiles Section */}
      <div className="py-16 bg-muted/20">
        {/* Header section with container */}
        <div className="max-w-7xl mx-auto px-6 mb-8">
          <motion.div 
            className="text-center"
            initial={{ opacity: 0, y: 20 }}
            animate={{ opacity: 1, y: 0 }}
            transition={{ duration: 0.6 }}
          >
            <h2 className="text-4xl md:text-5xl font-bold mb-4">
              Meet Amazing Singles
            </h2>
            <p className="text-xl text-muted-foreground max-w-2xl mx-auto">
              Discover incredible people from around the world ready to connect and find love
            </p>
          </motion.div>
        </div>

        {/* Auto-scrolling carousel controls */}
        <div className="max-w-7xl mx-auto px-6 mb-8">
          <div className="flex justify-center items-center gap-4">
            <Button
              variant="outline"
              size="sm"
              onClick={toggleScroll}
              className="rounded-full px-6"
            >
              {isScrolling ? (
                <>
                  <Pause className="w-4 h-4 mr-2" />
                  Pause Scrolling
                </>
              ) : (
                <>
                  <Play className="w-4 h-4 mr-2" />
                  Resume Scrolling
                </>
              )}
            </Button>
          </div>
        </div>

        {loading ? (
          <div className="space-y-8">
            {Array.from({ length: 4 }).map((_, rowIndex) => (
              <div key={rowIndex} className="flex gap-6 px-6 overflow-x-hidden">
                {Array.from({ length: 8 }).map((_, index) => (
                  <div key={index} className="flex-shrink-0 w-[300px] animate-pulse">
                    <div className="bg-muted rounded-lg h-96 mb-4"></div>
                    <div className="bg-muted rounded h-4 mb-2"></div>
                    <div className="bg-muted rounded h-4 w-3/4"></div>
                  </div>
                ))}
              </div>
            ))}
          </div>
        ) : (
          <div className="space-y-8">
            {/* Row 1 - Left to Right */}
            <div className="relative">
              <div 
                ref={scrollContainerRef1}
                className="flex gap-6 px-6 overflow-x-auto scrollbar-hide scroll-smooth"
                style={{ 
                  scrollbarWidth: 'none',
                  msOverflowStyle: 'none'
                }}
                onMouseEnter={() => setIsScrolling(false)}
                onMouseLeave={() => setIsScrolling(true)}
              >
                {[...profiles.slice(0, 12), ...profiles.slice(0, 12)].map((profile, index) => (
                  <ProfileCard key={`row1-${profile.id}-${index}`} profile={profile} index={index} onViewProfile={handleViewProfile} onLike={handleLike} />
                ))}
              </div>
            </div>

            {/* Row 2 - Right to Left */}
            <div className="relative">
              <div 
                ref={scrollContainerRef2}
                className="flex gap-6 px-6 overflow-x-auto scrollbar-hide scroll-smooth"
                style={{ 
                  scrollbarWidth: 'none',
                  msOverflowStyle: 'none'
                }}
                onMouseEnter={() => setIsScrolling(false)}
                onMouseLeave={() => setIsScrolling(true)}
              >
                {[...profiles.slice(12, 24), ...profiles.slice(12, 24)].map((profile, index) => (
                  <ProfileCard key={`row2-${profile.id}-${index}`} profile={profile} index={index} onViewProfile={handleViewProfile} onLike={handleLike} />
                ))}
              </div>
            </div>

            {/* Row 3 - Left to Right */}
            <div className="relative">
              <div 
                ref={scrollContainerRef3}
                className="flex gap-6 px-6 overflow-x-auto scrollbar-hide scroll-smooth"
                style={{ 
                  scrollbarWidth: 'none',
                  msOverflowStyle: 'none'
                }}
                onMouseEnter={() => setIsScrolling(false)}
                onMouseLeave={() => setIsScrolling(true)}
              >
                {[...profiles.slice(24, 36), ...profiles.slice(24, 36)].map((profile, index) => (
                  <ProfileCard key={`row3-${profile.id}-${index}`} profile={profile} index={index} onViewProfile={handleViewProfile} onLike={handleLike} />
                ))}
              </div>
            </div>

            {/* Row 4 - Right to Left */}
            <div className="relative">
              <div 
                ref={scrollContainerRef4}
                className="flex gap-6 px-6 overflow-x-auto scrollbar-hide scroll-smooth"
                style={{ 
                  scrollbarWidth: 'none',
                  msOverflowStyle: 'none'
                }}
                onMouseEnter={() => setIsScrolling(false)}
                onMouseLeave={() => setIsScrolling(true)}
              >
                {[...profiles.slice(36, 45), ...profiles.slice(0, 3), ...profiles.slice(36, 45), ...profiles.slice(0, 3)].map((profile, index) => (
                  <ProfileCard key={`row4-${profile.id}-${index}`} profile={profile} index={index} onViewProfile={handleViewProfile} onLike={handleLike} />
                ))}
              </div>
            </div>
          </div>
        )}

        {/* Call to Action section with container */}
        <div className="max-w-7xl mx-auto px-6">
          <motion.div 
            className="text-center mt-16"
            initial={{ opacity: 0, y: 20 }}
            animate={{ opacity: 1, y: 0 }}
            transition={{ duration: 0.6, delay: 0.3 }}
          >
            <Button 
              size="lg"
              className="bg-gradient-to-r from-purple-600 to-pink-600 hover:from-purple-700 hover:to-pink-700 text-white text-lg px-8 py-4"
              onClick={() => onNavigate("register")}
            >
              <Heart className="w-5 h-5 mr-2" />
              Start Your Love Story Today
            </Button>
            <p className="text-muted-foreground mt-4">
              Join thousands of singles who found love on SinglesZonne
            </p>
          </motion.div>
        </div>
      </div>
      
      {/* Footer */}
      <Footer onNavigate={onNavigate} onOpenCookieConsent={onOpenCookieConsent} />
    </div>
  );
}