import { useState } from "react";
import { Header } from "./Header";
import { Footer } from "./Footer";
import { Button } from "./ui/button";
import { Card } from "./ui/card";
import { Badge } from "./ui/badge";
import { Input } from "./ui/input";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "./ui/tabs";
import { Alert, AlertDescription } from "./ui/alert";
import { 
  Users, 
  Search, 
  MapPin, 
  Calendar, 
  Coffee, 
  Music, 
  Camera, 
  Dumbbell,
  Book,
  Plane,
  Heart,
  MessageCircle,
  UserPlus,
  Crown,
  Lock,
  AlertTriangle,
  Clock
} from "lucide-react";
import { motion } from "motion/react";
import { ImageWithFallback } from "./figma/ImageWithFallback";
import { useUser } from "./UserProvider";
import { RoleSwitcher } from "./RoleSwitcher";

interface GroupsPageProps {
  onNavigate: (view: string) => void;
  onOpenCookieConsent?: () => void;
}

interface Group {
  id: number;
  name: string;
  description: string;
  image: string;
  category: "interests" | "location" | "events" | "activities";
  memberCount: number;
  isJoined: boolean;
  isPrivate: boolean;
  location?: string;
  nextEvent?: string;
  recentActivity: string;
  tags: string[];
}

interface Event {
  id: number;
  title: string;
  description: string;
  image: string;
  location: string;
  date: string;
  time: string;
  price: string;
  spotsLeft: number;
  totalSpots: number;
  isAttending: boolean;
  category: "speed-dating" | "mixer" | "activity" | "workshop";
  ageRange?: string;
  requirements?: string[];
}

export function GroupsPage({ onNavigate, onOpenCookieConsent }: GroupsPageProps) {
  const { user } = useUser();
  const [activeTab, setActiveTab] = useState("discover");
  const [searchQuery, setSearchQuery] = useState("");
  const [eventSearchQuery, setEventSearchQuery] = useState("");
  
  // Check if user can access groups (Premium/Platinum only)
  const canAccessGroups = user?.membershipTier === "premium" || user?.membershipTier === "platinum";
  
  const [groups, setGroups] = useState<Group[]>([
    {
      id: 1,
      name: "Coffee Lovers NYC",
      description: "Weekly coffee meetups around Manhattan. Share your favorite spots and meet fellow coffee enthusiasts!",
      image: "https://images.unsplash.com/photo-1511920170033-f8396924c348?w=400&h=300&fit=crop",
      category: "interests",
      memberCount: 127,
      isJoined: true,
      isPrivate: false,
      location: "New York, NY",
      nextEvent: "Saturday 2pm - Central Park Cafe",
      recentActivity: "2 hours ago",
      tags: ["Coffee", "Social", "Weekly"]
    },
    {
      id: 2,
      name: "Photography Walks",
      description: "Explore the city through your lens. Perfect for beginners and pros alike!",
      image: "https://images.unsplash.com/photo-1606983340126-99ab4feaa64a?w=400&h=300&fit=crop",
      category: "activities",
      memberCount: 89,
      isJoined: false,
      isPrivate: false,
      location: "Los Angeles, CA",
      nextEvent: "Sunday 10am - Venice Beach",
      recentActivity: "5 hours ago",
      tags: ["Photography", "Art", "Outdoor"]
    },
    {
      id: 3,
      name: "Fitness & Dating",
      description: "Stay fit while meeting new people. Group workouts and healthy lifestyle discussions.",
      image: "https://images.unsplash.com/photo-1571019613454-1cb2f99b2d8b?w=400&h=300&fit=crop",
      category: "activities",
      memberCount: 203,
      isJoined: true,
      isPrivate: false,
      location: "Miami, FL",
      nextEvent: "Tomorrow 6pm - Beach Volleyball",
      recentActivity: "1 hour ago",
      tags: ["Fitness", "Health", "Social"]
    },
    {
      id: 4,
      name: "Book Club Singles",
      description: "Monthly book discussions over wine. Currently reading 'The Seven Husbands of Evelyn Hugo'",
      image: "https://images.unsplash.com/photo-1481627834876-b7833e8f5570?w=400&h=300&fit=crop",
      category: "interests",
      memberCount: 45,
      isJoined: false,
      isPrivate: true,
      location: "Chicago, IL",
      nextEvent: "Next Friday 7pm - Downtown Library",
      recentActivity: "3 hours ago",
      tags: ["Books", "Wine", "Monthly"]
    },
    {
      id: 5,
      name: "Travel Buddies",
      description: "Find travel companions for your next adventure. Share tips and plan group trips!",
      image: "https://images.unsplash.com/photo-1488646953014-85cb44e25828?w=400&h=300&fit=crop",
      category: "interests",
      memberCount: 156,
      isJoined: false,
      isPrivate: false,
      location: "San Francisco, CA",
      nextEvent: "Planning: Weekend Trip to Napa Valley",
      recentActivity: "30 minutes ago",
      tags: ["Travel", "Adventure", "Planning"]
    },
    {
      id: 6,
      name: "Music Lovers Society",
      description: "From jazz clubs to concert halls, discover music together in the city that never sleeps.",
      image: "https://images.unsplash.com/photo-1493225457124-a3eb161ffa5f?w=400&h=300&fit=crop",
      category: "interests",
      memberCount: 78,
      isJoined: true,
      isPrivate: false,
      location: "New York, NY",
      nextEvent: "Thursday 8pm - Blue Note Jazz Club",
      recentActivity: "4 hours ago",
      tags: ["Music", "Concerts", "Jazz"]
    }
  ]);

  const [events, setEvents] = useState<Event[]>([
    {
      id: 1,
      title: "Speed Dating Event - This Saturday",
      description: "Meet 15+ singles in one night! Professional matchmaking event in downtown Manhattan. Ages 25-35 welcome.",
      image: "https://images.unsplash.com/photo-1511632765486-a01980e01a18?w=400&h=300&fit=crop",
      location: "New York, NY",
      date: "2024-12-28",
      time: "7:00 PM",
      price: "$35",
      spotsLeft: 8,
      totalSpots: 30,
      isAttending: false,
      category: "speed-dating",
      ageRange: "25-35",
      requirements: ["Must be single", "Valid ID required", "Professional attire"]
    },
    {
      id: 2,
      title: "Wine Tasting Singles Mixer",
      description: "Sophisticated evening for wine lovers. Sample premium wines while mingling with other singles in an elegant setting.",
      image: "https://images.unsplash.com/photo-1474722883778-792e7990302f?w=400&h=300&fit=crop",
      location: "San Francisco, CA",
      date: "2025-01-05",
      time: "6:30 PM",
      price: "$45",
      spotsLeft: 12,
      totalSpots: 40,
      isAttending: true,
      category: "mixer",
      ageRange: "28-45",
      requirements: ["21+ only", "No walk-ins", "Smart casual dress code"]
    },
    {
      id: 3,
      title: "Hiking & Coffee Singles Meetup",
      description: "Start your weekend with a scenic hike followed by coffee and conversation. Perfect for nature lovers!",
      image: "https://images.unsplash.com/photo-1551632811-561732d1e306?w=400&h=300&fit=crop",
      location: "Los Angeles, CA",
      date: "2025-01-12",
      time: "9:00 AM",
      price: "$20",
      spotsLeft: 15,
      totalSpots: 25,
      isAttending: false,
      category: "activity",
      ageRange: "22-40",
      requirements: ["Comfortable hiking shoes", "Water bottle", "Moderate fitness level"]
    },
    {
      id: 4,
      title: "Cooking Class for Singles",
      description: "Learn to make authentic Italian pasta while meeting other food enthusiasts. Includes wine pairing!",
      image: "https://images.unsplash.com/photo-1556909114-f6e7ad7d3136?w=400&h=300&fit=crop",
      location: "Chicago, IL",
      date: "2025-01-18",
      time: "6:00 PM",
      price: "$65",
      spotsLeft: 6,
      totalSpots: 20,
      isAttending: false,
      category: "workshop",
      ageRange: "25-50",
      requirements: ["Apron provided", "No cooking experience needed", "Dietary restrictions accommodated"]
    },
    {
      id: 5,
      title: "Rooftop Sunset Singles Party",
      description: "Join us for cocktails, music, and stunning city views. Dance the night away under the stars!",
      image: "https://images.unsplash.com/photo-1514362545857-3bc16c4c7d1b?w=400&h=300&fit=crop",
      location: "Miami, FL",
      date: "2025-01-25",
      time: "7:30 PM",
      price: "$55",
      spotsLeft: 25,
      totalSpots: 80,
      isAttending: true,
      category: "mixer",
      ageRange: "21-45",
      requirements: ["Cocktail attire", "Valid ID", "No outside drinks"]
    },
    {
      id: 6,
      title: "Board Game Cafe Singles Night",
      description: "Perfect for introverts! Meet new people over board games in a cozy, low-pressure environment.",
      image: "https://images.unsplash.com/photo-1606092195730-5d7b9af1efc5?w=400&h=300&fit=crop",
      location: "Seattle, WA",
      date: "2025-02-01",
      time: "7:00 PM",
      price: "$25",
      spotsLeft: 18,
      totalSpots: 30,
      isAttending: false,
      category: "activity",
      ageRange: "23-38",
      requirements: ["All skill levels welcome", "Games provided", "Food and drinks available for purchase"]
    }
  ]);

  const categories = [
    { id: "interests", label: "Interests", icon: Heart },
    { id: "location", label: "Location", icon: MapPin },
    { id: "events", label: "Events", icon: Calendar },
    { id: "activities", label: "Activities", icon: Dumbbell }
  ];

  const handleJoinGroup = (groupId: number) => {
    if (!canAccessGroups) {
      return; // Prevent free users from joining groups
    }
    
    setGroups(groups.map(group => 
      group.id === groupId 
        ? { ...group, isJoined: !group.isJoined, memberCount: group.isJoined ? group.memberCount - 1 : group.memberCount + 1 }
        : group
    ));
  };

  const handleMessageGroup = (groupId: number) => {
    console.log(`Opening group chat for group ${groupId}`);
    // In a real app, this would navigate to the group chat
  };

  const handleAttendEvent = (eventId: number) => {
    if (!canAccessGroups) {
      return; // Prevent free users from attending events
    }
    
    setEvents(events.map(event => 
      event.id === eventId 
        ? { 
            ...event, 
            isAttending: !event.isAttending, 
            spotsLeft: event.isAttending ? event.spotsLeft + 1 : event.spotsLeft - 1 
          }
        : event
    ));
  };

  const myGroups = groups.filter(group => group.isJoined);
  const filteredGroups = groups.filter(group => 
    group.name.toLowerCase().includes(searchQuery.toLowerCase()) ||
    group.description.toLowerCase().includes(searchQuery.toLowerCase()) ||
    group.tags.some(tag => tag.toLowerCase().includes(searchQuery.toLowerCase()))
  );

  const myEvents = events.filter(event => event.isAttending);
  const filteredEvents = events.filter(event => 
    event.title.toLowerCase().includes(eventSearchQuery.toLowerCase()) ||
    event.description.toLowerCase().includes(eventSearchQuery.toLowerCase()) ||
    event.location.toLowerCase().includes(eventSearchQuery.toLowerCase())
  );

  const eventCategories = [
    { id: "speed-dating", label: "Speed Dating", icon: Heart },
    { id: "mixer", label: "Mixers", icon: Users },
    { id: "activity", label: "Activities", icon: Dumbbell },
    { id: "workshop", label: "Workshops", icon: Book }
  ];

  return (
    <div className="min-h-screen bg-background">
      {/* Header */}
      <Header currentView="groups" onNavigate={onNavigate} />
      
      {/* Groups Banner */}
      <div className="bg-gradient-to-r from-green-600 via-green-700 to-green-800 dark:from-green-700 dark:via-green-800 dark:to-green-900 text-white py-11 relative overflow-hidden">
        {/* Moving gradient overlay */}
        <motion.div 
          className="absolute inset-0 bg-gradient-to-r from-transparent via-green-300/30 to-transparent opacity-40"
          animate={{
            x: ['-100%', '100%']
          }}
          transition={{
            duration: 8,
            repeat: Infinity,
            ease: "linear",
            repeatDelay: 2
          }}
          style={{
            width: '200%',
            background: 'linear-gradient(90deg, transparent 0%, rgba(255,255,255,0.15) 25%, rgba(255,255,255,0.3) 50%, rgba(255,255,255,0.15) 75%, transparent 100%)'
          }}
        />
        
        {/* Additional gradient overlay */}
        <div className="absolute inset-0 bg-gradient-to-r from-transparent via-green-600/20 to-green-800/40 dark:from-transparent dark:via-green-700/20 dark:to-green-900/40"></div>
        
        <div className="max-w-7xl mx-auto px-6 text-center relative z-10">
          <motion.h1 
            className="text-4xl font-bold mb-4"
            initial={{ opacity: 0, y: 20 }}
            animate={{ opacity: 1, y: 0 }}
            transition={{ 
              duration: 0.8,
              ease: "easeOut"
            }}
          >
            <motion.span
              animate={{ 
                scale: [1, 1.05, 1],
              }}
              transition={{
                duration: 3,
                repeat: Infinity,
                ease: "easeInOut",
                repeatDelay: 0.5
              }}
            >
              👥 Groups & Events
            </motion.span>
          </motion.h1>
          <motion.p 
            className="text-lg text-green-100 dark:text-green-200"
            initial={{ opacity: 0, y: 20 }}
            animate={{ opacity: 1, y: 0 }}
            transition={{ 
              duration: 0.8,
              delay: 0.2,
              ease: "easeOut"
            }}
          >
            Connect with like-minded people through groups and attend exciting dating events
          </motion.p>
        </div>
      </div>

      {/* Main Content */}
      <div className="max-w-6xl mx-auto px-6 py-12">
        {/* Membership Access Alert */}
        {!canAccessGroups && (
          <Alert className="mb-8 border-orange-200 bg-orange-50 dark:bg-orange-900/20">
            <Lock className="w-4 h-4 text-orange-600" />
            <AlertDescription className="text-orange-800 dark:text-orange-400">
              <div className="flex items-center justify-between">
                <div>
                  <strong>Premium Feature:</strong> Groups & Communities are available to Premium and Platinum members only.
                </div>
                <Button 
                  size="sm" 
                  onClick={() => onNavigate("subscription")}
                  className="bg-orange-600 hover:bg-orange-700 text-white ml-4"
                >
                  Upgrade Now
                </Button>
              </div>
            </AlertDescription>
          </Alert>
        )}

        <Tabs value={activeTab} onValueChange={setActiveTab} className="w-full">
          {/* Tab Navigation */}
          <TabsList className="grid w-full grid-cols-4 mb-8 h-auto p-1">
            <TabsTrigger 
              value="discover" 
              className="flex items-center gap-2 px-6 py-3 data-[state=active]:bg-green-600 data-[state=active]:text-white"
            >
              <Search className="w-4 h-4" />
              <span>Discover Groups</span>
            </TabsTrigger>
            <TabsTrigger 
              value="my-groups" 
              className="flex items-center gap-2 px-6 py-3 data-[state=active]:bg-green-600 data-[state=active]:text-white"
            >
              <Users className="w-4 h-4" />
              <span>My Groups ({myGroups.length})</span>
            </TabsTrigger>
            <TabsTrigger 
              value="events" 
              className="flex items-center gap-2 px-6 py-3 data-[state=active]:bg-green-600 data-[state=active]:text-white"
            >
              <Calendar className="w-4 h-4" />
              <span>Events</span>
            </TabsTrigger>
            <TabsTrigger 
              value="my-events" 
              className="flex items-center gap-2 px-6 py-3 data-[state=active]:bg-green-600 data-[state=active]:text-white"
            >
              <Heart className="w-4 h-4" />
              <span>My Events ({myEvents.length})</span>
            </TabsTrigger>
          </TabsList>

          {/* Discover Groups Tab */}
          <TabsContent value="discover" className="space-y-6">
            <motion.div
              initial={{ opacity: 0, y: 20 }}
              animate={{ opacity: 1, y: 0 }}
              transition={{ duration: 0.5 }}
            >
              {/* Search */}
              <div className="flex gap-4 mb-8">
                <div className="relative flex-1">
                  <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-muted-foreground w-4 h-4" />
                  <Input
                    placeholder="Search groups by name, interests, or tags..."
                    value={searchQuery}
                    onChange={(e) => setSearchQuery(e.target.value)}
                    className="pl-10"
                  />
                </div>
              </div>

              {/* Category Filters */}
              <div className="flex flex-wrap gap-2 mb-8">
                {categories.map((category) => {
                  const Icon = category.icon;
                  return (
                    <Button
                      key={category.id}
                      variant="outline"
                      size="sm"
                      className="flex items-center gap-2"
                    >
                      <Icon className="w-4 h-4" />
                      {category.label}
                    </Button>
                  );
                })}
              </div>

              {/* Groups Grid */}
              <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
                {filteredGroups.map((group, index) => (
                  <motion.div
                    key={group.id}
                    initial={{ opacity: 0, y: 20 }}
                    animate={{ opacity: 1, y: 0 }}
                    transition={{ duration: 0.3, delay: index * 0.1 }}
                  >
                    <Card className="p-0 hover:shadow-lg transition-shadow">
                      <div className="relative">
                        <ImageWithFallback
                          src={group.image}
                          alt={group.name}
                          className="w-full h-48 object-cover rounded-t-lg"
                        />
                        {group.isPrivate && (
                          <Badge className="absolute top-2 left-2 bg-yellow-600 text-white">
                            <Crown className="w-3 h-3 mr-1" />
                            Private
                          </Badge>
                        )}
                        <div className="absolute top-2 right-2 bg-white/90 dark:bg-black/90 px-2 py-1 rounded-full">
                          <span className="text-sm font-medium">{group.memberCount} members</span>
                        </div>
                      </div>
                      
                      <div className="p-4 space-y-3">
                        <div>
                          <h3 className="font-semibold">{group.name}</h3>
                          <p className="text-muted-foreground text-sm mt-1 line-clamp-2">{group.description}</p>
                        </div>
                        
                        <div className="flex items-center gap-1 text-sm text-muted-foreground">
                          <MapPin className="w-4 h-4" />
                          <span>{group.location}</span>
                        </div>
                        
                        {group.nextEvent && (
                          <div className="flex items-center gap-1 text-sm text-green-600">
                            <Calendar className="w-4 h-4" />
                            <span>{group.nextEvent}</span>
                          </div>
                        )}
                        
                        <div className="flex flex-wrap gap-1">
                          {group.tags.slice(0, 3).map((tag) => (
                            <Badge key={tag} variant="secondary" className="text-xs">
                              {tag}
                            </Badge>
                          ))}
                        </div>
                        
                        <div className="flex gap-2 pt-2">
                          <Button
                            onClick={() => handleJoinGroup(group.id)}
                            variant={group.isJoined ? "outline" : "default"}
                            size="sm"
                            disabled={!canAccessGroups}
                            className={`flex-1 ${!group.isJoined ? 'bg-green-600 hover:bg-green-700' : ''}`}
                          >
                            {!canAccessGroups ? (
                              <>
                                <Lock className="w-4 h-4 mr-1" />
                                Premium Only
                              </>
                            ) : (
                              <>
                                <UserPlus className="w-4 h-4 mr-1" />
                                {group.isJoined ? 'Leave' : 'Join'}
                              </>
                            )}
                          </Button>
                          {group.isJoined && canAccessGroups && (
                            <Button
                              onClick={() => handleMessageGroup(group.id)}
                              variant="outline"
                              size="sm"
                              className="flex-1"
                            >
                              <MessageCircle className="w-4 h-4 mr-1" />
                              Chat
                            </Button>
                          )}
                        </div>
                      </div>
                    </Card>
                  </motion.div>
                ))}
              </div>
            </motion.div>
          </TabsContent>

          {/* My Groups Tab */}
          <TabsContent value="my-groups" className="space-y-6">
            <motion.div
              initial={{ opacity: 0, y: 20 }}
              animate={{ opacity: 1, y: 0 }}
              transition={{ duration: 0.5 }}
            >
              {myGroups.length === 0 ? (
                <div className="text-center py-12">
                  <Users className="w-16 h-16 text-muted-foreground mx-auto mb-4" />
                  <h3 className="text-xl font-semibold mb-2">No groups joined yet</h3>
                  <p className="text-muted-foreground mb-6">Discover and join groups that match your interests!</p>
                  <Button 
                    onClick={() => setActiveTab("discover")}
                    className="bg-green-600 hover:bg-green-700"
                  >
                    <Search className="w-4 h-4 mr-2" />
                    Explore Groups
                  </Button>
                </div>
              ) : (
                <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
                  {myGroups.map((group, index) => (
                    <motion.div
                      key={group.id}
                      initial={{ opacity: 0, y: 20 }}
                      animate={{ opacity: 1, y: 0 }}
                      transition={{ duration: 0.3, delay: index * 0.1 }}
                    >
                      <Card className="p-0 hover:shadow-lg transition-shadow">
                        <div className="relative">
                          <ImageWithFallback
                            src={group.image}
                            alt={group.name}
                            className="w-full h-48 object-cover rounded-t-lg"
                          />
                          <Badge className="absolute top-2 left-2 bg-green-600 text-white">
                            Joined
                          </Badge>
                          <div className="absolute top-2 right-2 bg-white/90 dark:bg-black/90 px-2 py-1 rounded-full">
                            <span className="text-sm font-medium">{group.memberCount} members</span>
                          </div>
                        </div>
                        
                        <div className="p-4 space-y-3">
                          <div>
                            <h3 className="font-semibold">{group.name}</h3>
                            <p className="text-muted-foreground text-sm mt-1">Active {group.recentActivity}</p>
                          </div>
                          
                          {group.nextEvent && (
                            <div className="flex items-center gap-1 text-sm text-green-600">
                              <Calendar className="w-4 h-4" />
                              <span>{group.nextEvent}</span>
                            </div>
                          )}
                          
                          <div className="flex gap-2 pt-2">
                            <Button
                              onClick={() => handleMessageGroup(group.id)}
                              size="sm"
                              className="flex-1 bg-green-600 hover:bg-green-700"
                            >
                              <MessageCircle className="w-4 h-4 mr-1" />
                              Open Chat
                            </Button>
                            <Button
                              onClick={() => handleJoinGroup(group.id)}
                              variant="outline"
                              size="sm"
                            >
                              Leave
                            </Button>
                          </div>
                        </div>
                      </Card>
                    </motion.div>
                  ))}
                </div>
              )}
            </motion.div>
          </TabsContent>

          {/* Events Tab */}
          <TabsContent value="events" className="space-y-6">
            <motion.div
              initial={{ opacity: 0, y: 20 }}
              animate={{ opacity: 1, y: 0 }}
              transition={{ duration: 0.5 }}
            >
              {/* Search */}
              <div className="flex gap-4 mb-8">
                <div className="relative flex-1">
                  <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-muted-foreground w-4 h-4" />
                  <Input
                    placeholder="Search events by name, location, or description..."
                    value={eventSearchQuery}
                    onChange={(e) => setEventSearchQuery(e.target.value)}
                    className="pl-10"
                  />
                </div>
              </div>

              {/* Event Category Filters */}
              <div className="flex flex-wrap gap-2 mb-8">
                {eventCategories.map((category) => {
                  const Icon = category.icon;
                  return (
                    <Button
                      key={category.id}
                      variant="outline"
                      size="sm"
                      className="flex items-center gap-2"
                    >
                      <Icon className="w-4 h-4" />
                      {category.label}
                    </Button>
                  );
                })}
              </div>

              {/* Events Grid */}
              <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
                {filteredEvents.map((event, index) => (
                  <motion.div
                    key={event.id}
                    initial={{ opacity: 0, y: 20 }}
                    animate={{ opacity: 1, y: 0 }}
                    transition={{ duration: 0.3, delay: index * 0.1 }}
                  >
                    <Card className="p-0 hover:shadow-lg transition-shadow">
                      <div className="relative">
                        <ImageWithFallback
                          src={event.image}
                          alt={event.title}
                          className="w-full h-48 object-cover rounded-t-lg"
                        />
                        <Badge className="absolute top-2 left-2 bg-green-600 text-white">
                          {event.category.replace("-", " ").replace(/\b\w/g, l => l.toUpperCase())}
                        </Badge>
                        <div className="absolute top-2 right-2 bg-white/90 dark:bg-black/90 px-2 py-1 rounded-full">
                          <span className="text-sm font-medium">{event.spotsLeft} spots left</span>
                        </div>
                      </div>
                      
                      <div className="p-4 space-y-3">
                        <div>
                          <h3 className="font-semibold">{event.title}</h3>
                          <p className="text-muted-foreground text-sm mt-1 line-clamp-2">{event.description}</p>
                        </div>
                        
                        <div className="space-y-2 text-sm">
                          <div className="flex items-center gap-1 text-muted-foreground">
                            <MapPin className="w-4 h-4" />
                            <span>{event.location}</span>
                          </div>
                          <div className="flex items-center gap-1 text-green-600">
                            <Calendar className="w-4 h-4" />
                            <span>{event.date} at {event.time}</span>
                          </div>
                          <div className="flex items-center justify-between">
                            <span className="font-semibold text-green-600">{event.price}</span>
                            {event.ageRange && (
                              <span className="text-muted-foreground">Ages {event.ageRange}</span>
                            )}
                          </div>
                        </div>
                        
                        <div className="flex gap-2 pt-2">
                          <Button
                            onClick={() => handleAttendEvent(event.id)}
                            variant={event.isAttending ? "outline" : "default"}
                            size="sm"
                            disabled={!canAccessGroups || event.spotsLeft === 0}
                            className={`flex-1 ${!event.isAttending ? 'bg-green-600 hover:bg-green-700' : ''}`}
                          >
                            {!canAccessGroups ? (
                              <>
                                <Lock className="w-4 h-4 mr-1" />
                                Premium Only
                              </>
                            ) : event.spotsLeft === 0 ? (
                              "Sold Out"
                            ) : (
                              <>
                                <UserPlus className="w-4 h-4 mr-1" />
                                {event.isAttending ? 'Cancel' : 'Attend'}
                              </>
                            )}
                          </Button>
                        </div>
                      </div>
                    </Card>
                  </motion.div>
                ))}
              </div>
            </motion.div>
          </TabsContent>

          {/* My Events Tab */}
          <TabsContent value="my-events" className="space-y-6">
            <motion.div
              initial={{ opacity: 0, y: 20 }}
              animate={{ opacity: 1, y: 0 }}
              transition={{ duration: 0.5 }}
            >
              {myEvents.length === 0 ? (
                <div className="text-center py-12">
                  <Calendar className="w-16 h-16 text-muted-foreground mx-auto mb-4" />
                  <h3 className="text-xl font-semibold mb-2">No events scheduled</h3>
                  <p className="text-muted-foreground mb-6">Discover and join exciting dating events!</p>
                  <Button 
                    onClick={() => setActiveTab("events")}
                    className="bg-green-600 hover:bg-green-700"
                  >
                    <Search className="w-4 h-4 mr-2" />
                    Browse Events
                  </Button>
                </div>
              ) : (
                <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
                  {myEvents.map((event, index) => (
                    <motion.div
                      key={event.id}
                      initial={{ opacity: 0, y: 20 }}
                      animate={{ opacity: 1, y: 0 }}
                      transition={{ duration: 0.3, delay: index * 0.1 }}
                    >
                      <Card className="p-0 hover:shadow-lg transition-shadow">
                        <div className="relative">
                          <ImageWithFallback
                            src={event.image}
                            alt={event.title}
                            className="w-full h-48 object-cover rounded-t-lg"
                          />
                          <Badge className="absolute top-2 left-2 bg-green-600 text-white">
                            Attending
                          </Badge>
                          <div className="absolute top-2 right-2 bg-white/90 dark:bg-black/90 px-2 py-1 rounded-full">
                            <span className="text-sm font-medium">{event.spotsLeft} spots left</span>
                          </div>
                        </div>
                        
                        <div className="p-4 space-y-3">
                          <div>
                            <h3 className="font-semibold">{event.title}</h3>
                            <p className="text-muted-foreground text-sm mt-1">{event.location}</p>
                          </div>
                          
                          <div className="flex items-center gap-1 text-sm text-green-600">
                            <Calendar className="w-4 h-4" />
                            <span>{event.date} at {event.time}</span>
                          </div>
                          
                          <div className="flex gap-2 pt-2">
                            <Button
                              size="sm"
                              className="flex-1 bg-green-600 hover:bg-green-700"
                            >
                              <MessageCircle className="w-4 h-4 mr-1" />
                              Event Chat
                            </Button>
                            <Button
                              onClick={() => handleAttendEvent(event.id)}
                              variant="outline"
                              size="sm"
                            >
                              Cancel
                            </Button>
                          </div>
                        </div>
                      </Card>
                    </motion.div>
                  ))}
                </div>
              )}
            </motion.div>
          </TabsContent>
        </Tabs>
      </div>
      
      {/* Footer */}
      <Footer onNavigate={onNavigate} onOpenCookieConsent={onOpenCookieConsent} />
      
      {/* Role Switcher for Testing */}
      <RoleSwitcher />
    </div>
  );
}