import { Alert, AlertDescription } from './ui/alert';
import { Card } from './ui/card';
import { MapPin, ExternalLink, Copy, Navigation } from 'lucide-react';
import { Button } from './ui/button';
import { useState } from 'react';

export function GoogleMapsSetupInstructions() {
  const [copiedStep, setCopiedStep] = useState<number | null>(null);

  const copyToClipboard = (text: string, step: number) => {
    navigator.clipboard.writeText(text).then(() => {
      setCopiedStep(step);
      setTimeout(() => setCopiedStep(null), 2000);
    });
  };

  return (
    <div className="max-w-4xl mx-auto p-6">
      <Card className="p-6">
        <div className="flex items-center gap-3 mb-6">
          <div className="w-12 h-12 bg-blue-100 dark:bg-blue-900 rounded-lg flex items-center justify-center">
            <MapPin className="w-6 h-6 text-blue-600 dark:text-blue-400" />
          </div>
          <div>
            <h2 className="text-2xl font-bold">Google Maps Setup Required</h2>
            <p className="text-muted-foreground">Enable interactive maps and location features</p>
          </div>
        </div>

        <Alert className="mb-6">
          <MapPin className="h-4 w-4" />
          <AlertDescription>
            To use map features in SinglesZonne, you need to set up Google Maps API. This enables location selection, 
            interactive user profiles, and location-based search functionality.
          </AlertDescription>
        </Alert>

        <div className="space-y-6">
          <div className="grid gap-4">
            <h3 className="text-lg font-semibold">Quick Setup Steps:</h3>
            
            {/* Step 1 */}
            <div className="border rounded-lg p-4">
              <div className="flex items-start gap-3">
                <div className="w-6 h-6 bg-primary text-primary-foreground rounded-full flex items-center justify-center text-sm font-bold">
                  1
                </div>
                <div className="flex-1">
                  <h4 className="font-medium mb-2">Get Google Maps API Key</h4>
                  <p className="text-sm text-muted-foreground mb-3">
                    Visit the Google Cloud Console to create an API key
                  </p>
                  <Button
                    variant="outline"
                    size="sm"
                    onClick={() => window.open('https://console.cloud.google.com/apis/credentials', '_blank')}
                  >
                    <ExternalLink className="w-4 h-4 mr-2" />
                    Open Google Cloud Console
                  </Button>
                </div>
              </div>
            </div>

            {/* Step 2 */}
            <div className="border rounded-lg p-4">
              <div className="flex items-start gap-3">
                <div className="w-6 h-6 bg-primary text-primary-foreground rounded-full flex items-center justify-center text-sm font-bold">
                  2
                </div>
                <div className="flex-1">
                  <h4 className="font-medium mb-2">Enable Required APIs</h4>
                  <p className="text-sm text-muted-foreground mb-3">
                    Enable these APIs in your Google Cloud project:
                  </p>
                  <ul className="space-y-1 text-sm text-muted-foreground">
                    <li>• Maps JavaScript API</li>
                    <li>• Places API</li>
                    <li>• Geocoding API</li>
                  </ul>
                </div>
              </div>
            </div>

            {/* Step 3 */}
            <div className="border rounded-lg p-4">
              <div className="flex items-start gap-3">
                <div className="w-6 h-6 bg-primary text-primary-foreground rounded-full flex items-center justify-center text-sm font-bold">
                  3
                </div>
                <div className="flex-1">
                  <h4 className="font-medium mb-2">Update API Key in Code</h4>
                  <p className="text-sm text-muted-foreground mb-3">
                    Replace the placeholder in GoogleMapsLoader.tsx:
                  </p>
                  <div className="bg-muted p-3 rounded-md relative">
                    <code className="text-sm">
                      const GOOGLE_MAPS_API_KEY = "your_actual_api_key_here";
                    </code>
                    <Button
                      variant="ghost"
                      size="sm"
                      className="absolute top-1 right-1"
                      onClick={() => copyToClipboard('const GOOGLE_MAPS_API_KEY = "your_actual_api_key_here";', 3)}
                    >
                      {copiedStep === 3 ? '✓' : <Copy className="w-3 h-3" />}
                    </Button>
                  </div>
                </div>
              </div>
            </div>

            {/* Step 4 */}
            <div className="border rounded-lg p-4">
              <div className="flex items-start gap-3">
                <div className="w-6 h-6 bg-primary text-primary-foreground rounded-full flex items-center justify-center text-sm font-bold">
                  4
                </div>
                <div className="flex-1">
                  <h4 className="font-medium mb-2">Restrict API Key (Production)</h4>
                  <p className="text-sm text-muted-foreground mb-3">
                    For production, restrict your API key to your domain:
                  </p>
                  <div className="bg-muted p-3 rounded-md relative">
                    <code className="text-sm">
                      yourdomain.com/*<br />
                      *.yourdomain.com/*
                    </code>
                    <Button
                      variant="ghost"
                      size="sm"
                      className="absolute top-1 right-1"
                      onClick={() => copyToClipboard('yourdomain.com/*\n*.yourdomain.com/*', 4)}
                    >
                      {copiedStep === 4 ? '✓' : <Copy className="w-3 h-3" />}
                    </Button>
                  </div>
                </div>
              </div>
            </div>
          </div>

          <div className="bg-yellow-50 dark:bg-yellow-900/20 border border-yellow-200 dark:border-yellow-800 rounded-lg p-4">
            <h4 className="font-medium text-yellow-800 dark:text-yellow-200 mb-2">
              💡 Development Tips
            </h4>
            <ul className="space-y-1 text-sm text-yellow-700 dark:text-yellow-300">
              <li>• Start with unrestricted API key for development</li>
              <li>• Monitor usage in Google Cloud Console</li>
              <li>• Enable billing to avoid quota limits</li>
              <li>• Test with different locations and devices</li>
            </ul>
          </div>

          <div className="bg-blue-50 dark:bg-blue-900/20 border border-blue-200 dark:border-blue-800 rounded-lg p-4">
            <h4 className="font-medium text-blue-800 dark:text-blue-200 mb-2">
              🎯 Features Enabled After Setup
            </h4>
            <div className="grid grid-cols-1 md:grid-cols-2 gap-2 text-sm text-blue-700 dark:text-blue-300">
              <div>• Interactive user profile maps</div>
              <div>• Location-based member search</div>
              <div>• City and address search</div>
              <div>• Current location detection</div>
              <div>• Distance calculations</div>
              <div>• Event venue mapping</div>
            </div>
          </div>
        </div>
      </Card>
    </div>
  );
}

export function GoogleMapsPlaceholder({ error }: { error?: string }) {
  return (
    <div className="w-full h-96 bg-gradient-to-br from-blue-50/50 to-purple-50/50 dark:from-blue-900/20 dark:to-purple-900/20 border border-border rounded-lg flex items-center justify-center">
      <div className="text-center p-6 max-w-md">
        <div className="w-16 h-16 bg-gradient-to-br from-blue-100 to-purple-100 dark:from-blue-800 dark:to-purple-800 rounded-full flex items-center justify-center mx-auto mb-4">
          <MapPin className="w-8 h-8 text-blue-600 dark:text-blue-400" />
        </div>
        <h3 className="font-medium mb-2">Interactive Map</h3>
        <p className="text-sm text-muted-foreground mb-4">
          Interactive map functionality available with Google Maps API
        </p>
        <div className="space-y-2">
          <div className="grid grid-cols-2 gap-2 text-xs">
            <div className="bg-background/50 border rounded-md p-2 text-center">
              <MapPin className="w-3 h-3 mx-auto mb-1 text-blue-500" />
              <span>Location Markers</span>
            </div>
            <div className="bg-background/50 border rounded-md p-2 text-center">
              <Navigation className="w-3 h-3 mx-auto mb-1 text-purple-500" />
              <span>Navigation</span>
            </div>
          </div>
        </div>
      </div>
    </div>
  );
}