import { useState, useEffect } from "react";
import { Card } from "./ui/card";
import { Button } from "./ui/button";
import { Badge } from "./ui/badge";
import { Avatar, AvatarImage, AvatarFallback } from "./ui/avatar";
import { Dialog, DialogContent, DialogHeader, DialogTitle, DialogDescription, DialogFooter } from "./ui/dialog";
import { Textarea } from "./ui/textarea";
import { Alert, AlertDescription } from "./ui/alert";
import { 
  Trophy, 
  Users, 
  Clock, 
  Star, 
  Play, 
  Flag,
  MessageCircle,
  Crown,
  Zap,
  Heart,
  Target,
  Award,
  AlertCircle
} from "lucide-react";
import { motion, AnimatePresence } from "motion/react";
import { useTranslation } from "./TranslationProvider";
import { useUser } from "./UserProvider";
import { usePoints } from "./PointsProvider";
import { toast } from "sonner@2.0.3";

interface JeopardyQuestion {
  id: string;
  category: string;
  points: number;
  question: string;
  answer: string;
  difficulty: "easy" | "medium" | "hard";
  isAnswered: boolean;
}

interface GameRoom {
  id: string;
  players: Array<{
    id: string;
    name: string;
    avatar: string;
    score: number;
    isOnline: boolean;
  }>;
  status: "waiting" | "playing" | "finished";
  currentQuestion: JeopardyQuestion | null;
  startTime: Date;
  maxPlayers: 2;
}

interface ReportModalProps {
  isOpen: boolean;
  onClose: () => void;
  reportedPlayer: string;
  gameId: string;
}

function ReportModal({ isOpen, onClose, reportedPlayer, gameId }: ReportModalProps) {
  const [reason, setReason] = useState("");
  const [isSubmitting, setIsSubmitting] = useState(false);

  const handleSubmit = async () => {
    if (!reason.trim()) {
      toast.error("Please provide a reason for reporting");
      return;
    }

    setIsSubmitting(true);
    
    // Simulate API call
    await new Promise(resolve => setTimeout(resolve, 1000));
    
    toast.success("Report submitted successfully. Our team will review it shortly.");
    setReason("");
    setIsSubmitting(false);
    onClose();
  };

  return (
    <Dialog open={isOpen} onOpenChange={onClose}>
      <DialogContent className="max-w-md">
        <DialogHeader>
          <DialogTitle className="flex items-center gap-2">
            <Flag className="w-5 h-5 text-red-500" />
            Report Abusive Behavior
          </DialogTitle>
          <DialogDescription>
            Reporting player: <span className="font-semibold">{reportedPlayer}</span>
          </DialogDescription>
        </DialogHeader>
        
        <div className="space-y-4">
          <div>
            <label className="text-sm font-medium mb-2 block">
              Reason for Report
            </label>
            <Textarea
              value={reason}
              onChange={(e) => setReason(e.target.value)}
              placeholder="Please describe the abusive behavior..."
              className="min-h-[120px]"
            />
          </div>
          
          <Alert>
            <AlertCircle className="h-4 w-4" />
            <AlertDescription className="text-sm">
              False reports may result in account restrictions. Please only report genuine cases of abuse.
            </AlertDescription>
          </Alert>
        </div>
        
        <DialogFooter>
          <Button variant="outline" onClick={onClose} disabled={isSubmitting}>
            Cancel
          </Button>
          <Button 
            onClick={handleSubmit} 
            disabled={isSubmitting}
            className="bg-red-600 hover:bg-red-700"
          >
            {isSubmitting ? "Submitting..." : "Submit Report"}
          </Button>
        </DialogFooter>
      </DialogContent>
    </Dialog>
  );
}

export function DatingGamesPage() {
  const { t } = useTranslation();
  const { user } = useUser();
  const { points, addPoints } = usePoints();
  const [gameState, setGameState] = useState<"lobby" | "playing" | "results">("lobby");
  const [currentRoom, setCurrentRoom] = useState<GameRoom | null>(null);
  const [selectedQuestion, setSelectedQuestion] = useState<JeopardyQuestion | null>(null);
  const [userAnswer, setUserAnswer] = useState("");
  const [showAnswer, setShowAnswer] = useState(false);
  const [reportModalOpen, setReportModalOpen] = useState(false);
  const [reportedPlayer, setReportedPlayer] = useState("");
  const [chatMessages, setChatMessages] = useState<Array<{id: string, user: string, message: string}>>([]);
  const [chatInput, setChatInput] = useState("");

  // Mock Jeopardy board categories and questions
  const categories = [
    "First Date Tips",
    "Relationship Facts",
    "Love Languages",
    "Dating Etiquette",
    "Romance History"
  ];

  const [board, setBoard] = useState<JeopardyQuestion[][]>([
    // First Date Tips
    [
      { id: "1-1", category: "First Date Tips", points: 100, question: "What percentage of first dates lead to a second date?", answer: "53%", difficulty: "easy", isAnswered: false },
      { id: "1-2", category: "First Date Tips", points: 200, question: "What is the most popular first date activity?", answer: "Dinner or Coffee", difficulty: "easy", isAnswered: false },
      { id: "1-3", category: "First Date Tips", points: 300, question: "How many minutes late is acceptable on a first date?", answer: "5-10 minutes", difficulty: "medium", isAnswered: false },
      { id: "1-4", category: "First Date Tips", points: 400, question: "What is the recommended duration for a first date?", answer: "1-2 hours", difficulty: "medium", isAnswered: false },
      { id: "1-5", category: "First Date Tips", points: 500, question: "What percentage of people look at their phone during a first date?", answer: "75%", difficulty: "hard", isAnswered: false },
    ],
    // Relationship Facts
    [
      { id: "2-1", category: "Relationship Facts", points: 100, question: "What is the average length of a first date?", answer: "2-3 hours", difficulty: "easy", isAnswered: false },
      { id: "2-2", category: "Relationship Facts", points: 200, question: "How many dates before most couples become exclusive?", answer: "5-6 dates", difficulty: "easy", isAnswered: false },
      { id: "2-3", category: "Relationship Facts", points: 300, question: "What day of the week is most popular for first dates?", answer: "Saturday", difficulty: "medium", isAnswered: false },
      { id: "2-4", category: "Relationship Facts", points: 400, question: "What is the average time before saying 'I love you'?", answer: "3-4 months", difficulty: "medium", isAnswered: false },
      { id: "2-5", category: "Relationship Facts", points: 500, question: "What percentage of relationships start online?", answer: "40%", difficulty: "hard", isAnswered: false },
    ],
    // Love Languages
    [
      { id: "3-1", category: "Love Languages", points: 100, question: "Name one of the 5 love languages", answer: "Words of Affirmation, Acts of Service, Receiving Gifts, Quality Time, Physical Touch", difficulty: "easy", isAnswered: false },
      { id: "3-2", category: "Love Languages", points: 200, question: "Who created the concept of love languages?", answer: "Dr. Gary Chapman", difficulty: "easy", isAnswered: false },
      { id: "3-3", category: "Love Languages", points: 300, question: "What is the most common love language?", answer: "Words of Affirmation", difficulty: "medium", isAnswered: false },
      { id: "3-4", category: "Love Languages", points: 400, question: "Can people have more than one primary love language?", answer: "Yes", difficulty: "medium", isAnswered: false },
      { id: "3-5", category: "Love Languages", points: 500, question: "What year was the book 'The 5 Love Languages' published?", answer: "1992", difficulty: "hard", isAnswered: false },
    ],
    // Dating Etiquette
    [
      { id: "4-1", category: "Dating Etiquette", points: 100, question: "Should you split the bill on a first date?", answer: "Discuss or offer to split", difficulty: "easy", isAnswered: false },
      { id: "4-2", category: "Dating Etiquette", points: 200, question: "When should you text after a first date?", answer: "Within 24 hours", difficulty: "easy", isAnswered: false },
      { id: "4-3", category: "Dating Etiquette", points: 300, question: "Is it okay to check your phone during a date?", answer: "Only for emergencies", difficulty: "medium", isAnswered: false },
      { id: "4-4", category: "Dating Etiquette", points: 400, question: "How many days should you wait before a second date?", answer: "3-5 days", difficulty: "medium", isAnswered: false },
      { id: "4-5", category: "Dating Etiquette", points: 500, question: "What topics should you avoid on first dates?", answer: "Ex-partners, politics, religion, money", difficulty: "hard", isAnswered: false },
    ],
    // Romance History
    [
      { id: "5-1", category: "Romance History", points: 100, question: "What date is Valentine's Day?", answer: "February 14th", difficulty: "easy", isAnswered: false },
      { id: "5-2", category: "Romance History", points: 200, question: "What flower is most associated with romance?", answer: "Rose", difficulty: "easy", isAnswered: false },
      { id: "5-3", category: "Romance History", points: 300, question: "Where did the tradition of engagement rings begin?", answer: "Ancient Rome", difficulty: "medium", isAnswered: false },
      { id: "5-4", category: "Romance History", points: 400, question: "What year did online dating begin?", answer: "1995", difficulty: "medium", isAnswered: false },
      { id: "5-5", category: "Romance History", points: 500, question: "Who invented speed dating?", answer: "Rabbi Yaacov Deyo", difficulty: "hard", isAnswered: false },
    ],
  ]);

  const [activeGames, setActiveGames] = useState<GameRoom[]>([
    {
      id: "1",
      players: [
        { id: "1", name: "Sarah K.", avatar: "https://images.unsplash.com/photo-1494790108377-be9c29b29330?w=100", score: 0, isOnline: true },
        { id: "2", name: "Mike R.", avatar: "https://images.unsplash.com/photo-1500648767791-00dcc994a43e?w=100", score: 0, isOnline: true },
      ],
      status: "waiting",
      currentQuestion: null,
      startTime: new Date(),
      maxPlayers: 2,
    },
  ]);

  const [leaderboard] = useState([
    { rank: 1, name: "Emma Wilson", score: 2500, gamesPlayed: 12, winRate: 75 },
    { rank: 2, name: "James Miller", score: 2300, gamesPlayed: 15, winRate: 67 },
    { rank: 3, name: "Sofia Garcia", score: 2100, gamesPlayed: 10, winRate: 80 },
    { rank: 4, name: "Alex Chen", score: 1900, gamesPlayed: 8, winRate: 62 },
    { rank: 5, name: "Maya Patel", score: 1700, gamesPlayed: 11, winRate: 55 },
  ]);

  const handleQuickMatch = () => {
    // Simulate finding a match
    toast.info("Searching for an opponent...");
    
    setTimeout(() => {
      const newRoom: GameRoom = {
        id: "new-game",
        players: [
          { id: user?.id || "current-user", name: user?.name || "You", avatar: user?.profilePicture || "", score: 0, isOnline: true },
          { id: "opponent", name: "Jessica T.", avatar: "https://images.unsplash.com/photo-1438761681033-6461ffad8d80?w=100", score: 0, isOnline: true },
        ],
        status: "playing",
        currentQuestion: null,
        startTime: new Date(),
        maxPlayers: 2,
      };
      
      setCurrentRoom(newRoom);
      setGameState("playing");
      toast.success("Match found! Game starting...");
    }, 2000);
  };

  const handleSelectQuestion = (categoryIndex: number, questionIndex: number) => {
    const question = board[categoryIndex][questionIndex];
    
    if (question.isAnswered) {
      toast.error("This question has already been answered!");
      return;
    }

    setSelectedQuestion(question);
    setShowAnswer(false);
    setUserAnswer("");
  };

  const handleSubmitAnswer = () => {
    if (!selectedQuestion || !currentRoom) return;

    // Mark question as answered
    const newBoard = [...board];
    const categoryIndex = categories.indexOf(selectedQuestion.category);
    const questionIndex = newBoard[categoryIndex].findIndex(q => q.id === selectedQuestion.id);
    newBoard[categoryIndex][questionIndex].isAnswered = true;
    setBoard(newBoard);

    // Simple answer checking (in production, this would be server-side)
    const isCorrect = userAnswer.toLowerCase().includes(selectedQuestion.answer.toLowerCase().split(" ")[0]);
    
    if (isCorrect) {
      const newRoom = { ...currentRoom };
      newRoom.players[0].score += selectedQuestion.points;
      setCurrentRoom(newRoom);
      addPoints(selectedQuestion.points);
      toast.success(`Correct! +${selectedQuestion.points} points`);
    } else {
      toast.error("Incorrect answer. Your opponent gets a chance!");
    }

    setShowAnswer(true);
    
    setTimeout(() => {
      setSelectedQuestion(null);
      setUserAnswer("");
      setShowAnswer(false);
    }, 3000);
  };

  const handleEndGame = () => {
    if (!currentRoom) return;

    const winner = currentRoom.players.reduce((prev, current) => 
      (prev.score > current.score) ? prev : current
    );

    if (winner.id === user?.id || winner.id === "current-user") {
      addPoints(500); // Bonus for winning
      toast.success("🎉 You won! +500 bonus points!");
    } else {
      addPoints(100); // Participation points
      toast.info("Good game! +100 participation points");
    }

    setGameState("results");
  };

  const handleReportPlayer = (playerName: string) => {
    setReportedPlayer(playerName);
    setReportModalOpen(true);
  };

  const handleSendMessage = () => {
    if (!chatInput.trim()) return;

    setChatMessages([
      ...chatMessages,
      { id: Date.now().toString(), user: user?.name || "You", message: chatInput }
    ]);
    setChatInput("");
  };

  const handleNewGame = () => {
    setGameState("lobby");
    setCurrentRoom(null);
    // Reset board
    const resetBoard = board.map(category => 
      category.map(q => ({ ...q, isAnswered: false }))
    );
    setBoard(resetBoard);
  };

  return (
    <div className="min-h-screen bg-gradient-to-br from-pink-50 via-purple-50 to-blue-50 dark:from-gray-900 dark:via-purple-900/20 dark:to-gray-900 py-8">
      <div className="max-w-7xl mx-auto px-6">
        {/* Header */}
        <motion.div
          initial={{ opacity: 0, y: -20 }}
          animate={{ opacity: 1, y: 0 }}
          className="text-center mb-8"
        >
          <div className="flex items-center justify-center gap-3 mb-4">
            <div className="bg-gradient-to-br from-purple-500 to-pink-500 p-3 rounded-2xl shadow-lg">
              <Trophy className="w-8 h-8 text-white" />
            </div>
            <h1 className="text-4xl font-bold bg-gradient-to-r from-purple-600 to-pink-600 bg-clip-text text-transparent">
              Dating Jeopardy
            </h1>
          </div>
          <p className="text-muted-foreground text-lg">
            Test your dating knowledge and compete with other singles!
          </p>
        </motion.div>

        {gameState === "lobby" && (
          <div className="space-y-6">
            {/* Quick Actions */}
            <div className="grid md:grid-cols-2 gap-6">
              <Card className="p-6 border-2 border-purple-200 dark:border-purple-800 hover:border-purple-400 dark:hover:border-purple-600 transition-colors">
                <div className="flex items-start gap-4">
                  <div className="bg-purple-100 dark:bg-purple-900/30 p-3 rounded-xl">
                    <Zap className="w-6 h-6 text-purple-600 dark:text-purple-400" />
                  </div>
                  <div className="flex-1">
                    <h3 className="font-semibold text-lg mb-2">Quick Match</h3>
                    <p className="text-sm text-muted-foreground mb-4">
                      Find a random opponent and start playing immediately
                    </p>
                    <Button 
                      onClick={handleQuickMatch}
                      className="w-full bg-gradient-to-r from-purple-600 to-pink-600 hover:from-purple-700 hover:to-pink-700"
                    >
                      <Play className="w-4 h-4 mr-2" />
                      Start Quick Match
                    </Button>
                  </div>
                </div>
              </Card>

              <Card className="p-6 border-2 border-blue-200 dark:border-blue-800 hover:border-blue-400 dark:hover:border-blue-600 transition-colors">
                <div className="flex items-start gap-4">
                  <div className="bg-blue-100 dark:bg-blue-900/30 p-3 rounded-xl">
                    <Users className="w-6 h-6 text-blue-600 dark:text-blue-400" />
                  </div>
                  <div className="flex-1">
                    <h3 className="font-semibold text-lg mb-2">Join Active Game</h3>
                    <p className="text-sm text-muted-foreground mb-4">
                      Join an ongoing game or watch others play
                    </p>
                    <Button 
                      variant="outline"
                      className="w-full border-blue-300 hover:bg-blue-50 dark:border-blue-700 dark:hover:bg-blue-900/30"
                    >
                      <Target className="w-4 h-4 mr-2" />
                      Browse Games
                    </Button>
                  </div>
                </div>
              </Card>
            </div>

            {/* Leaderboard */}
            <Card className="p-6">
              <div className="flex items-center gap-2 mb-6">
                <Crown className="w-5 h-5 text-yellow-600" />
                <h2 className="text-xl font-semibold">Top Players This Week</h2>
              </div>
              
              <div className="space-y-3">
                {leaderboard.map((player) => (
                  <motion.div
                    key={player.rank}
                    initial={{ opacity: 0, x: -20 }}
                    animate={{ opacity: 1, x: 0 }}
                    transition={{ delay: player.rank * 0.1 }}
                    className="flex items-center gap-4 p-4 rounded-lg bg-gradient-to-r from-purple-50 to-pink-50 dark:from-purple-900/20 dark:to-pink-900/20 hover:shadow-md transition-shadow"
                  >
                    <div className={`flex items-center justify-center w-8 h-8 rounded-full font-bold ${
                      player.rank === 1 ? "bg-yellow-400 text-yellow-900" :
                      player.rank === 2 ? "bg-gray-300 text-gray-700" :
                      player.rank === 3 ? "bg-orange-400 text-orange-900" :
                      "bg-purple-200 dark:bg-purple-800 text-purple-900 dark:text-purple-100"
                    }`}>
                      {player.rank}
                    </div>
                    
                    <div className="flex-1">
                      <div className="font-semibold">{player.name}</div>
                      <div className="text-sm text-muted-foreground">
                        {player.gamesPlayed} games • {player.winRate}% win rate
                      </div>
                    </div>
                    
                    <div className="text-right">
                      <div className="font-bold text-purple-600 dark:text-purple-400">
                        {player.score.toLocaleString()}
                      </div>
                      <div className="text-xs text-muted-foreground">points</div>
                    </div>
                  </motion.div>
                ))}
              </div>
            </Card>

            {/* How to Play */}
            <Card className="p-6 bg-gradient-to-br from-purple-50 to-pink-50 dark:from-purple-900/20 dark:to-pink-900/20">
              <h2 className="text-xl font-semibold mb-4 flex items-center gap-2">
                <Star className="w-5 h-5 text-purple-600" />
                How to Play
              </h2>
              <div className="grid md:grid-cols-2 gap-4">
                <div className="space-y-3">
                  <div className="flex gap-3">
                    <div className="flex-shrink-0 w-6 h-6 rounded-full bg-purple-600 text-white flex items-center justify-center text-sm font-bold">1</div>
                    <p className="text-sm">Select a category and point value on the game board</p>
                  </div>
                  <div className="flex gap-3">
                    <div className="flex-shrink-0 w-6 h-6 rounded-full bg-purple-600 text-white flex items-center justify-center text-sm font-bold">2</div>
                    <p className="text-sm">Read the dating-related question carefully</p>
                  </div>
                  <div className="flex gap-3">
                    <div className="flex-shrink-0 w-6 h-6 rounded-full bg-purple-600 text-white flex items-center justify-center text-sm font-bold">3</div>
                    <p className="text-sm">Submit your answer before time runs out</p>
                  </div>
                </div>
                <div className="space-y-3">
                  <div className="flex gap-3">
                    <div className="flex-shrink-0 w-6 h-6 rounded-full bg-pink-600 text-white flex items-center justify-center text-sm font-bold">4</div>
                    <p className="text-sm">Earn points for correct answers</p>
                  </div>
                  <div className="flex gap-3">
                    <div className="flex-shrink-0 w-6 h-6 rounded-full bg-pink-600 text-white flex items-center justify-center text-sm font-bold">5</div>
                    <p className="text-sm">The player with the most points wins!</p>
                  </div>
                  <div className="flex gap-3">
                    <div className="flex-shrink-0 w-6 h-6 rounded-full bg-pink-600 text-white flex items-center justify-center text-sm font-bold">6</div>
                    <p className="text-sm">Winners earn bonus reward points</p>
                  </div>
                </div>
              </div>
            </Card>
          </div>
        )}

        {gameState === "playing" && currentRoom && (
          <div className="space-y-6">
            {/* Game Header */}
            <Card className="p-4 bg-gradient-to-r from-purple-600 to-pink-600 text-white">
              <div className="flex items-center justify-between">
                <div className="flex items-center gap-6">
                  {currentRoom.players.map((player, idx) => (
                    <div key={player.id} className="flex items-center gap-3">
                      <Avatar className="w-12 h-12 border-2 border-white">
                        <AvatarImage src={player.avatar} />
                        <AvatarFallback>{player.name[0]}</AvatarFallback>
                      </Avatar>
                      <div>
                        <div className="font-semibold">{player.name}</div>
                        <div className="text-sm text-purple-100">
                          {player.score} points
                        </div>
                      </div>
                      {idx === 0 && (
                        <Button
                          size="sm"
                          variant="ghost"
                          className="text-white hover:bg-white/20"
                          onClick={() => handleReportPlayer(currentRoom.players[1].name)}
                        >
                          <Flag className="w-4 h-4 mr-1" />
                          Report
                        </Button>
                      )}
                    </div>
                  ))}
                </div>
                
                <Button
                  variant="outline"
                  className="bg-white text-purple-600 hover:bg-purple-50"
                  onClick={handleEndGame}
                >
                  End Game
                </Button>
              </div>
            </Card>

            {/* Jeopardy Board */}
            <Card className="p-6">
              <div className="mb-4">
                <h2 className="text-2xl font-bold text-center mb-2">Dating Jeopardy Board</h2>
                <p className="text-center text-sm text-muted-foreground">Select a category and point value</p>
              </div>
              
              <div className="overflow-x-auto">
                <div className="min-w-[640px]">
                  {/* Category Headers */}
                  <div className="grid grid-cols-5 gap-2 mb-2">
                    {categories.map((category, idx) => (
                      <div
                        key={idx}
                        className="bg-gradient-to-br from-purple-600 to-purple-700 text-white p-3 rounded-lg text-center font-bold text-sm"
                      >
                        {category}
                      </div>
                    ))}
                  </div>

                  {/* Question Grid */}
                  {[0, 1, 2, 3, 4].map((questionIdx) => (
                    <div key={questionIdx} className="grid grid-cols-5 gap-2 mb-2">
                      {board.map((category, categoryIdx) => {
                        const question = category[questionIdx];
                        return (
                          <motion.button
                            key={question.id}
                            onClick={() => handleSelectQuestion(categoryIdx, questionIdx)}
                            disabled={question.isAnswered}
                            className={`p-6 rounded-lg font-bold text-2xl transition-all ${
                              question.isAnswered
                                ? "bg-gray-200 dark:bg-gray-800 text-gray-400 cursor-not-allowed"
                                : "bg-gradient-to-br from-blue-500 to-blue-600 text-white hover:from-blue-600 hover:to-blue-700 hover:scale-105 cursor-pointer shadow-lg"
                            }`}
                            whileHover={!question.isAnswered ? { scale: 1.05 } : {}}
                            whileTap={!question.isAnswered ? { scale: 0.95 } : {}}
                          >
                            {question.isAnswered ? "✓" : `$${question.points}`}
                          </motion.button>
                        );
                      })}
                    </div>
                  ))}
                </div>
              </div>
            </Card>

            {/* Question Modal */}
            <AnimatePresence>
              {selectedQuestion && (
                <Dialog open={!!selectedQuestion} onOpenChange={() => setSelectedQuestion(null)}>
                  <DialogContent className="max-w-2xl">
                    <DialogHeader>
                      <DialogTitle className="flex items-center justify-between">
                        <span>{selectedQuestion.category}</span>
                        <Badge className="bg-gradient-to-r from-purple-600 to-pink-600 text-white">
                          ${selectedQuestion.points}
                        </Badge>
                      </DialogTitle>
                    </DialogHeader>
                    
                    <div className="space-y-4">
                      <div className="bg-gradient-to-br from-purple-50 to-pink-50 dark:from-purple-900/20 dark:to-pink-900/20 p-6 rounded-lg">
                        <p className="text-lg font-semibold text-center">
                          {selectedQuestion.question}
                        </p>
                      </div>

                      {!showAnswer ? (
                        <>
                          <div>
                            <label className="text-sm font-medium mb-2 block">Your Answer</label>
                            <Textarea
                              value={userAnswer}
                              onChange={(e) => setUserAnswer(e.target.value)}
                              placeholder="Type your answer here..."
                              className="min-h-[100px]"
                            />
                          </div>
                          
                          <Button
                            onClick={handleSubmitAnswer}
                            disabled={!userAnswer.trim()}
                            className="w-full bg-gradient-to-r from-purple-600 to-pink-600 hover:from-purple-700 hover:to-pink-700"
                          >
                            Submit Answer
                          </Button>
                        </>
                      ) : (
                        <div className="space-y-3">
                          <Alert className="bg-green-50 dark:bg-green-900/20 border-green-200 dark:border-green-800">
                            <AlertDescription>
                              <div className="font-semibold mb-1">Correct Answer:</div>
                              <div className="text-lg">{selectedQuestion.answer}</div>
                            </AlertDescription>
                          </Alert>
                        </div>
                      )}
                    </div>
                  </DialogContent>
                </Dialog>
              )}
            </AnimatePresence>

            {/* Game Chat */}
            <Card className="p-4">
              <div className="flex items-center gap-2 mb-4">
                <MessageCircle className="w-5 h-5 text-purple-600" />
                <h3 className="font-semibold">Game Chat</h3>
              </div>
              
              <div className="space-y-2 mb-4 max-h-[200px] overflow-y-auto">
                {chatMessages.map((msg) => (
                  <div key={msg.id} className="flex gap-2">
                    <span className="font-semibold text-sm text-purple-600">{msg.user}:</span>
                    <span className="text-sm">{msg.message}</span>
                  </div>
                ))}
                {chatMessages.length === 0 && (
                  <p className="text-sm text-muted-foreground text-center py-4">
                    No messages yet. Start the conversation!
                  </p>
                )}
              </div>
              
              <div className="flex gap-2">
                <input
                  type="text"
                  value={chatInput}
                  onChange={(e) => setChatInput(e.target.value)}
                  onKeyPress={(e) => e.key === "Enter" && handleSendMessage()}
                  placeholder="Type a message..."
                  className="flex-1 px-3 py-2 border rounded-lg text-sm"
                />
                <Button size="sm" onClick={handleSendMessage}>
                  Send
                </Button>
              </div>
            </Card>
          </div>
        )}

        {gameState === "results" && currentRoom && (
          <motion.div
            initial={{ opacity: 0, scale: 0.9 }}
            animate={{ opacity: 1, scale: 1 }}
            className="space-y-6"
          >
            <Card className="p-8 text-center">
              <div className="mb-6">
                <div className="inline-flex items-center justify-center w-20 h-20 bg-gradient-to-br from-yellow-400 to-yellow-600 rounded-full mb-4">
                  <Trophy className="w-10 h-10 text-white" />
                </div>
                <h2 className="text-3xl font-bold mb-2">Game Over!</h2>
                <p className="text-muted-foreground">Great game everyone!</p>
              </div>

              <div className="space-y-4 mb-6">
                {currentRoom.players
                  .sort((a, b) => b.score - a.score)
                  .map((player, idx) => (
                    <div
                      key={player.id}
                      className={`flex items-center justify-between p-4 rounded-lg ${
                        idx === 0
                          ? "bg-gradient-to-r from-yellow-50 to-yellow-100 dark:from-yellow-900/20 dark:to-yellow-800/20 border-2 border-yellow-400"
                          : "bg-gray-50 dark:bg-gray-800"
                      }`}
                    >
                      <div className="flex items-center gap-3">
                        {idx === 0 && <Crown className="w-6 h-6 text-yellow-600" />}
                        <Avatar className="w-12 h-12">
                          <AvatarImage src={player.avatar} />
                          <AvatarFallback>{player.name[0]}</AvatarFallback>
                        </Avatar>
                        <div className="text-left">
                          <div className="font-semibold">{player.name}</div>
                          <div className="text-sm text-muted-foreground">
                            {idx === 0 ? "Winner!" : `${idx + 1}${idx === 1 ? "nd" : "rd"} Place`}
                          </div>
                        </div>
                      </div>
                      <div className="text-2xl font-bold text-purple-600 dark:text-purple-400">
                        {player.score}
                      </div>
                    </div>
                  ))}
              </div>

              <div className="flex gap-4 justify-center">
                <Button
                  onClick={handleNewGame}
                  className="bg-gradient-to-r from-purple-600 to-pink-600 hover:from-purple-700 hover:to-pink-700"
                >
                  <Play className="w-4 h-4 mr-2" />
                  Play Again
                </Button>
                <Button variant="outline" onClick={() => setGameState("lobby")}>
                  Back to Lobby
                </Button>
              </div>
            </Card>
          </motion.div>
        )}
      </div>

      <ReportModal
        isOpen={reportModalOpen}
        onClose={() => setReportModalOpen(false)}
        reportedPlayer={reportedPlayer}
        gameId={currentRoom?.id || ""}
      />
    </div>
  );
}
