import { ArrowLeft, Users, Heart, Shield, MessageCircle, Camera, Flag, AlertTriangle } from "lucide-react";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "./ui/card";
import { Button } from "./ui/button";
import { Alert, AlertDescription } from "./ui/alert";
import { Badge } from "./ui/badge";
import { Separator } from "./ui/separator";

interface CommunityGuidelinesPageProps {
  onBack: () => void;
}

export function CommunityGuidelinesPage({ onBack }: CommunityGuidelinesPageProps) {
  const guidelines = [
    {
      id: "respect",
      title: "Treat Everyone with Respect",
      icon: Heart,
      color: "pink",
      rules: [
        {
          title: "Be Kind and Considerate",
          description: "Treat all members with dignity, respect, and kindness. We're all here looking for meaningful connections.",
          required: true
        },
        {
          title: "No Harassment or Bullying",
          description: "Harassment, bullying, stalking, or intimidating behavior towards other members is strictly prohibited.",
          required: true
        },
        {
          title: "Respect Boundaries",
          description: "If someone says no or asks you to stop contacting them, respect their wishes immediately.",
          required: true
        },
        {
          title: "Use Appropriate Language",
          description: "Keep your language respectful and appropriate. Excessive profanity or offensive language is not allowed.",
          required: false
        }
      ]
    },
    {
      id: "authentic",
      title: "Be Authentic and Honest",
      icon: Shield,
      color: "blue",
      rules: [
        {
          title: "Use Real Information",
          description: "Provide accurate information about yourself, including your age, location, and relationship status.",
          required: true
        },
        {
          title: "Use Your Own Photos",
          description: "Only use photos of yourself. Fake photos, celebrity pictures, or images of other people are not allowed.",
          required: true
        },
        {
          title: "One Account Per Person",
          description: "Each person may only have one active account. Creating multiple accounts is prohibited.",
          required: true
        },
        {
          title: "Be Honest About Intentions",
          description: "Be clear about what you're looking for and don't mislead others about your intentions.",
          required: false
        }
      ]
    },
    {
      id: "appropriate-content",
      title: "Keep Content Appropriate",
      icon: Camera,
      color: "green",
      rules: [
        {
          title: "No Nudity or Sexual Content",
          description: "Profile photos and messages must not contain nudity, sexual content, or suggestive material.",
          required: true
        },
        {
          title: "Family-Friendly Content",
          description: "All content should be appropriate for a general audience and reflect the respectful nature of our community.",
          required: true
        },
        {
          title: "No Promotional Content",
          description: "Don't use your profile to advertise businesses, services, or promote external websites.",
          required: true
        },
        {
          title: "Appropriate Profile Content",
          description: "Keep your bio and photos appropriate and relevant to dating. No political rants or controversial content.",
          required: false
        }
      ]
    },
    {
      id: "communication",
      title: "Communicate Responsibly",
      icon: MessageCircle,
      color: "purple",
      rules: [
        {
          title: "No Spam or Solicitation",
          description: "Don't send unsolicited promotional messages, spam, or try to solicit money or services.",
          required: true
        },
        {
          title: "Keep Conversations Meaningful",
          description: "Engage in genuine conversations. Copy-paste messages or inappropriate pickup lines are discouraged.",
          required: false
        },
        {
          title: "No Hate Speech",
          description: "Messages containing hate speech, discrimination, or threats based on race, religion, gender, or other characteristics are prohibited.",
          required: true
        },
        {
          title: "Protect Privacy",
          description: "Don't share other people's personal information without their consent.",
          required: true
        }
      ]
    }
  ];

  const prohibitedBehaviors = [
    "Creating fake profiles or using false information",
    "Sending sexually explicit messages or images",
    "Harassment, stalking, or threatening behavior",
    "Requesting money, gifts, or financial information",
    "Promoting businesses, services, or external websites",
    "Using hate speech or discriminatory language",
    "Sharing inappropriate or offensive content",
    "Creating multiple accounts or impersonating others",
    "Engaging in illegal activities or promoting them",
    "Violating intellectual property rights"
  ];

  const reportingReasons = [
    { reason: "Fake Profile", description: "User is using false information or fake photos" },
    { reason: "Inappropriate Content", description: "Profile contains inappropriate photos or information" },
    { reason: "Harassment", description: "User is harassing, bullying, or threatening others" },
    { reason: "Spam", description: "User is sending spam or promotional messages" },
    { reason: "Scam", description: "User appears to be running a scam or asking for money" },
    { reason: "Hate Speech", description: "User is using discriminatory or hateful language" }
  ];

  const consequences = [
    {
      level: "Warning",
      description: "First-time minor violations result in a warning and educational guidance.",
      color: "yellow"
    },
    {
      level: "Content Removal",
      description: "Inappropriate content is removed, and the user is notified.",
      color: "orange"
    },
    {
      level: "Temporary Suspension",
      description: "Account is temporarily suspended for repeated or serious violations.",
      color: "red"
    },
    {
      level: "Permanent Ban",
      description: "Severe violations or repeated offenses result in permanent account termination.",
      color: "red"
    }
  ];

  return (
    <div className="min-h-screen bg-background">
      {/* Header */}
      <div className="bg-gradient-to-r from-blue-600 to-purple-600 text-white">
        <div className="max-w-6xl mx-auto px-6 py-8">
          <div className="flex items-center gap-4 mb-6">
            <Button
              variant="ghost"
              size="sm"
              onClick={onBack}
              className="text-white hover:bg-white/20"
            >
              <ArrowLeft className="w-4 h-4 mr-2" />
              Back
            </Button>
          </div>
          
          <div className="text-center max-w-2xl mx-auto">
            <Users className="w-16 h-16 mx-auto mb-4 text-white" />
            <h1 className="text-4xl font-bold mb-4">Community Guidelines</h1>
            <p className="text-blue-100 text-lg">
              Help us create a safe, respectful, and welcoming environment for everyone on SinglesZonne.
            </p>
          </div>
        </div>
      </div>

      {/* Content */}
      <div className="max-w-6xl mx-auto px-6 py-12">
        {/* Introduction */}
        <div className="mb-12">
          <Card className="bg-gradient-to-r from-blue-50 to-purple-50 dark:from-blue-900/20 dark:to-purple-900/20 border-blue-200 dark:border-blue-800">
            <CardContent className="pt-6">
              <h2 className="text-2xl font-bold mb-4">Welcome to Our Community</h2>
              <p className="text-muted-foreground leading-relaxed mb-4">
                SinglesZonne is built on the foundation of respect, authenticity, and meaningful connections. 
                These guidelines help ensure that everyone can have a positive experience while looking for love and companionship.
              </p>
              <p className="text-muted-foreground leading-relaxed">
                By using SinglesZonne, you agree to follow these guidelines. Violations may result in warnings, 
                content removal, account suspension, or permanent bans depending on the severity and frequency of the violation.
              </p>
            </CardContent>
          </Card>
        </div>

        {/* Guidelines Sections */}
        <div className="grid gap-8 mb-12">
          {guidelines.map((section) => {
            const IconComponent = section.icon;
            return (
              <Card key={section.id}>
                <CardHeader>
                  <div className="flex items-center gap-3">
                    <div className={`p-3 rounded-lg bg-${section.color}-100 dark:bg-${section.color}-900/20`}>
                      <IconComponent className={`w-6 h-6 text-${section.color}-600`} />
                    </div>
                    <div>
                      <CardTitle className="text-2xl">{section.title}</CardTitle>
                    </div>
                  </div>
                </CardHeader>
                <CardContent>
                  <div className="grid gap-4">
                    {section.rules.map((rule, index) => (
                      <div key={index} className="border rounded-lg p-4">
                        <div className="flex items-start justify-between gap-3 mb-2">
                          <h4 className="font-semibold">{rule.title}</h4>
                          {rule.required && (
                            <Badge variant="destructive" className="shrink-0">
                              Required
                            </Badge>
                          )}
                        </div>
                        <p className="text-muted-foreground leading-relaxed">{rule.description}</p>
                      </div>
                    ))}
                  </div>
                </CardContent>
              </Card>
            );
          })}
        </div>

        {/* Prohibited Behaviors */}
        <div className="mb-12">
          <Card className="border-red-200 bg-red-50 dark:bg-red-900/20">
            <CardHeader>
              <div className="flex items-center gap-3">
                <AlertTriangle className="w-6 h-6 text-red-600" />
                <div>
                  <CardTitle className="text-red-800 dark:text-red-400">Prohibited Behaviors</CardTitle>
                  <CardDescription className="text-red-700 dark:text-red-500">
                    These actions are strictly forbidden and will result in immediate action
                  </CardDescription>
                </div>
              </div>
            </CardHeader>
            <CardContent>
              <div className="grid md:grid-cols-2 gap-3">
                {prohibitedBehaviors.map((behavior, index) => (
                  <div key={index} className="flex items-center gap-3 p-3 bg-white dark:bg-red-900/30 rounded-lg">
                    <div className="w-2 h-2 bg-red-600 rounded-full shrink-0"></div>
                    <span className="text-sm text-red-800 dark:text-red-300">{behavior}</span>
                  </div>
                ))}
              </div>
            </CardContent>
          </Card>
        </div>

        {/* Reporting */}
        <div className="mb-12">
          <Card>
            <CardHeader>
              <div className="flex items-center gap-3">
                <Flag className="w-6 h-6 text-orange-600" />
                <div>
                  <CardTitle>Reporting Violations</CardTitle>
                  <CardDescription>
                    Help us maintain a safe community by reporting inappropriate behavior
                  </CardDescription>
                </div>
              </div>
            </CardHeader>
            <CardContent>
              <p className="text-muted-foreground mb-6">
                If you encounter someone who violates these guidelines, please report them immediately. 
                All reports are reviewed by our moderation team within 24 hours.
              </p>
              
              <div className="grid md:grid-cols-2 gap-4 mb-6">
                {reportingReasons.map((item, index) => (
                  <div key={index} className="border rounded-lg p-4">
                    <h4 className="font-semibold mb-2">{item.reason}</h4>
                    <p className="text-sm text-muted-foreground">{item.description}</p>
                  </div>
                ))}
              </div>

              <Alert>
                <Flag className="w-4 h-4" />
                <AlertDescription>
                  To report a user, visit their profile and click the three dots menu, then select "Report User". 
                  Provide as much detail as possible to help our team investigate.
                </AlertDescription>
              </Alert>
            </CardContent>
          </Card>
        </div>

        {/* Consequences */}
        <div className="mb-12">
          <Card>
            <CardHeader>
              <CardTitle>Enforcement and Consequences</CardTitle>
              <CardDescription>
                Understanding what happens when guidelines are violated
              </CardDescription>
            </CardHeader>
            <CardContent>
              <div className="grid gap-4">
                {consequences.map((consequence, index) => (
                  <div key={index} className="flex items-center gap-4 p-4 border rounded-lg">
                    <div className={`w-3 h-3 bg-${consequence.color}-500 rounded-full shrink-0`}></div>
                    <div>
                      <h4 className="font-semibold">{consequence.level}</h4>
                      <p className="text-sm text-muted-foreground">{consequence.description}</p>
                    </div>
                  </div>
                ))}
              </div>

              <Separator className="my-6" />

              <div className="bg-muted/50 rounded-lg p-4">
                <h4 className="font-semibold mb-2">Appeals Process</h4>
                <p className="text-sm text-muted-foreground">
                  If you believe your account was suspended or banned in error, you can appeal the decision by 
                  contacting our support team at support@singleszonne.com. Include your username and a detailed 
                  explanation of why you believe the action was taken in error.
                </p>
              </div>
            </CardContent>
          </Card>
        </div>

        {/* Call to Action */}
        <div className="text-center">
          <Card className="bg-gradient-to-r from-purple-50 to-pink-50 dark:from-purple-900/20 dark:to-pink-900/20 border-purple-200 dark:border-purple-800">
            <CardContent className="pt-6">
              <h3 className="text-2xl font-bold mb-3">Let's Build Something Beautiful Together</h3>
              <p className="text-muted-foreground mb-6 max-w-2xl mx-auto">
                By following these guidelines, you're helping create a community where everyone feels safe, 
                respected, and excited to find their perfect match. Thank you for being part of the SinglesZonne family.
              </p>
              <div className="flex flex-col sm:flex-row gap-4 justify-center">
                <Button size="lg" className="bg-purple-600 hover:bg-purple-700">
                  Report a Violation
                </Button>
                <Button size="lg" variant="outline">
                  Contact Support
                </Button>
              </div>
            </CardContent>
          </Card>
        </div>
      </div>
    </div>
  );
}