import { useState } from 'react';
import { Card } from './ui/card';
import { Button } from './ui/button';
import { Badge } from './ui/badge';
import { Separator } from './ui/separator';
import { 
  ColorPicker, 
  ColorThemePreset, 
  ProfilePreview 
} from './ColorPicker';
import { useColorTheme } from './ColorThemeProvider';
import { 
  Palette, 
  RotateCcw, 
  Save,
  Sparkles,
  Wand2,
  Eye,
  Info
} from 'lucide-react';
import { Alert, AlertDescription } from './ui/alert';

export function ColorThemeTab() {
  const { theme, setTheme, resetToDefault, predefinedThemes } = useColorTheme();
  const [hasChanges, setHasChanges] = useState(false);
  
  const updateColor = (key: keyof typeof theme, value: string) => {
    const newTheme = { ...theme, [key]: value };
    setTheme(newTheme);
    setHasChanges(true);
  };

  const applyPreset = (presetName: string) => {
    const preset = predefinedThemes[presetName];
    if (preset) {
      setTheme(preset);
      setHasChanges(true);
    }
  };

  const saveChanges = () => {
    // Changes are automatically saved by the ColorThemeProvider
    setHasChanges(false);
  };

  const isCurrentPreset = (presetName: string) => {
    const preset = predefinedThemes[presetName];
    return JSON.stringify(preset) === JSON.stringify(theme);
  };

  return (
    <div className="space-y-6">
      {/* Header */}
      <div className="flex items-center justify-between">
        <div className="flex items-center gap-3">
          <div className="p-2 rounded-full bg-gradient-to-r from-purple-500 to-pink-500">
            <Palette className="w-6 h-6 text-white" />
          </div>
          <div>
            <h2 className="text-2xl font-bold">Profile Colors</h2>
            <p className="text-muted-foreground">Customize your profile's color scheme</p>
          </div>
        </div>
        
        <div className="flex items-center gap-2">
          <Button
            variant="outline"
            onClick={resetToDefault}
            className="flex items-center gap-2"
          >
            <RotateCcw className="w-4 h-4" />
            Reset to Default
          </Button>
          
          {hasChanges && (
            <Button
              onClick={saveChanges}
              className="flex items-center gap-2"
            >
              <Save className="w-4 h-4" />
              Save Changes
            </Button>
          )}
        </div>
      </div>

      {/* Info Alert */}
      <Alert>
        <Info className="h-4 w-4" />
        <AlertDescription>
          Your custom colors will be applied to various elements in your profile, including section headers, 
          badges, gradients, and accent elements. Changes are saved automatically and visible to other members.
        </AlertDescription>
      </Alert>

      <div className="grid grid-cols-1 lg:grid-cols-2 gap-8">
        {/* Color Customization Panel */}
        <div className="space-y-6">
          {/* Preset Themes */}
          <Card className="p-6">
            <div className="flex items-center gap-2 mb-4">
              <Wand2 className="w-5 h-5 text-primary" />
              <h3 className="text-lg font-semibold">Quick Themes</h3>
            </div>
            
            <div className="grid grid-cols-2 gap-3">
              {Object.entries(predefinedThemes).map(([name, colors]) => (
                <ColorThemePreset
                  key={name}
                  name={name.charAt(0).toUpperCase() + name.slice(1)}
                  colors={colors}
                  isActive={isCurrentPreset(name)}
                  onClick={() => applyPreset(name)}
                />
              ))}
            </div>
          </Card>

          {/* Custom Colors */}
          <Card className="p-6">
            <div className="flex items-center gap-2 mb-6">
              <Sparkles className="w-5 h-5 text-primary" />
              <h3 className="text-lg font-semibold">Custom Colors</h3>
            </div>
            
            <div className="space-y-6">
              <ColorPicker
                label="Primary Color"
                value={theme.primary}
                onChange={(value) => updateColor('primary', value)}
              />
              
              <ColorPicker
                label="Secondary Color"
                value={theme.secondary}
                onChange={(value) => updateColor('secondary', value)}
              />
              
              <ColorPicker
                label="Accent Color"
                value={theme.accent}
                onChange={(value) => updateColor('accent', value)}
              />
              
              <Separator />
              
              <div className="space-y-4">
                <h4 className="font-medium text-sm text-muted-foreground">Gradient Colors</h4>
                
                <ColorPicker
                  label="Gradient Start"
                  value={theme.gradient1}
                  onChange={(value) => updateColor('gradient1', value)}
                />
                
                <ColorPicker
                  label="Gradient Middle"
                  value={theme.gradient2}
                  onChange={(value) => updateColor('gradient2', value)}
                />
                
                <ColorPicker
                  label="Gradient End"
                  value={theme.gradient3}
                  onChange={(value) => updateColor('gradient3', value)}
                />
              </div>
            </div>
          </Card>

          {/* Color Usage Guide */}
          <Card className="p-6 bg-gradient-to-br from-blue-50 to-indigo-50 dark:from-blue-950/20 dark:to-indigo-950/20">
            <h4 className="font-semibold mb-3 text-blue-900 dark:text-blue-100">Color Usage Guide</h4>
            <div className="space-y-2 text-sm text-blue-800 dark:text-blue-200">
              <div className="flex items-center gap-2">
                <div className="w-3 h-3 rounded-full bg-gradient-to-r from-primary to-secondary"></div>
                <span><strong>Primary:</strong> Main section headers and icons</span>
              </div>
              <div className="flex items-center gap-2">
                <div className="w-3 h-3 rounded-full" style={{ backgroundColor: 'var(--color-profile-secondary)' }}></div>
                <span><strong>Secondary:</strong> Subheadings and highlights</span>
              </div>
              <div className="flex items-center gap-2">
                <div className="w-3 h-3 rounded-full" style={{ backgroundColor: 'var(--color-profile-accent)' }}></div>
                <span><strong>Accent:</strong> Badges and special elements</span>
              </div>
              <div className="flex items-center gap-2">
                <div className="w-3 h-3 rounded-full bg-gradient-to-r" style={{ 
                  background: `linear-gradient(135deg, ${theme.gradient1}, ${theme.gradient2}, ${theme.gradient3})`
                }}></div>
                <span><strong>Gradients:</strong> Backgrounds and accent bars</span>
              </div>
            </div>
          </Card>
        </div>

        {/* Preview Panel */}
        <div className="space-y-6">
          <ProfilePreview theme={theme} />
          
          {/* Current Theme Info */}
          <Card className="p-6">
            <h4 className="font-semibold mb-4 flex items-center gap-2">
              <Eye className="w-5 h-5" />
              Current Theme Values
            </h4>
            
            <div className="space-y-3 font-mono text-sm">
              <div className="flex items-center justify-between">
                <span className="text-muted-foreground">Primary:</span>
                <div className="flex items-center gap-2">
                  <div 
                    className="w-4 h-4 rounded border"
                    style={{ backgroundColor: theme.primary }}
                  />
                  <Badge variant="outline" className="text-xs">
                    {theme.primary}
                  </Badge>
                </div>
              </div>
              
              <div className="flex items-center justify-between">
                <span className="text-muted-foreground">Secondary:</span>
                <div className="flex items-center gap-2">
                  <div 
                    className="w-4 h-4 rounded border"
                    style={{ backgroundColor: theme.secondary }}
                  />
                  <Badge variant="outline" className="text-xs">
                    {theme.secondary}
                  </Badge>
                </div>
              </div>
              
              <div className="flex items-center justify-between">
                <span className="text-muted-foreground">Accent:</span>
                <div className="flex items-center gap-2">
                  <div 
                    className="w-4 h-4 rounded border"
                    style={{ backgroundColor: theme.accent }}
                  />
                  <Badge variant="outline" className="text-xs">
                    {theme.accent}
                  </Badge>
                </div>
              </div>
            </div>
          </Card>
        </div>
      </div>
    </div>
  );
}