import { createContext, useContext, useState, useEffect, ReactNode } from 'react';

interface ColorTheme {
  primary: string;
  secondary: string;
  accent: string;
  gradient1: string;
  gradient2: string;
  gradient3: string;
}

interface ColorThemeContextType {
  theme: ColorTheme;
  setTheme: (theme: ColorTheme) => void;
  resetToDefault: () => void;
  predefinedThemes: { [key: string]: ColorTheme };
}

const defaultTheme: ColorTheme = {
  primary: '#6366f1', // Indigo
  secondary: '#8b5cf6', // Purple
  accent: '#ec4899', // Pink
  gradient1: '#3b82f6', // Blue
  gradient2: '#8b5cf6', // Purple
  gradient3: '#ec4899', // Pink
};

const predefinedThemes = {
  default: defaultTheme,
  sunset: {
    primary: '#f59e0b', // Amber
    secondary: '#ef4444', // Red
    accent: '#ec4899', // Pink
    gradient1: '#f59e0b',
    gradient2: '#ef4444',
    gradient3: '#ec4899',
  },
  ocean: {
    primary: '#0ea5e9', // Sky blue
    secondary: '#06b6d4', // Cyan
    accent: '#10b981', // Emerald
    gradient1: '#0ea5e9',
    gradient2: '#06b6d4',
    gradient3: '#10b981',
  },
  forest: {
    primary: '#10b981', // Emerald
    secondary: '#22c55e', // Green
    accent: '#84cc16', // Lime
    gradient1: '#10b981',
    gradient2: '#22c55e',
    gradient3: '#84cc16',
  },
  berry: {
    primary: '#ec4899', // Pink
    secondary: '#a855f7', // Purple
    accent: '#6366f1', // Indigo
    gradient1: '#ec4899',
    gradient2: '#a855f7',
    gradient3: '#6366f1',
  },
  monochrome: {
    primary: '#374151', // Gray
    secondary: '#6b7280', // Gray
    accent: '#9ca3af', // Gray
    gradient1: '#374151',
    gradient2: '#6b7280',
    gradient3: '#9ca3af',
  },
};

const ColorThemeContext = createContext<ColorThemeContextType | undefined>(undefined);

export function ColorThemeProvider({ children }: { children: ReactNode }) {
  const [theme, setTheme] = useState<ColorTheme>(defaultTheme);

  // Load saved theme from localStorage
  useEffect(() => {
    const savedTheme = localStorage.getItem('profile-color-theme');
    if (savedTheme) {
      try {
        setTheme(JSON.parse(savedTheme));
      } catch (error) {
        console.error('Failed to parse saved theme:', error);
      }
    }
  }, []);

  // Apply theme to CSS custom properties
  useEffect(() => {
    const root = document.documentElement;
    
    // Apply custom colors to CSS variables
    root.style.setProperty('--color-profile-primary', theme.primary);
    root.style.setProperty('--color-profile-secondary', theme.secondary);
    root.style.setProperty('--color-profile-accent', theme.accent);
    root.style.setProperty('--color-profile-gradient-1', theme.gradient1);
    root.style.setProperty('--color-profile-gradient-2', theme.gradient2);
    root.style.setProperty('--color-profile-gradient-3', theme.gradient3);
    
    // Create gradient combinations
    root.style.setProperty(
      '--gradient-profile-primary',
      `linear-gradient(135deg, ${theme.gradient1}, ${theme.gradient2})`
    );
    root.style.setProperty(
      '--gradient-profile-secondary',
      `linear-gradient(135deg, ${theme.gradient2}, ${theme.gradient3})`
    );
    root.style.setProperty(
      '--gradient-profile-accent',
      `linear-gradient(135deg, ${theme.gradient1}, ${theme.gradient3})`
    );
    
    // Save to localStorage
    localStorage.setItem('profile-color-theme', JSON.stringify(theme));
  }, [theme]);

  const resetToDefault = () => {
    setTheme(defaultTheme);
  };

  return (
    <ColorThemeContext.Provider 
      value={{ 
        theme, 
        setTheme, 
        resetToDefault, 
        predefinedThemes 
      }}
    >
      {children}
    </ColorThemeContext.Provider>
  );
}

export function useColorTheme() {
  const context = useContext(ColorThemeContext);
  if (!context) {
    throw new Error('useColorTheme must be used within a ColorThemeProvider');
  }
  return context;
}

// Utility function to convert hex to RGB
export function hexToRgb(hex: string): { r: number; g: number; b: number } | null {
  const result = /^#?([a-f\d]{2})([a-f\d]{2})([a-f\d]{2})$/i.exec(hex);
  return result ? {
    r: parseInt(result[1], 16),
    g: parseInt(result[2], 16),
    b: parseInt(result[3], 16)
  } : null;
}

// Utility function to check if a color is light or dark
export function isLightColor(hex: string): boolean {
  const rgb = hexToRgb(hex);
  if (!rgb) return false;
  
  // Calculate luminance
  const luminance = (0.299 * rgb.r + 0.587 * rgb.g + 0.114 * rgb.b) / 255;
  return luminance > 0.5;
}