import { useState } from "react";
import { Header } from "./Header";
import { Footer } from "./Footer";
import { Button } from "./ui/button";
import { Card } from "./ui/card";
import { Badge } from "./ui/badge";
import { Megaphone, ExternalLink, Clock, MapPin, DollarSign, Users, Plus, Coins, Eye, MousePointer } from "lucide-react";
import { motion } from "motion/react";
import { ImageWithFallback } from "./figma/ImageWithFallback";
import { AdCreationModal } from "./AdCreationModal";
import { AdViewModal } from "./AdViewModal";
import { PointsStore } from "./PointsStore";
import { usePoints } from "./PointsProvider";

interface AdsPageProps {
  onNavigate: (view: string) => void;
  onOpenCookieConsent?: () => void;
}

interface Ad {
  id: number;
  title: string;
  description: string;
  image: string;
  category: "services" | "products" | "dating-tips";
  location?: string;
  price?: string;
  timeLeft?: string;
  isSponsored: boolean;
  link: string;
  type: "video" | "photo" | "text";
  duration: string;
  pointsReward: number;
  imageUrl?: string;
  videoUrl?: string;
  impressions: number;
  clicks: number;
  contactInfo: string;
}

export function AdsPage({ onNavigate, onOpenCookieConsent }: AdsPageProps) {
  const [activeCategory, setActiveCategory] = useState<string>("all");
  const [isAdCreationOpen, setIsAdCreationOpen] = useState(false);
  const [isAdViewOpen, setIsAdViewOpen] = useState(false);
  const [isPointsStoreOpen, setIsPointsStoreOpen] = useState(false);
  const [selectedAd, setSelectedAd] = useState<Ad | null>(null);
  const { points } = usePoints();
  
  const [ads] = useState<Ad[]>([
    {
      id: 1,
      title: "Professional Dating Photography",
      description: "Get stunning profile photos that get results. 50+ 5-star reviews!",
      image: "https://images.unsplash.com/photo-1554048612-b6a482b224b8?w=400&h=300&fit=crop",
      imageUrl: "https://images.unsplash.com/photo-1554048612-b6a482b224b8?w=400&h=300&fit=crop",
      category: "services",
      location: "Los Angeles, CA",
      price: "From $150",
      isSponsored: true,
      link: "#",
      type: "photo",
      duration: "30",
      pointsReward: 10,
      impressions: 12847,
      clicks: 892,
      contactInfo: "contact@photopro.com"
    },
    {
      id: 2,
      title: "Dating Confidence Masterclass",
      description: "Transform your dating life with proven strategies from relationship experts. Watch our exclusive 5-minute preview!",
      image: "https://images.unsplash.com/photo-1517245386807-bb43f82c33c4?w=400&h=300&fit=crop",
      imageUrl: "https://images.unsplash.com/photo-1517245386807-bb43f82c33c4?w=400&h=300&fit=crop",
      category: "dating-tips",
      price: "$97",
      isSponsored: false,
      link: "#",
      type: "video",
      duration: "14",
      pointsReward: 18,
      impressions: 8934,
      clicks: 623,
      contactInfo: "info@datingmasterclass.com"
    },
    {
      id: 3,
      title: "Premium Cologne Collection",
      description: "Irresistible fragrances that make lasting impressions on your dates.",
      image: "https://images.unsplash.com/photo-1541643600914-78b084683601?w=400&h=300&fit=crop",
      imageUrl: "https://images.unsplash.com/photo-1541643600914-78b084683601?w=400&h=300&fit=crop",
      category: "products",
      price: "$29.99",
      isSponsored: true,
      link: "#",
      type: "photo",
      duration: "7",
      pointsReward: 8,
      impressions: 15672,
      clicks: 1205,
      contactInfo: "orders@premiumscents.com"
    },
    {
      id: 4,
      title: "Dating Success Formula",
      description: "Discover the 3 secrets that guarantee dating success. Limited time offer - learn the psychology of attraction!",
      image: "https://images.unsplash.com/photo-1441986300917-64674bd600d8?w=400&h=300&fit=crop",
      category: "dating-tips",
      location: "Online Course",
      price: "Special: $47",
      isSponsored: true,
      link: "#",
      type: "text",
      duration: "7",
      pointsReward: 5,
      impressions: 9823,
      clicks: 456,
      contactInfo: "support@datingformula.com"
    },
    {
      id: 5,
      title: "Personal Style Transformation",
      description: "Get a complete style makeover from expert stylists. See the dramatic before/after results in our video showcase!",
      image: "https://images.unsplash.com/photo-1556157382-97eda2d62296?w=400&h=300&fit=crop",
      imageUrl: "https://images.unsplash.com/photo-1556157382-97eda2d62296?w=400&h=300&fit=crop",
      category: "services",
      location: "Multiple Cities",
      price: "From $299",
      isSponsored: true,
      link: "#",
      type: "video",
      duration: "30",
      pointsReward: 15,
      impressions: 11234,
      clicks: 789,
      contactInfo: "bookings@styletransform.com"
    },
    {
      id: 6,
      title: "Confidence Building Workshop",
      description: "Join thousands who've transformed their dating confidence. Special 50% off this week only!",
      image: "https://images.unsplash.com/photo-1556228720-195a672e8a03?w=400&h=300&fit=crop",
      category: "dating-tips",
      price: "Was $200, Now $99",
      timeLeft: "3 days left",
      isSponsored: false,
      link: "#",
      type: "text",
      duration: "14",
      pointsReward: 4,
      impressions: 6789,
      clicks: 234,
      contactInfo: "register@confidenceworkshop.com"
    },
    {
      id: 7,
      title: "Luxury Men's Grooming Kit",
      description: "Everything you need to look and smell amazing on every date. Premium quality guaranteed.",
      image: "https://images.unsplash.com/photo-1507003211169-0a1dd7228f2d?w=400&h=300&fit=crop",
      imageUrl: "https://images.unsplash.com/photo-1507003211169-0a1dd7228f2d?w=400&h=300&fit=crop",
      category: "products",
      price: "From $89",
      isSponsored: true,
      link: "#",
      type: "photo",
      duration: "60",
      pointsReward: 4,
      impressions: 18934,
      clicks: 1456,
      contactInfo: "orders@luxurygrooming.com"
    },
    {
      id: 8,
      title: "First Date Conversation Mastery",
      description: "Never run out of things to say! Learn the conversation techniques that create instant chemistry and connection.",
      image: "https://images.unsplash.com/photo-1521791136064-7986c2920216?w=400&h=300&fit=crop",
      category: "dating-tips",
      price: "$67",
      isSponsored: false,
      link: "#",
      type: "text",
      duration: "30",
      pointsReward: 3,
      impressions: 7456,
      clicks: 367,
      contactInfo: "learn@conversationmastery.com"
    }
  ]);

  const categories = [
    { id: "all", label: "All Ads", icon: Megaphone },
    { id: "services", label: "Services", icon: ExternalLink },
    { id: "products", label: "Products", icon: DollarSign },
    { id: "dating-tips", label: "Dating Tips", icon: Clock }
  ];

  const filteredAds = activeCategory === "all" 
    ? ads 
    : ads.filter(ad => ad.category === activeCategory);

  const handleAdClick = (ad: Ad) => {
    setSelectedAd(ad);
    setIsAdViewOpen(true);
  };

  return (
    <div className="min-h-screen bg-background">
      {/* Header */}
      <Header currentView="ads" onNavigate={onNavigate} />
      
      {/* Ads Banner */}
      <div className="bg-gradient-to-r from-orange-600 via-orange-700 to-orange-800 dark:from-orange-700 dark:via-orange-800 dark:to-orange-900 text-white py-11 relative overflow-hidden">
        {/* Moving gradient overlay */}
        <motion.div 
          className="absolute inset-0 bg-gradient-to-r from-transparent via-orange-300/30 to-transparent opacity-40"
          animate={{
            x: ['-100%', '100%']
          }}
          transition={{
            duration: 8,
            repeat: Infinity,
            ease: "linear",
            repeatDelay: 2
          }}
          style={{
            width: '200%',
            background: 'linear-gradient(90deg, transparent 0%, rgba(255,255,255,0.15) 25%, rgba(255,255,255,0.3) 50%, rgba(255,255,255,0.15) 75%, transparent 100%)'
          }}
        />
        
        {/* Additional gradient overlay */}
        <div className="absolute inset-0 bg-gradient-to-r from-transparent via-orange-600/20 to-orange-800/40 dark:from-transparent dark:via-orange-700/20 dark:to-orange-900/40"></div>
        
        <div className="max-w-7xl mx-auto px-6 text-center relative z-10">
          <motion.h1 
            className="text-4xl font-bold mb-4"
            initial={{ opacity: 0, y: 20 }}
            animate={{ opacity: 1, y: 0 }}
            transition={{ 
              duration: 0.8,
              ease: "easeOut"
            }}
          >
            <motion.span
              animate={{ 
                scale: [1, 1.05, 1],
              }}
              transition={{
                duration: 3,
                repeat: Infinity,
                ease: "easeInOut",
                repeatDelay: 0.5
              }}
            >
              📢 Dating & Lifestyle
            </motion.span>
          </motion.h1>
          <motion.p 
            className="text-lg text-orange-100 dark:text-orange-200"
            initial={{ opacity: 0, y: 20 }}
            animate={{ opacity: 1, y: 0 }}
            transition={{ 
              duration: 0.8,
              delay: 0.2,
              ease: "easeOut"
            }}
          >
            Discover services, products, and tips to enhance your dating life!
          </motion.p>
        </div>
      </div>

      {/* Main Content */}
      <div className="max-w-6xl mx-auto px-6 py-12">
        {/* Category Filters */}
        <div className="flex flex-wrap justify-center gap-2 mb-8">
          {categories.map((category) => {
            const Icon = category.icon;
            return (
              <button
                key={category.id}
                onClick={() => setActiveCategory(category.id)}
                className={`flex items-center gap-2 px-4 py-2 rounded-full transition-colors ${
                  activeCategory === category.id
                    ? "bg-orange-600 text-white"
                    : "bg-muted text-muted-foreground hover:bg-orange-100 hover:text-orange-700 dark:hover:bg-orange-900/20"
                }`}
              >
                <Icon className="w-4 h-4" />
                {category.label}
              </button>
            );
          })}
        </div>

        {/* Stats */}
        <div className="text-center mb-8">
          <div className="flex justify-center gap-8">
            <div className="text-center">
              <div className="text-2xl font-bold text-orange-600">{ads.length}</div>
              <div className="text-muted-foreground">Total Ads</div>
            </div>
            <div className="text-center">
              <div className="text-2xl font-bold text-green-600">{ads.filter(a => a.isSponsored).length}</div>
              <div className="text-muted-foreground">Sponsored</div>
            </div>
            <div className="text-center">
              <div className="text-2xl font-bold text-blue-600">{ads.filter(a => a.category === "dating-tips").length}</div>
              <div className="text-muted-foreground">Dating Tips</div>
            </div>
          </div>
        </div>

        {/* Ads Grid */}
        {filteredAds.length === 0 ? (
          <div className="text-center py-12">
            <Megaphone className="w-16 h-16 text-muted-foreground mx-auto mb-4" />
            <h3 className="text-xl font-semibold mb-2">No ads in this category</h3>
            <p className="text-muted-foreground">Try browsing a different category!</p>
          </div>
        ) : (
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
            {filteredAds.map((ad, index) => (
              <motion.div
                key={ad.id}
                initial={{ opacity: 0, y: 20 }}
                animate={{ opacity: 1, y: 0 }}
                transition={{ duration: 0.3, delay: index * 0.1 }}
              >
                <Card className="p-0 hover:shadow-lg transition-shadow cursor-pointer group" onClick={() => handleAdClick(ad)}>
                  <div className="relative">
                    <ImageWithFallback
                      src={ad.image}
                      alt={ad.title}
                      className="w-full h-48 object-cover rounded-t-lg"
                    />
                    {ad.isSponsored && (
                      <Badge className="absolute top-2 left-2 bg-orange-600 text-white">
                        Sponsored
                      </Badge>
                    )}
                    {ad.timeLeft && (
                      <Badge className="absolute top-2 right-2 bg-red-600 text-white">
                        {ad.timeLeft}
                      </Badge>
                    )}
                  </div>
                  
                  <div className="p-4 space-y-3">
                    <div>
                      <h3 className="font-semibold group-hover:text-orange-600 transition-colors">{ad.title}</h3>
                      <p className="text-muted-foreground text-sm mt-1">{ad.description}</p>
                    </div>
                    
                    <div className="flex items-center justify-between text-sm">
                      {ad.location && (
                        <div className="flex items-center gap-1 text-muted-foreground">
                          <MapPin className="w-4 h-4" />
                          <span>{ad.location}</span>
                        </div>
                      )}
                      {ad.price && (
                        <div className="font-semibold text-orange-600">
                          {ad.price}
                        </div>
                      )}
                    </div>
                    
                    {/* Points and Stats */}
                    <div className="flex items-center gap-4 text-xs text-muted-foreground mb-3">
                      <div className="flex items-center gap-1">
                        <Eye className="w-3 h-3" />
                        <span>{ad.impressions.toLocaleString()}</span>
                      </div>
                      <div className="flex items-center gap-1">
                        <MousePointer className="w-3 h-3" />
                        <span>{ad.clicks.toLocaleString()}</span>
                      </div>
                      <div className="flex items-center gap-1 text-orange-600">
                        <Coins className="w-3 h-3" />
                        <span className="font-semibold">+{ad.pointsReward} pts</span>
                      </div>
                    </div>
                    
                    <div className="flex items-center justify-between pt-2">
                      <Badge variant="secondary" className="text-xs capitalize">
                        {ad.category.replace("-", " ")}
                      </Badge>
                      <div className="flex gap-2">
                        <Button size="sm" variant="outline" className="group-hover:bg-orange-600 group-hover:text-white transition-colors">
                          <Coins className="w-4 h-4 mr-1" />
                          Watch & Earn
                        </Button>
                      </div>
                    </div>
                  </div>
                </Card>
              </motion.div>
            ))}
          </div>
        )}

        {/* Call to Action */}
        <div className="text-center mt-12 p-8 bg-gradient-to-r from-orange-50 to-orange-100 dark:from-orange-900/20 dark:to-orange-800/20 rounded-lg">
          <h3 className="text-xl font-semibold mb-4">Want to advertise with us?</h3>
          <p className="text-muted-foreground mb-6">
            Reach thousands of singles looking for dating services and lifestyle products.
          </p>
          <Button 
            onClick={() => setIsAdCreationOpen(true)}
            className="bg-orange-600 hover:bg-orange-700"
          >
            <Plus className="w-4 h-4 mr-2" />
            Post Your Ad
          </Button>
        </div>
      </div>
      
      {/* Footer */}
      <Footer onNavigate={onNavigate} onOpenCookieConsent={onOpenCookieConsent} />
      
      {/* Ad Creation Modal */}
      <AdCreationModal 
        isOpen={isAdCreationOpen} 
        onClose={() => setIsAdCreationOpen(false)} 
      />
      
      {/* Ad View Modal */}
      <AdViewModal
        isOpen={isAdViewOpen}
        onClose={() => setIsAdViewOpen(false)}
        ad={selectedAd}
      />
      
      {/* Points Store Modal */}
      <PointsStore 
        isOpen={isPointsStoreOpen} 
        onClose={() => setIsPointsStoreOpen(false)} 
      />
    </div>
  );
}