import { useState } from "react";
import { Button } from "./ui/button";
import { Card } from "./ui/card";
import { Badge } from "./ui/badge";
import { Input } from "./ui/input";
import { Textarea } from "./ui/textarea";
import { Alert, AlertDescription } from "./ui/alert";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "./ui/tabs";
import { Dialog, DialogContent, DialogDescription, DialogHeader, DialogTitle } from "./ui/dialog";
import { 
  Shield, 
  CheckCircle, 
  X, 
  Clock, 
  Search, 
  Filter,
  Eye,
  Trash2,
  UserCheck,
  UserX,
  FileImage,
  AlertTriangle,
  Calendar,
  Download
} from "lucide-react";
import { ImageWithFallback } from "./figma/ImageWithFallback";

interface VerificationRequest {
  id: string;
  userId: string;
  userName: string;
  userEmail: string;
  userPhoto: string;
  documentType: 'drivers_license' | 'passport' | 'government_id';
  frontImage: string;
  backImage?: string;
  submittedAt: string;
  status: 'pending' | 'approved' | 'rejected';
  reviewedAt?: string;
  reviewedBy?: string;
  rejectionReason?: string;
  autoDeleteAt?: string;
}

interface VerifiedUser {
  id: string;
  userName: string;
  userEmail: string;
  userPhoto: string;
  verifiedAt: string;
  verifiedBy: string;
  documentType: string;
}

export function AdminVerificationManagement() {
  const [activeTab, setActiveTab] = useState('pending');
  const [searchTerm, setSearchTerm] = useState('');
  const [selectedRequest, setSelectedRequest] = useState<VerificationRequest | null>(null);
  const [showReviewModal, setShowReviewModal] = useState(false);
  const [reviewAction, setReviewAction] = useState<'approve' | 'reject' | null>(null);
  const [rejectionReason, setRejectionReason] = useState('');

  // Mock data for verification requests
  const [verificationRequests, setVerificationRequests] = useState<VerificationRequest[]>([
    {
      id: '1',
      userId: 'user1',
      userName: 'Sarah Johnson',
      userEmail: 'sarah.johnson@email.com',
      userPhoto: 'https://images.unsplash.com/photo-1494790108755-2616b612b47c?crop=entropy&cs=tinysrgb&fit=max&fm=jpg&q=80&w=400',
      documentType: 'drivers_license',
      frontImage: 'https://images.unsplash.com/photo-1554224155-6726b3ff858f?crop=entropy&cs=tinysrgb&fit=max&fm=jpg&q=80&w=400',
      backImage: 'https://images.unsplash.com/photo-1554224155-8d04cb21cd6c?crop=entropy&cs=tinysrgb&fit=max&fm=jpg&q=80&w=400',
      submittedAt: '2024-01-20T14:30:00Z',
      status: 'pending'
    },
    {
      id: '2',
      userId: 'user2',
      userName: 'Michael Chen',
      userEmail: 'michael.chen@email.com',
      userPhoto: 'https://images.unsplash.com/photo-1507003211169-0a1dd7228f2d?crop=entropy&cs=tinysrgb&fit=max&fm=jpg&q=80&w=400',
      documentType: 'passport',
      frontImage: 'https://images.unsplash.com/photo-1578662996442-48f60103fc96?crop=entropy&cs=tinysrgb&fit=max&fm=jpg&q=80&w=400',
      submittedAt: '2024-01-19T09:15:00Z',
      status: 'pending'
    },
    {
      id: '3',
      userId: 'user3',
      userName: 'Emma Rodriguez',
      userEmail: 'emma.rodriguez@email.com',
      userPhoto: 'https://images.unsplash.com/photo-1438761681033-6461ffad8d80?crop=entropy&cs=tinysrgb&fit=max&fm=jpg&q=80&w=400',
      documentType: 'government_id',
      frontImage: 'https://images.unsplash.com/photo-1554224155-6726b3ff858f?crop=entropy&cs=tinysrgb&fit=max&fm=jpg&q=80&w=400',
      backImage: 'https://images.unsplash.com/photo-1554224155-8d04cb21cd6c?crop=entropy&cs=tinysrgb&fit=max&fm=jpg&q=80&w=400',
      submittedAt: '2024-01-18T16:45:00Z',
      status: 'approved',
      reviewedAt: '2024-01-19T10:00:00Z',
      reviewedBy: 'Admin User',
      autoDeleteAt: '2024-01-26T10:00:00Z'
    }
  ]);

  // Mock data for verified users
  const [verifiedUsers, setVerifiedUsers] = useState<VerifiedUser[]>([
    {
      id: 'user3',
      userName: 'Emma Rodriguez',
      userEmail: 'emma.rodriguez@email.com',
      userPhoto: 'https://images.unsplash.com/photo-1438761681033-6461ffad8d80?crop=entropy&cs=tinysrgb&fit=max&fm=jpg&q=80&w=400',
      verifiedAt: '2024-01-19T10:00:00Z',
      verifiedBy: 'Admin User',
      documentType: 'Government ID'
    },
    {
      id: 'user4',
      userName: 'David Kim',
      userEmail: 'david.kim@email.com',
      userPhoto: 'https://images.unsplash.com/photo-1472099645785-5658abf4ff4e?crop=entropy&cs=tinysrgb&fit=max&fm=jpg&q=80&w=400',
      verifiedAt: '2024-01-17T14:30:00Z',
      verifiedBy: 'Admin User',
      documentType: 'Driver\'s License'
    }
  ]);

  const documentTypeLabels = {
    drivers_license: 'Driver\'s License',
    passport: 'Passport',
    government_id: 'Government ID'
  };

  const getStatusColor = (status: string) => {
    switch (status) {
      case 'approved': return 'bg-green-500';
      case 'rejected': return 'bg-red-500';
      default: return 'bg-yellow-500';
    }
  };

  const getStatusIcon = (status: string) => {
    switch (status) {
      case 'approved': return <CheckCircle className="w-4 h-4" />;
      case 'rejected': return <X className="w-4 h-4" />;
      default: return <Clock className="w-4 h-4" />;
    }
  };

  const handleReviewRequest = (request: VerificationRequest, action: 'approve' | 'reject') => {
    setSelectedRequest(request);
    setReviewAction(action);
    setShowReviewModal(true);
    if (action === 'approve') {
      setRejectionReason('');
    }
  };

  const submitReview = () => {
    if (!selectedRequest || !reviewAction) return;

    const updatedRequest: VerificationRequest = {
      ...selectedRequest,
      status: reviewAction,
      reviewedAt: new Date().toISOString(),
      reviewedBy: 'Current Admin',
      rejectionReason: reviewAction === 'reject' ? rejectionReason : undefined,
      autoDeleteAt: reviewAction === 'approve' ? 
        new Date(Date.now() + 7 * 24 * 60 * 60 * 1000).toISOString() : undefined // Delete in 7 days
    };

    setVerificationRequests(prev => 
      prev.map(req => req.id === selectedRequest.id ? updatedRequest : req)
    );

    // If approved, add to verified users
    if (reviewAction === 'approve') {
      const newVerifiedUser: VerifiedUser = {
        id: selectedRequest.userId,
        userName: selectedRequest.userName,
        userEmail: selectedRequest.userEmail,
        userPhoto: selectedRequest.userPhoto,
        verifiedAt: new Date().toISOString(),
        verifiedBy: 'Current Admin',
        documentType: documentTypeLabels[selectedRequest.documentType]
      };

      setVerifiedUsers(prev => [newVerifiedUser, ...prev]);
    }

    setShowReviewModal(false);
    setSelectedRequest(null);
    setReviewAction(null);
    setRejectionReason('');
  };

  const removeVerification = (userId: string) => {
    setVerifiedUsers(prev => prev.filter(user => user.id !== userId));
    // In a real app, this would also update the user's verification status
  };

  const deleteDocuments = (requestId: string) => {
    setVerificationRequests(prev =>
      prev.map(req => req.id === requestId ? {
        ...req,
        frontImage: 'DELETED',
        backImage: 'DELETED',
        autoDeleteAt: new Date().toISOString()
      } : req)
    );
  };

  const filteredRequests = verificationRequests.filter(request => {
    const matchesSearch = request.userName.toLowerCase().includes(searchTerm.toLowerCase()) ||
                         request.userEmail.toLowerCase().includes(searchTerm.toLowerCase());
    const matchesStatus = activeTab === 'all' || request.status === activeTab;
    return matchesSearch && matchesStatus;
  });

  const filteredVerifiedUsers = verifiedUsers.filter(user =>
    user.userName.toLowerCase().includes(searchTerm.toLowerCase()) ||
    user.userEmail.toLowerCase().includes(searchTerm.toLowerCase())
  );

  const pendingCount = verificationRequests.filter(req => req.status === 'pending').length;
  const approvedCount = verificationRequests.filter(req => req.status === 'approved').length;
  const rejectedCount = verificationRequests.filter(req => req.status === 'rejected').length;

  return (
    <div className="space-y-6">
      <div className="flex items-center justify-between">
        <div>
          <h2 className="text-2xl font-bold flex items-center gap-2">
            <Shield className="w-6 h-6" />
            Verification Management
          </h2>
          <p className="text-muted-foreground">
            Review and manage member verification requests
          </p>
        </div>
        
        <div className="text-right">
          <div className="text-sm text-muted-foreground">
            Pending Reviews: <span className="font-semibold text-yellow-600">{pendingCount}</span>
          </div>
          <div className="text-sm text-muted-foreground">
            Total Verified: <span className="font-semibold text-green-600">{verifiedUsers.length}</span>
          </div>
        </div>
      </div>

      {/* Search and Filters */}
      <Card className="p-4">
        <div className="flex gap-4">
          <div className="flex-1">
            <div className="relative">
              <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-muted-foreground w-4 h-4" />
              <Input
                placeholder="Search by name or email..."
                value={searchTerm}
                onChange={(e) => setSearchTerm(e.target.value)}
                className="pl-10"
              />
            </div>
          </div>
          <Button variant="outline" size="sm">
            <Filter className="w-4 h-4 mr-2" />
            Filters
          </Button>
        </div>
      </Card>

      {/* Tabs for different views */}
      <Tabs value={activeTab} onValueChange={setActiveTab}>
        <TabsList className="grid w-full grid-cols-5">
          <TabsTrigger value="pending" className="flex items-center gap-2">
            <Clock className="w-4 h-4" />
            Pending ({pendingCount})
          </TabsTrigger>
          <TabsTrigger value="approved" className="flex items-center gap-2">
            <CheckCircle className="w-4 h-4" />
            Approved ({approvedCount})
          </TabsTrigger>
          <TabsTrigger value="rejected" className="flex items-center gap-2">
            <X className="w-4 h-4" />
            Rejected ({rejectedCount})
          </TabsTrigger>
          <TabsTrigger value="verified" className="flex items-center gap-2">
            <Shield className="w-4 h-4" />
            Verified Users ({verifiedUsers.length})
          </TabsTrigger>
          <TabsTrigger value="all">All Requests</TabsTrigger>
        </TabsList>

        <TabsContent value="pending" className="space-y-4">
          <Alert>
            <AlertTriangle className="h-4 w-4" />
            <AlertDescription>
              {pendingCount} verification requests require your review. 
              Please process them within 48 hours to maintain good user experience.
            </AlertDescription>
          </Alert>

          {filteredRequests.filter(req => req.status === 'pending').map((request) => (
            <Card key={request.id} className="p-6">
              <div className="flex items-start gap-4">
                <ImageWithFallback
                  src={request.userPhoto}
                  alt={request.userName}
                  className="w-16 h-16 rounded-full object-cover"
                />
                
                <div className="flex-1">
                  <div className="flex items-center justify-between mb-2">
                    <h3 className="font-semibold">{request.userName}</h3>
                    <Badge className={`${getStatusColor(request.status)} text-white`}>
                      {getStatusIcon(request.status)}
                      <span className="ml-1 capitalize">{request.status}</span>
                    </Badge>
                  </div>
                  
                  <p className="text-sm text-muted-foreground mb-2">{request.userEmail}</p>
                  <p className="text-sm mb-2">
                    <strong>Document Type:</strong> {documentTypeLabels[request.documentType]}
                  </p>
                  <p className="text-sm text-muted-foreground">
                    <Calendar className="w-3 h-3 inline mr-1" />
                    Submitted: {new Date(request.submittedAt).toLocaleString()}
                  </p>
                </div>

                <div className="flex gap-2">
                  <Button
                    variant="outline"
                    size="sm"
                    onClick={() => setSelectedRequest(request)}
                  >
                    <Eye className="w-4 h-4 mr-1" />
                    View
                  </Button>
                  <Button
                    variant="outline"
                    size="sm"
                    onClick={() => handleReviewRequest(request, 'approve')}
                    className="text-green-600 border-green-200 hover:bg-green-50"
                  >
                    <CheckCircle className="w-4 h-4 mr-1" />
                    Approve
                  </Button>
                  <Button
                    variant="outline"
                    size="sm"
                    onClick={() => handleReviewRequest(request, 'reject')}
                    className="text-red-600 border-red-200 hover:bg-red-50"
                  >
                    <X className="w-4 h-4 mr-1" />
                    Reject
                  </Button>
                </div>
              </div>
            </Card>
          ))}
        </TabsContent>

        <TabsContent value="approved" className="space-y-4">
          {filteredRequests.filter(req => req.status === 'approved').map((request) => (
            <Card key={request.id} className="p-6">
              <div className="flex items-start gap-4">
                <ImageWithFallback
                  src={request.userPhoto}
                  alt={request.userName}
                  className="w-16 h-16 rounded-full object-cover"
                />
                
                <div className="flex-1">
                  <div className="flex items-center justify-between mb-2">
                    <h3 className="font-semibold">{request.userName}</h3>
                    <Badge className="bg-green-500 text-white">
                      <CheckCircle className="w-3 h-3 mr-1" />
                      Approved
                    </Badge>
                  </div>
                  
                  <p className="text-sm text-muted-foreground mb-2">{request.userEmail}</p>
                  <p className="text-sm mb-1">
                    <strong>Document Type:</strong> {documentTypeLabels[request.documentType]}
                  </p>
                  <p className="text-sm text-muted-foreground mb-1">
                    Approved by: {request.reviewedBy} on {request.reviewedAt && new Date(request.reviewedAt).toLocaleString()}
                  </p>
                  {request.autoDeleteAt && (
                    <p className="text-sm text-orange-600">
                      <AlertTriangle className="w-3 h-3 inline mr-1" />
                      Documents auto-delete: {new Date(request.autoDeleteAt).toLocaleString()}
                    </p>
                  )}
                </div>

                <div className="flex gap-2">
                  <Button
                    variant="outline"
                    size="sm"
                    onClick={() => deleteDocuments(request.id)}
                    className="text-orange-600 border-orange-200 hover:bg-orange-50"
                  >
                    <Trash2 className="w-4 h-4 mr-1" />
                    Delete Documents
                  </Button>
                </div>
              </div>
            </Card>
          ))}
        </TabsContent>

        <TabsContent value="rejected" className="space-y-4">
          {filteredRequests.filter(req => req.status === 'rejected').map((request) => (
            <Card key={request.id} className="p-6">
              <div className="flex items-start gap-4">
                <ImageWithFallback
                  src={request.userPhoto}
                  alt={request.userName}
                  className="w-16 h-16 rounded-full object-cover"
                />
                
                <div className="flex-1">
                  <div className="flex items-center justify-between mb-2">
                    <h3 className="font-semibold">{request.userName}</h3>
                    <Badge className="bg-red-500 text-white">
                      <X className="w-3 h-3 mr-1" />
                      Rejected
                    </Badge>
                  </div>
                  
                  <p className="text-sm text-muted-foreground mb-2">{request.userEmail}</p>
                  <p className="text-sm mb-1">
                    <strong>Document Type:</strong> {documentTypeLabels[request.documentType]}
                  </p>
                  <p className="text-sm text-muted-foreground mb-1">
                    Rejected by: {request.reviewedBy} on {request.reviewedAt && new Date(request.reviewedAt).toLocaleString()}
                  </p>
                  {request.rejectionReason && (
                    <p className="text-sm text-red-600">
                      <strong>Reason:</strong> {request.rejectionReason}
                    </p>
                  )}
                </div>
              </div>
            </Card>
          ))}
        </TabsContent>

        <TabsContent value="verified" className="space-y-4">
          {filteredVerifiedUsers.map((user) => (
            <Card key={user.id} className="p-6">
              <div className="flex items-start gap-4">
                <ImageWithFallback
                  src={user.userPhoto}
                  alt={user.userName}
                  className="w-16 h-16 rounded-full object-cover"
                />
                
                <div className="flex-1">
                  <div className="flex items-center justify-between mb-2">
                    <h3 className="font-semibold">{user.userName}</h3>
                    <Badge className="bg-blue-500 text-white">
                      <Shield className="w-3 h-3 mr-1" />
                      Verified
                    </Badge>
                  </div>
                  
                  <p className="text-sm text-muted-foreground mb-2">{user.userEmail}</p>
                  <p className="text-sm mb-1">
                    <strong>Document Type:</strong> {user.documentType}
                  </p>
                  <p className="text-sm text-muted-foreground">
                    Verified by: {user.verifiedBy} on {new Date(user.verifiedAt).toLocaleString()}
                  </p>
                </div>

                <div className="flex gap-2">
                  <Button
                    variant="outline"
                    size="sm"
                    onClick={() => removeVerification(user.id)}
                    className="text-red-600 border-red-200 hover:bg-red-50"
                  >
                    <UserX className="w-4 h-4 mr-1" />
                    Remove Verification
                  </Button>
                </div>
              </div>
            </Card>
          ))}
        </TabsContent>

        <TabsContent value="all" className="space-y-4">
          {filteredRequests.map((request) => (
            <Card key={request.id} className="p-6">
              <div className="flex items-start gap-4">
                <ImageWithFallback
                  src={request.userPhoto}
                  alt={request.userName}
                  className="w-16 h-16 rounded-full object-cover"
                />
                
                <div className="flex-1">
                  <div className="flex items-center justify-between mb-2">
                    <h3 className="font-semibold">{request.userName}</h3>
                    <Badge className={`${getStatusColor(request.status)} text-white`}>
                      {getStatusIcon(request.status)}
                      <span className="ml-1 capitalize">{request.status}</span>
                    </Badge>
                  </div>
                  
                  <p className="text-sm text-muted-foreground mb-2">{request.userEmail}</p>
                  <p className="text-sm mb-1">
                    <strong>Document Type:</strong> {documentTypeLabels[request.documentType]}
                  </p>
                  <p className="text-sm text-muted-foreground">
                    <Calendar className="w-3 h-3 inline mr-1" />
                    Submitted: {new Date(request.submittedAt).toLocaleString()}
                  </p>
                </div>

                <div className="flex gap-2">
                  <Button
                    variant="outline"
                    size="sm"
                    onClick={() => setSelectedRequest(request)}
                  >
                    <Eye className="w-4 h-4 mr-1" />
                    View
                  </Button>
                  {request.status === 'pending' && (
                    <>
                      <Button
                        variant="outline"
                        size="sm"
                        onClick={() => handleReviewRequest(request, 'approve')}
                        className="text-green-600 border-green-200 hover:bg-green-50"
                      >
                        <CheckCircle className="w-4 h-4 mr-1" />
                        Approve
                      </Button>
                      <Button
                        variant="outline"
                        size="sm"
                        onClick={() => handleReviewRequest(request, 'reject')}
                        className="text-red-600 border-red-200 hover:bg-red-50"
                      >
                        <X className="w-4 h-4 mr-1" />
                        Reject
                      </Button>
                    </>
                  )}
                </div>
              </div>
            </Card>
          ))}
        </TabsContent>
      </Tabs>

      {/* Document View Modal */}
      {selectedRequest && !showReviewModal && (
        <Dialog open={!!selectedRequest} onOpenChange={() => setSelectedRequest(null)}>
          <DialogContent className="max-w-4xl">
            <DialogHeader>
              <DialogTitle>
                Verification Documents - {selectedRequest.userName}
              </DialogTitle>
              <DialogDescription>
                Review the submitted verification documents
              </DialogDescription>
            </DialogHeader>

            <div className="space-y-6">
              <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                <div>
                  <h4 className="font-medium mb-2">Front of Document</h4>
                  {selectedRequest.frontImage === 'DELETED' ? (
                    <div className="w-full h-64 bg-gray-100 rounded-lg flex items-center justify-center">
                      <p className="text-gray-500">Document has been deleted</p>
                    </div>
                  ) : (
                    <ImageWithFallback
                      src={selectedRequest.frontImage}
                      alt="Front of document"
                      className="w-full h-64 object-cover rounded-lg border"
                    />
                  )}
                </div>

                {selectedRequest.backImage && (
                  <div>
                    <h4 className="font-medium mb-2">Back of Document</h4>
                    {selectedRequest.backImage === 'DELETED' ? (
                      <div className="w-full h-64 bg-gray-100 rounded-lg flex items-center justify-center">
                        <p className="text-gray-500">Document has been deleted</p>
                      </div>
                    ) : (
                      <ImageWithFallback
                        src={selectedRequest.backImage}
                        alt="Back of document"
                        className="w-full h-64 object-cover rounded-lg border"
                      />
                    )}
                  </div>
                )}
              </div>

              <div className="grid grid-cols-1 md:grid-cols-2 gap-4 text-sm">
                <div>
                  <p><strong>Document Type:</strong> {documentTypeLabels[selectedRequest.documentType]}</p>
                  <p><strong>Submitted:</strong> {new Date(selectedRequest.submittedAt).toLocaleString()}</p>
                  <p><strong>Status:</strong> {selectedRequest.status}</p>
                </div>
                <div>
                  <p><strong>User:</strong> {selectedRequest.userName}</p>
                  <p><strong>Email:</strong> {selectedRequest.userEmail}</p>
                  {selectedRequest.reviewedBy && (
                    <p><strong>Reviewed by:</strong> {selectedRequest.reviewedBy}</p>
                  )}
                </div>
              </div>

              {selectedRequest.status === 'pending' && (
                <div className="flex gap-3">
                  <Button
                    onClick={() => handleReviewRequest(selectedRequest, 'approve')}
                    className="flex-1 bg-green-600 hover:bg-green-700"
                  >
                    <CheckCircle className="w-4 h-4 mr-2" />
                    Approve Verification
                  </Button>
                  <Button
                    onClick={() => handleReviewRequest(selectedRequest, 'reject')}
                    variant="outline"
                    className="flex-1 text-red-600 border-red-200 hover:bg-red-50"
                  >
                    <X className="w-4 h-4 mr-2" />
                    Reject Verification
                  </Button>
                </div>
              )}
            </div>
          </DialogContent>
        </Dialog>
      )}

      {/* Review Action Modal */}
      {showReviewModal && selectedRequest && (
        <Dialog open={showReviewModal} onOpenChange={setShowReviewModal}>
          <DialogContent>
            <DialogHeader>
              <DialogTitle>
                {reviewAction === 'approve' ? 'Approve' : 'Reject'} Verification
              </DialogTitle>
              <DialogDescription>
                {reviewAction === 'approve' 
                  ? 'Confirm that you want to approve this verification request.'
                  : 'Please provide a reason for rejecting this verification request.'
                }
              </DialogDescription>
            </DialogHeader>

            <div className="space-y-4">
              <div className="p-4 border rounded-lg">
                <p className="font-medium">{selectedRequest.userName}</p>
                <p className="text-sm text-muted-foreground">{selectedRequest.userEmail}</p>
                <p className="text-sm">Document: {documentTypeLabels[selectedRequest.documentType]}</p>
              </div>

              {reviewAction === 'reject' && (
                <div>
                  <label className="block font-medium mb-2">Rejection Reason *</label>
                  <Textarea
                    value={rejectionReason}
                    onChange={(e) => setRejectionReason(e.target.value)}
                    placeholder="Please explain why this verification is being rejected..."
                    rows={3}
                  />
                </div>
              )}

              {reviewAction === 'approve' && (
                <Alert>
                  <CheckCircle className="h-4 w-4" />
                  <AlertDescription>
                    The user will receive verified status and the uploaded documents will be automatically 
                    deleted from our servers within 7 days for security purposes.
                  </AlertDescription>
                </Alert>
              )}

              <div className="flex gap-3">
                <Button 
                  onClick={submitReview}
                  disabled={reviewAction === 'reject' && !rejectionReason.trim()}
                  className={reviewAction === 'approve' ? 'bg-green-600 hover:bg-green-700' : 'bg-red-600 hover:bg-red-700'}
                >
                  {reviewAction === 'approve' ? (
                    <>
                      <CheckCircle className="w-4 h-4 mr-2" />
                      Approve Verification
                    </>
                  ) : (
                    <>
                      <X className="w-4 h-4 mr-2" />
                      Reject Verification
                    </>
                  )}
                </Button>
                <Button variant="outline" onClick={() => setShowReviewModal(false)}>
                  Cancel
                </Button>
              </div>
            </div>
          </DialogContent>
        </Dialog>
      )}
    </div>
  );
}