import { useState } from "react";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "./ui/card";
import { Button } from "./ui/button";
import { Badge } from "./ui/badge";
import { Alert, AlertDescription } from "./ui/alert";
import { Separator } from "./ui/separator";
import { Textarea } from "./ui/textarea";
import { Label } from "./ui/label";
import { 
  User, 
  CheckCircle, 
  XCircle, 
  Clock, 
  Edit3,
  MessageSquare,
  Calendar,
  AlertTriangle,
  Filter
} from "lucide-react";
import { Dialog, DialogContent, DialogDescription, DialogHeader, DialogTitle, DialogTrigger } from "./ui/dialog";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "./ui/select";

interface UsernameRequest {
  id: string;
  userId: string;
  currentUsername: string;
  requestedUsername: string;
  reason: string;
  status: 'pending' | 'approved' | 'rejected';
  requestDate: string;
  reviewDate?: string;
  reviewedBy?: string;
  adminComment?: string;
  userEmail: string;
  userName: string;
}

// Mock data for username change requests
const mockUsernameRequests: UsernameRequest[] = [
  {
    id: "req-1",
    userId: "user-123",
    currentUsername: "john_doe_87",
    requestedUsername: "john_photographer",
    reason: "I want my username to reflect my profession as a photographer",
    status: "pending",
    requestDate: "2025-01-20",
    userEmail: "john@example.com",
    userName: "John Doe"
  },
  {
    id: "req-2", 
    userId: "user-456",
    currentUsername: "sarah_music",
    requestedUsername: "sarah_violinist",
    reason: "More specific to my musical instrument specialization",
    status: "pending",
    requestDate: "2025-01-19",
    userEmail: "sarah@example.com",
    userName: "Sarah Music"
  },
  {
    id: "req-3",
    userId: "user-789",
    currentUsername: "mike_runner",
    requestedUsername: "mike_marathon",
    reason: "I specialize in marathon running and want to reflect that in my username",
    status: "approved",
    requestDate: "2025-01-18",
    reviewDate: "2025-01-19",
    reviewedBy: "Admin Sarah",
    adminComment: "Approved - username appropriately reflects user's athletic focus",
    userEmail: "mike@example.com",
    userName: "Mike Runner"
  },
  {
    id: "req-4",
    userId: "user-101",
    currentUsername: "lisa_art",
    requestedUsername: "lisa_creative",
    reason: "I do more than just art, I'm a creative professional in multiple fields",
    status: "rejected",
    requestDate: "2025-01-17",
    reviewDate: "2025-01-18",
    reviewedBy: "Admin Mike",
    adminComment: "Too vague and doesn't add value. Please be more specific about your creative focus.",
    userEmail: "lisa@example.com",
    userName: "Lisa Art"
  },
  {
    id: "req-5",
    userId: "user-202",
    currentUsername: "alex_travel",
    requestedUsername: "alex_wanderlust",
    reason: "Wanderlust better captures my passion for exploring the world",
    status: "pending",
    requestDate: "2025-01-16",
    userEmail: "alex@example.com",
    userName: "Alex Travel"
  }
];

export function AdminUsernameRequests() {
  const [requests, setRequests] = useState<UsernameRequest[]>(mockUsernameRequests);
  const [filterStatus, setFilterStatus] = useState<string>("all");
  const [selectedRequest, setSelectedRequest] = useState<UsernameRequest | null>(null);
  const [adminComment, setAdminComment] = useState("");
  const [isProcessing, setIsProcessing] = useState(false);

  // Filter requests based on status
  const filteredRequests = requests.filter(request => {
    if (filterStatus === "all") return true;
    return request.status === filterStatus;
  });

  // Get counts for badges
  const pendingCount = requests.filter(r => r.status === 'pending').length;
  const approvedCount = requests.filter(r => r.status === 'approved').length;
  const rejectedCount = requests.filter(r => r.status === 'rejected').length;

  const getStatusColor = (status: string) => {
    switch (status) {
      case 'pending': return 'bg-yellow-100 text-yellow-800 dark:bg-yellow-900/20 dark:text-yellow-400';
      case 'approved': return 'bg-green-100 text-green-800 dark:bg-green-900/20 dark:text-green-400';
      case 'rejected': return 'bg-red-100 text-red-800 dark:bg-red-900/20 dark:text-red-400';
      default: return 'bg-gray-100 text-gray-800 dark:bg-gray-900/20 dark:text-gray-400';
    }
  };

  const getStatusIcon = (status: string) => {
    switch (status) {
      case 'pending': return <Clock className="w-4 h-4" />;
      case 'approved': return <CheckCircle className="w-4 h-4" />;
      case 'rejected': return <XCircle className="w-4 h-4" />;
      default: return null;
    }
  };

  const handleApprove = async (requestId: string) => {
    setIsProcessing(true);
    
    // Simulate API call
    await new Promise(resolve => setTimeout(resolve, 1000));
    
    setRequests(prev => prev.map(request => 
      request.id === requestId 
        ? {
            ...request,
            status: 'approved' as const,
            reviewDate: new Date().toISOString().split('T')[0],
            reviewedBy: "Admin User", // Would be current admin in real app
            adminComment: adminComment || "Approved without additional comment"
          }
        : request
    ));
    
    setSelectedRequest(null);
    setAdminComment("");
    setIsProcessing(false);
  };

  const handleReject = async (requestId: string) => {
    if (!adminComment.trim()) {
      alert("Please provide a reason for rejection");
      return;
    }
    
    setIsProcessing(true);
    
    // Simulate API call
    await new Promise(resolve => setTimeout(resolve, 1000));
    
    setRequests(prev => prev.map(request => 
      request.id === requestId 
        ? {
            ...request,
            status: 'rejected' as const,
            reviewDate: new Date().toISOString().split('T')[0],
            reviewedBy: "Admin User", // Would be current admin in real app
            adminComment: adminComment
          }
        : request
    ));
    
    setSelectedRequest(null);
    setAdminComment("");
    setIsProcessing(false);
  };

  return (
    <div className="space-y-6">
      {/* Header with Statistics */}
      <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
        <Card>
          <CardContent className="p-4">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm text-muted-foreground">Pending</p>
                <p className="text-2xl font-bold text-yellow-600">{pendingCount}</p>
              </div>
              <Clock className="w-8 h-8 text-yellow-600" />
            </div>
          </CardContent>
        </Card>
        
        <Card>
          <CardContent className="p-4">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm text-muted-foreground">Approved</p>
                <p className="text-2xl font-bold text-green-600">{approvedCount}</p>
              </div>
              <CheckCircle className="w-8 h-8 text-green-600" />
            </div>
          </CardContent>
        </Card>
        
        <Card>
          <CardContent className="p-4">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm text-muted-foreground">Rejected</p>
                <p className="text-2xl font-bold text-red-600">{rejectedCount}</p>
              </div>
              <XCircle className="w-8 h-8 text-red-600" />
            </div>
          </CardContent>
        </Card>
        
        <Card>
          <CardContent className="p-4">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm text-muted-foreground">Total</p>
                <p className="text-2xl font-bold">{requests.length}</p>
              </div>
              <User className="w-8 h-8 text-blue-600" />
            </div>
          </CardContent>
        </Card>
      </div>

      {/* Filters and Actions */}
      <Card>
        <CardContent className="p-4">
          <div className="flex items-center justify-between">
            <div className="flex items-center gap-4">
              <Filter className="w-5 h-5 text-muted-foreground" />
              <Select value={filterStatus} onValueChange={setFilterStatus}>
                <SelectTrigger className="w-40">
                  <SelectValue placeholder="Filter by status" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="all">All Requests</SelectItem>
                  <SelectItem value="pending">Pending</SelectItem>
                  <SelectItem value="approved">Approved</SelectItem>
                  <SelectItem value="rejected">Rejected</SelectItem>
                </SelectContent>
              </Select>
            </div>
            <Badge variant="outline">
              {filteredRequests.length} request{filteredRequests.length !== 1 ? 's' : ''}
            </Badge>
          </div>
        </CardContent>
      </Card>

      {/* Username Requests List */}
      <Card>
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <Edit3 className="w-5 h-5" />
            Username Change Requests
          </CardTitle>
          <CardDescription>
            Review and manage user requests to change their usernames
          </CardDescription>
        </CardHeader>
        <CardContent>
          {filteredRequests.length === 0 ? (
            <div className="text-center py-8">
              <User className="w-12 h-12 text-muted-foreground mx-auto mb-4" />
              <h3 className="text-lg font-medium mb-2">No requests found</h3>
              <p className="text-muted-foreground">
                {filterStatus === "all" 
                  ? "No username change requests have been submitted yet."
                  : `No ${filterStatus} username change requests.`
                }
              </p>
            </div>
          ) : (
            <div className="space-y-4">
              {filteredRequests.map((request) => (
                <div key={request.id} className="border rounded-lg p-4">
                  <div className="flex items-start justify-between">
                    <div className="flex-1">
                      <div className="flex items-center gap-3 mb-2">
                        <div className="flex items-center gap-2">
                          <User className="w-5 h-5 text-muted-foreground" />
                          <span className="font-medium">{request.userName}</span>
                          <span className="text-sm text-muted-foreground">({request.userEmail})</span>
                        </div>
                        <Badge className={getStatusColor(request.status)}>
                          {getStatusIcon(request.status)}
                          <span className="ml-1 capitalize">{request.status}</span>
                        </Badge>
                      </div>
                      
                      <div className="space-y-2 mb-3">
                        <div className="flex items-center gap-2">
                          <span className="text-sm font-medium">Current:</span>
                          <code className="bg-muted px-2 py-1 rounded text-sm">@{request.currentUsername}</code>
                          <span className="text-muted-foreground">→</span>
                          <span className="text-sm font-medium">Requested:</span>
                          <code className="bg-blue-50 dark:bg-blue-950/20 px-2 py-1 rounded text-sm text-blue-700 dark:text-blue-400">
                            @{request.requestedUsername}
                          </code>
                        </div>
                        
                        <div>
                          <span className="text-sm font-medium">Reason: </span>
                          <span className="text-sm text-muted-foreground">{request.reason}</span>
                        </div>
                        
                        <div className="flex items-center gap-4 text-xs text-muted-foreground">
                          <span className="flex items-center gap-1">
                            <Calendar className="w-3 h-3" />
                            Requested: {new Date(request.requestDate).toLocaleDateString()}
                          </span>
                          {request.reviewDate && (
                            <span className="flex items-center gap-1">
                              <Calendar className="w-3 h-3" />
                              Reviewed: {new Date(request.reviewDate).toLocaleDateString()}
                            </span>
                          )}
                        </div>
                        
                        {request.adminComment && (
                          <>
                            <Separator className="my-2" />
                            <div>
                              <span className="text-sm font-medium">Admin comment: </span>
                              <span className="text-sm text-muted-foreground">{request.adminComment}</span>
                              {request.reviewedBy && (
                                <span className="text-xs text-muted-foreground block">
                                  — {request.reviewedBy}
                                </span>
                              )}
                            </div>
                          </>
                        )}
                      </div>
                    </div>
                    
                    {request.status === 'pending' && (
                      <div className="flex gap-2 ml-4">
                        <Dialog>
                          <DialogTrigger asChild>
                            <Button 
                              size="sm" 
                              variant="outline"
                              onClick={() => {
                                setSelectedRequest(request);
                                setAdminComment("");
                              }}
                            >
                              <MessageSquare className="w-4 h-4 mr-2" />
                              Review
                            </Button>
                          </DialogTrigger>
                          <DialogContent className="max-w-2xl">
                            <DialogHeader>
                              <DialogTitle>Review Username Change Request</DialogTitle>
                              <DialogDescription>
                                Decide whether to approve or reject this username change request
                              </DialogDescription>
                            </DialogHeader>
                            
                            {selectedRequest && (
                              <div className="space-y-4">
                                <div className="bg-muted/50 rounded-lg p-4">
                                  <h4 className="font-medium mb-2">Request Details</h4>
                                  <div className="space-y-2 text-sm">
                                    <div><strong>User:</strong> {selectedRequest.userName} ({selectedRequest.userEmail})</div>
                                    <div><strong>Current Username:</strong> @{selectedRequest.currentUsername}</div>
                                    <div><strong>Requested Username:</strong> @{selectedRequest.requestedUsername}</div>
                                    <div><strong>Reason:</strong> {selectedRequest.reason}</div>
                                    <div><strong>Requested:</strong> {new Date(selectedRequest.requestDate).toLocaleDateString()}</div>
                                  </div>
                                </div>
                                
                                <div>
                                  <Label htmlFor="adminComment">Admin Comment (required for rejection)</Label>
                                  <Textarea
                                    id="adminComment"
                                    value={adminComment}
                                    onChange={(e) => setAdminComment(e.target.value)}
                                    placeholder="Enter your decision reasoning..."
                                    className="mt-1"
                                    rows={3}
                                  />
                                </div>
                                
                                <Alert>
                                  <AlertTriangle className="w-4 h-4" />
                                  <AlertDescription>
                                    Approved username changes will be immediately applied to the user's account.
                                    This action cannot be undone.
                                  </AlertDescription>
                                </Alert>
                                
                                <div className="flex gap-2 pt-4">
                                  <Button 
                                    onClick={() => handleApprove(selectedRequest.id)}
                                    disabled={isProcessing}
                                    className="bg-green-600 hover:bg-green-700"
                                  >
                                    <CheckCircle className="w-4 h-4 mr-2" />
                                    {isProcessing ? "Processing..." : "Approve"}
                                  </Button>
                                  <Button 
                                    onClick={() => handleReject(selectedRequest.id)}
                                    disabled={isProcessing || !adminComment.trim()}
                                    variant="destructive"
                                  >
                                    <XCircle className="w-4 h-4 mr-2" />
                                    {isProcessing ? "Processing..." : "Reject"}
                                  </Button>
                                  <Button variant="outline" onClick={() => setSelectedRequest(null)}>
                                    Cancel
                                  </Button>
                                </div>
                              </div>
                            )}
                          </DialogContent>
                        </Dialog>
                      </div>
                    )}
                  </div>
                </div>
              ))}
            </div>
          )}
        </CardContent>
      </Card>
    </div>
  );
}