import { useState } from "react";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "./ui/card";
import { Button } from "./ui/button";
import { Badge } from "./ui/badge";
import { Input } from "./ui/input";
import { Label } from "./ui/label";
import { Textarea } from "./ui/textarea";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "./ui/select";
import { Switch } from "./ui/switch";
import { Dialog, DialogContent, DialogDescription, DialogHeader, DialogTitle, DialogTrigger } from "./ui/dialog";
import { Alert, AlertDescription } from "./ui/alert";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "./ui/tabs";
import { 
  Users, 
  Plus, 
  Edit, 
  Trash2, 
  Search, 
  MapPin, 
  Calendar, 
  Settings,
  Eye,
  Ban,
  UserMinus,
  Crown,
  AlertTriangle,
  CheckCircle,
  Filter,
  MoreVertical
} from "lucide-react";
import { DropdownMenu, DropdownMenuContent, DropdownMenuItem, DropdownMenuTrigger } from "./ui/dropdown-menu";
import { ImageWithFallback } from "./figma/ImageWithFallback";

interface Group {
  id: number;
  name: string;
  description: string;
  image: string;
  category: "interests" | "location" | "events" | "activities";
  memberCount: number;
  isPrivate: boolean;
  location?: string;
  maxMembers?: number;
  createdBy: string;
  createdAt: string;
  status: "active" | "suspended" | "archived";
  tags: string[];
  requiresApproval: boolean;
}

interface GroupMember {
  id: number;
  groupId: number;
  userId: number;
  username: string;
  avatar: string;
  membershipTier: "free" | "premium" | "platinum";
  joinedAt: string;
  status: "active" | "suspended" | "banned";
  role: "member" | "moderator";
}

export function AdminGroupsManagement() {
  const [activeTab, setActiveTab] = useState("groups");
  const [searchQuery, setSearchQuery] = useState("");
  const [selectedCategory, setSelectedCategory] = useState<string>("all");
  const [isCreateDialogOpen, setIsCreateDialogOpen] = useState(false);
  const [isEditDialogOpen, setIsEditDialogOpen] = useState(false);
  const [selectedGroup, setSelectedGroup] = useState<Group | null>(null);

  // Mock data for groups
  const [groups, setGroups] = useState<Group[]>([
    {
      id: 1,
      name: "Coffee Lovers NYC",
      description: "Weekly coffee meetups around Manhattan. Share your favorite spots and meet fellow coffee enthusiasts!",
      image: "https://images.unsplash.com/photo-1511920170033-f8396924c348?w=400&h=300&fit=crop",
      category: "interests",
      memberCount: 127,
      isPrivate: false,
      location: "New York, NY",
      maxMembers: 200,
      createdBy: "admin@singleszonne.com",
      createdAt: "2024-01-15",
      status: "active",
      tags: ["Coffee", "Social", "Weekly"],
      requiresApproval: false
    },
    {
      id: 2,
      name: "Photography Walks",
      description: "Explore the city through your lens. Perfect for beginners and pros alike!",
      image: "https://images.unsplash.com/photo-1606983340126-99ab4feaa64a?w=400&h=300&fit=crop",
      category: "activities",
      memberCount: 89,
      isPrivate: false,
      location: "Los Angeles, CA",
      maxMembers: 150,
      createdBy: "admin@singleszonne.com",
      createdAt: "2024-01-10",
      status: "active",
      tags: ["Photography", "Art", "Outdoor"],
      requiresApproval: true
    }
  ]);

  // Mock data for group members
  const [groupMembers, setGroupMembers] = useState<GroupMember[]>([
    {
      id: 1,
      groupId: 1,
      userId: 101,
      username: "sarah_m",
      avatar: "https://images.unsplash.com/photo-1494790108755-2616b612b593?w=100&h=100&fit=crop",
      membershipTier: "premium",
      joinedAt: "2024-01-16",
      status: "active",
      role: "member"
    },
    {
      id: 2,
      groupId: 1,
      userId: 102,
      username: "mike_coffee",
      avatar: "https://images.unsplash.com/photo-1472099645785-5658abf4ff4e?w=100&h=100&fit=crop",
      membershipTier: "platinum",
      joinedAt: "2024-01-18",
      status: "active",
      role: "moderator"
    }
  ]);

  const [newGroup, setNewGroup] = useState({
    name: "",
    description: "",
    category: "interests" as const,
    location: "",
    maxMembers: 100,
    isPrivate: false,
    requiresApproval: false,
    tags: "",
    image: ""
  });

  const categories = [
    { value: "all", label: "All Categories" },
    { value: "interests", label: "Interests" },
    { value: "location", label: "Location" },
    { value: "events", label: "Events" },
    { value: "activities", label: "Activities" }
  ];

  const handleCreateGroup = () => {
    const group: Group = {
      id: Date.now(),
      name: newGroup.name,
      description: newGroup.description,
      image: newGroup.image || "https://images.unsplash.com/photo-1522202176988-66273c2fd55f?w=400&h=300&fit=crop",
      category: newGroup.category,
      memberCount: 0,
      isPrivate: newGroup.isPrivate,
      location: newGroup.location,
      maxMembers: newGroup.maxMembers,
      createdBy: "admin@singleszonne.com",
      createdAt: new Date().toISOString().split('T')[0],
      status: "active",
      tags: newGroup.tags.split(",").map(tag => tag.trim()).filter(Boolean),
      requiresApproval: newGroup.requiresApproval
    };

    setGroups([...groups, group]);
    setNewGroup({
      name: "",
      description: "",
      category: "interests",
      location: "",
      maxMembers: 100,
      isPrivate: false,
      requiresApproval: false,
      tags: "",
      image: ""
    });
    setIsCreateDialogOpen(false);
  };

  const handleDeleteGroup = (groupId: number) => {
    setGroups(groups.filter(group => group.id !== groupId));
    setGroupMembers(groupMembers.filter(member => member.groupId !== groupId));
  };

  const handleSuspendGroup = (groupId: number) => {
    setGroups(groups.map(group => 
      group.id === groupId 
        ? { ...group, status: group.status === "suspended" ? "active" : "suspended" }
        : group
    ));
  };

  const handleRemoveMember = (memberId: number) => {
    setGroupMembers(groupMembers.filter(member => member.id !== memberId));
    // Update group member count
    const member = groupMembers.find(m => m.id === memberId);
    if (member) {
      setGroups(groups.map(group => 
        group.id === member.groupId 
          ? { ...group, memberCount: Math.max(0, group.memberCount - 1) }
          : group
      ));
    }
  };

  const handleSuspendMember = (memberId: number) => {
    setGroupMembers(groupMembers.map(member => 
      member.id === memberId 
        ? { ...member, status: member.status === "suspended" ? "active" : "suspended" }
        : member
    ));
  };

  const filteredGroups = groups.filter(group => {
    const matchesSearch = group.name.toLowerCase().includes(searchQuery.toLowerCase()) ||
                         group.description.toLowerCase().includes(searchQuery.toLowerCase());
    const matchesCategory = selectedCategory === "all" || group.category === selectedCategory;
    return matchesSearch && matchesCategory;
  });

  const getStatusBadge = (status: string) => {
    switch (status) {
      case "active":
        return <Badge className="bg-green-100 text-green-800">Active</Badge>;
      case "suspended":
        return <Badge className="bg-red-100 text-red-800">Suspended</Badge>;
      case "archived":
        return <Badge className="bg-gray-100 text-gray-800">Archived</Badge>;
      default:
        return <Badge>{status}</Badge>;
    }
  };

  const getMembershipBadge = (tier: string) => {
    switch (tier) {
      case "free":
        return <Badge variant="secondary">Free</Badge>;
      case "premium":
        return <Badge className="bg-purple-100 text-purple-800">Premium</Badge>;
      case "platinum":
        return <Badge className="bg-yellow-100 text-yellow-800">Platinum</Badge>;
      default:
        return <Badge>{tier}</Badge>;
    }
  };

  return (
    <div className="space-y-6">
      <div className="flex items-center justify-between">
        <div>
          <h2 className="text-3xl font-bold">Groups & Events Management</h2>
          <p className="text-muted-foreground">Create and manage groups, communities & dating events</p>
        </div>
        <Dialog open={isCreateDialogOpen} onOpenChange={setIsCreateDialogOpen}>
          <DialogTrigger asChild>
            <Button className="bg-green-600 hover:bg-green-700">
              <Plus className="w-4 h-4 mr-2" />
              Create Group
            </Button>
          </DialogTrigger>
          <DialogContent className="max-w-2xl">
            <DialogHeader>
              <DialogTitle>Create New Group</DialogTitle>
              <DialogDescription>
                Create a new group or community for members to join
              </DialogDescription>
            </DialogHeader>
            <div className="grid gap-4 py-4">
              <div className="grid grid-cols-2 gap-4">
                <div>
                  <Label htmlFor="name">Group Name</Label>
                  <Input
                    id="name"
                    value={newGroup.name}
                    onChange={(e) => setNewGroup({...newGroup, name: e.target.value})}
                    placeholder="Enter group name"
                  />
                </div>
                <div>
                  <Label htmlFor="category">Category</Label>
                  <Select value={newGroup.category} onValueChange={(value: any) => setNewGroup({...newGroup, category: value})}>
                    <SelectTrigger>
                      <SelectValue />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="interests">Interests</SelectItem>
                      <SelectItem value="location">Location</SelectItem>
                      <SelectItem value="events">Events</SelectItem>
                      <SelectItem value="activities">Activities</SelectItem>
                    </SelectContent>
                  </Select>
                </div>
              </div>
              
              <div>
                <Label htmlFor="description">Description</Label>
                <Textarea
                  id="description"
                  value={newGroup.description}
                  onChange={(e) => setNewGroup({...newGroup, description: e.target.value})}
                  placeholder="Describe what this group is about"
                  rows={3}
                />
              </div>

              <div className="grid grid-cols-2 gap-4">
                <div>
                  <Label htmlFor="location">Location</Label>
                  <Input
                    id="location"
                    value={newGroup.location}
                    onChange={(e) => setNewGroup({...newGroup, location: e.target.value})}
                    placeholder="City, State/Country"
                  />
                </div>
                <div>
                  <Label htmlFor="maxMembers">Max Members</Label>
                  <Input
                    id="maxMembers"
                    type="number"
                    value={newGroup.maxMembers}
                    onChange={(e) => setNewGroup({...newGroup, maxMembers: parseInt(e.target.value)})}
                    min={1}
                    max={1000}
                  />
                </div>
              </div>

              <div>
                <Label htmlFor="tags">Tags (comma-separated)</Label>
                <Input
                  id="tags"
                  value={newGroup.tags}
                  onChange={(e) => setNewGroup({...newGroup, tags: e.target.value})}
                  placeholder="Coffee, Social, Weekly"
                />
              </div>

              <div>
                <Label htmlFor="image">Group Image URL</Label>
                <Input
                  id="image"
                  value={newGroup.image}
                  onChange={(e) => setNewGroup({...newGroup, image: e.target.value})}
                  placeholder="https://example.com/image.jpg"
                />
              </div>

              <div className="flex items-center space-x-4">
                <div className="flex items-center space-x-2">
                  <Switch
                    id="private"
                    checked={newGroup.isPrivate}
                    onCheckedChange={(checked) => setNewGroup({...newGroup, isPrivate: checked})}
                  />
                  <Label htmlFor="private">Private Group</Label>
                </div>
                <div className="flex items-center space-x-2">
                  <Switch
                    id="approval"
                    checked={newGroup.requiresApproval}
                    onCheckedChange={(checked) => setNewGroup({...newGroup, requiresApproval: checked})}
                  />
                  <Label htmlFor="approval">Requires Approval</Label>
                </div>
              </div>
            </div>
            <div className="flex justify-end space-x-2">
              <Button variant="outline" onClick={() => setIsCreateDialogOpen(false)}>
                Cancel
              </Button>
              <Button onClick={handleCreateGroup} className="bg-green-600 hover:bg-green-700">
                Create Group
              </Button>
            </div>
          </DialogContent>
        </Dialog>
      </div>

      <Alert>
        <AlertTriangle className="w-4 h-4" />
        <AlertDescription>
          Only Premium and Platinum members can join groups. Free members cannot access groups & communities.
        </AlertDescription>
      </Alert>

      <Tabs value={activeTab} onValueChange={setActiveTab} className="space-y-6">
        <TabsList className="grid w-full grid-cols-2">
          <TabsTrigger value="groups">Groups ({groups.length})</TabsTrigger>
          <TabsTrigger value="members">Members ({groupMembers.length})</TabsTrigger>
        </TabsList>

        <TabsContent value="groups" className="space-y-6">
          {/* Search and Filters */}
          <div className="flex gap-4">
            <div className="relative flex-1">
              <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-muted-foreground w-4 h-4" />
              <Input
                placeholder="Search groups..."
                value={searchQuery}
                onChange={(e) => setSearchQuery(e.target.value)}
                className="pl-10"
              />
            </div>
            <Select value={selectedCategory} onValueChange={setSelectedCategory}>
              <SelectTrigger className="w-48">
                <SelectValue />
              </SelectTrigger>
              <SelectContent>
                {categories.map((category) => (
                  <SelectItem key={category.value} value={category.value}>
                    {category.label}
                  </SelectItem>
                ))}
              </SelectContent>
            </Select>
          </div>

          {/* Groups List */}
          <div className="grid gap-4">
            {filteredGroups.map((group) => (
              <Card key={group.id}>
                <CardContent className="p-6">
                  <div className="flex items-start gap-4">
                    <ImageWithFallback
                      src={group.image}
                      alt={group.name}
                      className="w-20 h-20 object-cover rounded-lg"
                    />
                    <div className="flex-1">
                      <div className="flex items-start justify-between">
                        <div>
                          <div className="flex items-center gap-2 mb-1">
                            <h3 className="font-semibold text-lg">{group.name}</h3>
                            {getStatusBadge(group.status)}
                            {group.isPrivate && (
                              <Badge variant="outline">
                                <Crown className="w-3 h-3 mr-1" />
                                Private
                              </Badge>
                            )}
                          </div>
                          <p className="text-muted-foreground mb-2">{group.description}</p>
                          <div className="flex items-center gap-4 text-sm text-muted-foreground">
                            <span className="flex items-center gap-1">
                              <Users className="w-4 h-4" />
                              {group.memberCount}/{group.maxMembers} members
                            </span>
                            {group.location && (
                              <span className="flex items-center gap-1">
                                <MapPin className="w-4 h-4" />
                                {group.location}
                              </span>
                            )}
                            <span className="flex items-center gap-1">
                              <Calendar className="w-4 h-4" />
                              Created {group.createdAt}
                            </span>
                          </div>
                          <div className="flex gap-1 mt-2">
                            {group.tags.map((tag) => (
                              <Badge key={tag} variant="secondary" className="text-xs">
                                {tag}
                              </Badge>
                            ))}
                          </div>
                        </div>
                        <DropdownMenu>
                          <DropdownMenuTrigger asChild>
                            <Button variant="ghost" size="sm">
                              <MoreVertical className="w-4 h-4" />
                            </Button>
                          </DropdownMenuTrigger>
                          <DropdownMenuContent align="end">
                            <DropdownMenuItem onClick={() => {
                              setSelectedGroup(group);
                              setIsEditDialogOpen(true);
                            }}>
                              <Edit className="w-4 h-4 mr-2" />
                              Edit Group
                            </DropdownMenuItem>
                            <DropdownMenuItem>
                              <Eye className="w-4 h-4 mr-2" />
                              View Members
                            </DropdownMenuItem>
                            <DropdownMenuItem onClick={() => handleSuspendGroup(group.id)}>
                              <Ban className="w-4 h-4 mr-2" />
                              {group.status === "suspended" ? "Unsuspend" : "Suspend"}
                            </DropdownMenuItem>
                            <DropdownMenuItem 
                              onClick={() => handleDeleteGroup(group.id)}
                              className="text-red-600"
                            >
                              <Trash2 className="w-4 h-4 mr-2" />
                              Delete Group
                            </DropdownMenuItem>
                          </DropdownMenuContent>
                        </DropdownMenu>
                      </div>
                    </div>
                  </div>
                </CardContent>
              </Card>
            ))}
          </div>
        </TabsContent>

        <TabsContent value="members" className="space-y-6">
          <div className="grid gap-4">
            {groupMembers.map((member) => {
              const group = groups.find(g => g.id === member.groupId);
              return (
                <Card key={member.id}>
                  <CardContent className="p-4">
                    <div className="flex items-center justify-between">
                      <div className="flex items-center gap-4">
                        <ImageWithFallback
                          src={member.avatar}
                          alt={member.username}
                          className="w-12 h-12 object-cover rounded-full"
                        />
                        <div>
                          <div className="flex items-center gap-2 mb-1">
                            <span className="font-medium">{member.username}</span>
                            {getMembershipBadge(member.membershipTier)}
                            {member.role === "moderator" && (
                              <Badge className="bg-blue-100 text-blue-800">Moderator</Badge>
                            )}
                          </div>
                          <p className="text-sm text-muted-foreground">
                            Member of {group?.name} • Joined {member.joinedAt}
                          </p>
                        </div>
                      </div>
                      <div className="flex items-center gap-2">
                        {getStatusBadge(member.status)}
                        <DropdownMenu>
                          <DropdownMenuTrigger asChild>
                            <Button variant="ghost" size="sm">
                              <MoreVertical className="w-4 h-4" />
                            </Button>
                          </DropdownMenuTrigger>
                          <DropdownMenuContent align="end">
                            <DropdownMenuItem onClick={() => handleSuspendMember(member.id)}>
                              <Ban className="w-4 h-4 mr-2" />
                              {member.status === "suspended" ? "Unsuspend" : "Suspend"}
                            </DropdownMenuItem>
                            <DropdownMenuItem 
                              onClick={() => handleRemoveMember(member.id)}
                              className="text-red-600"
                            >
                              <UserMinus className="w-4 h-4 mr-2" />
                              Remove from Group
                            </DropdownMenuItem>
                          </DropdownMenuContent>
                        </DropdownMenu>
                      </div>
                    </div>
                  </CardContent>
                </Card>
              );
            })}
          </div>
        </TabsContent>
      </Tabs>
    </div>
  );
}